// vim: foldmethod=marker commentstring=//%s
package mn.jp.kekkouyakan.collision;
import mn.jp.kekkouyakan.geom.KyVector2f;
import java.util.ArrayList;
import mn.jp.kekkouyakan.functional.IKyAction1;
import mn.jp.kekkouyakan.util.KyFloatComparator;

public class ColResult
{//{{{
	final static KyFloatComparator defaultComparator = new KyFloatComparator();

	ColShape ballShape1;
	ColShape ballShape2;
	ColShape wallShape;

	int ballIndex1 = -1;
	int ballIndex2 = -1;
	int wallIndex1 = -1;
	int wallIndex2 = -1;
	float progress = -1f;

	public ColResult( ColShape ballShape1_, ColShape ballShape2_, ColShape wallShape_ )
	{//{{{
		ballShape1 = ballShape1_;
		ballShape2 = ballShape2_;
		wallShape = wallShape_;
	}//}}}
	public ColResult( ColShape ballShape1_, ColShape ballShape2_, ColShape wallShape_, int ballIndex1_, int ballIndex2_, int wallIndex1_, int wallIndex2_ )
	{//{{{
		this( ballShape1_, ballShape2_, wallShape_ );
		ballIndex1 = ballIndex1_;
		ballIndex2 = ballIndex2_;
		wallIndex1 = wallIndex1_;
		wallIndex2 = wallIndex2_;
	}//}}}
	public ColResult getClone()
	{//{{{
		ColResult res_ = new ColResult( ballShape1, ballShape2, wallShape, ballIndex1, ballIndex2, wallIndex1, wallIndex2 );
		res_.progress = progress;
		return res_;
	}//}}}
	@Override
	public boolean equals( Object rhs_ )
	{//{{{
		if( rhs_ == this ){
			return true;
		}
		else if( rhs_ == null ){
			return false;
		}
		else if( rhs_ instanceof ColResult ){
			ColResult res_ = (ColResult)rhs_;
			if( ballShape1.body != res_.ballShape1.body ){
				return false;
			}
			if( ballShape2.body != res_.ballShape2.body ){
				return false;
			}
			if( wallShape.body != res_.wallShape.body ){
				return false;
			}
			if( ballIndex1 == res_.ballIndex1 ){
				if( ballIndex2 != res_.ballIndex2 ){
					return false;
				}
			}
			else if( ballIndex1 == res_.ballIndex2 ){
				if( ballIndex2 != res_.ballIndex1 ){
					return false;
				}
			}
			else{
				return false;
			}
			if( wallIndex1 == res_.wallIndex1 ){
				if( wallIndex2 != res_.wallIndex2 ){
					return false;
				}
			}
			else if( wallIndex1 == res_.wallIndex2 ){
				if( wallIndex2 != res_.wallIndex1 ){
					return false;
				}
			}
			else{
				return false;
			}
			return true;
		}
		else{
			return false;
		}
	}//}}}
	public boolean canBound()
	{//{{{
		assert( ballIndex1 >= 0 );
		assert( wallIndex1 >= 0 );
		if( ballIndex2 < 0 ){
			return wallIndex2 >= 0;
		}
		else{
			return wallIndex2 < 0;
		}
	}//}}}
	public KyVector2f getSlipVector( KyFloatComparator cmp_ )
	{//{{{
		return getSlipVector( cmp_, null );
	}//}}}
	public KyVector2f getSlipVector( KyFloatComparator cmp_, KyVector2f ballVec_ )
	{//{{{
		if( !canBound() ){
			return null;
		}
		ColShape ballShape_ = ballShape1.body.shape;
		if( ballVec_ == null ){
			ballVec_ = ballShape_.delta.position;
		}
		assert( ballVec_ != null );
		if( ballVec_.equals( cmp_, 0f, 0f ) ){
			return null;
		}
		assert( ballIndex1 >= 0 );
		assert( wallIndex1 >= 0 );
		if( ballIndex2 < 0 ){
			assert( wallIndex2 >= 0 );
			KyVector2f wallPos1_ = wallShape.getVector( wallIndex1 );
			KyVector2f wallPos2_ = wallShape.getVector( wallIndex2 );
			return getSlipVector( cmp_, ballVec_, wallPos1_, wallPos2_ );
		}
		else{
			KyVector2f wallVec_ = ballVec_.mul( -1f );
			KyVector2f ballPos1_ = ballShape_.getVector( ballIndex1 );
			KyVector2f ballPos2_ = ballShape_.getVector( ballIndex2 );
			KyVector2f vec_ = getSlipVector( cmp_, wallVec_, ballPos1_, ballPos2_ );
			if( vec_ == null ){
				return null;
			}
			return vec_.mulLocal( -1f );
		}
	}//}}}
	public KyVector2f getBoundVector( KyFloatComparator cmp_, KyVector2f ballVec_, float elast_ )
	{//{{{
		if( !canBound() ){
			return null;
		}
		ColShape ballShape_ = ballShape1.body.shape;
		if( ballVec_ == null ){
			ballVec_ = ballShape_.delta.position;
		}
		assert( ballVec_ != null );
		if( ballVec_.equals( cmp_, 0f, 0f ) ){
			return null;
		}
		assert( ballIndex1 >= 0 );
		assert( wallIndex1 >= 0 );
		if( ballIndex2 < 0 ){
			assert( wallIndex2 >= 0 );
			KyVector2f wallPos1_ = wallShape.getVector( wallIndex1 );
			KyVector2f wallPos2_ = wallShape.getVector( wallIndex2 );
			return getBoundVector( cmp_, ballVec_, wallPos1_, wallPos2_, elast_ );
		}
		else{
			KyVector2f wallVec_ = ballVec_.mul( -1f );
			KyVector2f ballPos1_ = ballShape_.getVector( ballIndex1 );
			KyVector2f ballPos2_ = ballShape_.getVector( ballIndex2 );
			return getBoundVector( cmp_, wallVec_, ballPos1_, ballPos2_, elast_ ).mulLocal( -1f );
		}
	}//}}}
	public boolean isUnspecifiedProgress( KyFloatComparator cmp_ )
	{//{{{
		return cmp_.less( progress, 0f );
	}//}}}

	public static boolean isClosedPoints( KyVector2f[] ptLs_ )
	{//{{{
		int sz_ = ptLs_.length;
		if( sz_ <= 3 ){
			return false;
		}
		return ptLs_[0] == ptLs_[sz_ - 1];
	}//}}}
	public static KyVector2f[] closePoints( KyVector2f[] ptLs_ )
	{//{{{
		if( ptLs_[0].equals( ptLs_[ ptLs_.length - 1] ) ){
			return ptLs_;
		}
		KyVector2f[] ptLs2_ = new KyVector2f[ ptLs_.length + 1 ];
		for( int i_ = 0; i_ < ptLs_.length; ++i_ ){
			ptLs2_[ i_ ] = ptLs_[ i_ ];
		}
		ptLs2_[ ptLs_.length ] = ptLs_[0];
		return ptLs2_;
	}//}}}
	public static boolean isPointInTriangle( KyFloatComparator cmp_, KyVector2f tgt_, KyVector2f[] ptLs_ )
	{//{{{
		if( ptLs_.length < 3 ){
			throw new AssertionError();
		}
		return isPointInTriangle( cmp_, tgt_, ptLs_[0], ptLs_[1], ptLs_[2] );
	}//}}}
	public static boolean isPointInTriangle( KyVector2f tgt_, KyVector2f[] ptLs_ )
	{//{{{
		return isPointInTriangle( defaultComparator, tgt_, ptLs_ );
	}//}}}
	public static boolean isPointInTriangle( KyFloatComparator cmp_, KyVector2f tgt_, KyVector2f pt1_, KyVector2f pt2_, KyVector2f pt3_ )
	{//{{{
		int c1_ = cmp_.compare( pt1_.sub( pt2_ ).cross( tgt_.sub( pt2_ ) ), 0f );
		int c2_ = cmp_.compare( pt2_.sub( pt3_ ).cross( tgt_.sub( pt3_ ) ), 0f );
		int c3_ = cmp_.compare( pt3_.sub( pt1_ ).cross( tgt_.sub( pt1_ ) ), 0f );
		return (c1_ == c2_) && (c1_ == c3_);
	}//}}}
	public static boolean isPointInTriangle( KyVector2f tgt_, KyVector2f pt1_, KyVector2f pt2_, KyVector2f pt3_ )
	{//{{{
		return isPointInTriangle( defaultComparator, tgt_, pt1_, pt2_, pt3_ );
	}//}}}
	public static boolean isPointInQuadrangle( KyFloatComparator cmp_, KyVector2f tgt_, KyVector2f[] ptLs_ )
	{//{{{
		if( ptLs_.length < 4 ){
			throw new AssertionError();
		}
		return isPointInQuadrangle( cmp_, tgt_, ptLs_[0], ptLs_[1], ptLs_[2], ptLs_[3] );
	}//}}}
	public static boolean isPointInQuadrangle( KyVector2f tgt_, KyVector2f[] ptLs_ )
	{//{{{
		return isPointInQuadrangle( defaultComparator, tgt_, ptLs_ );
	}//}}}
	public static boolean isPointInQuadrangle( KyFloatComparator cmp_, KyVector2f tgt_, KyVector2f pt1_, KyVector2f pt2_, KyVector2f pt3_, KyVector2f pt4_ )
	{//{{{
		KyVector2f ln_ = pt1_.sub( pt3_ );
		int c2_ = cmp_.compare( ln_.cross( pt2_.sub( pt3_ ) ), 0f );
		int c4_ = cmp_.compare( ln_.cross( pt4_.sub( pt3_ ) ), 0f );
		if( c2_ * c4_ < 0 ){
			return isPointOnLine( cmp_, tgt_, pt1_, pt3_ )
			|| isPointInTriangle( cmp_, tgt_, pt1_, pt2_, pt3_ )
			|| isPointInTriangle( cmp_, tgt_, pt1_, pt3_, pt4_ );
		}
		else{
			return isPointOnLine( cmp_, tgt_, pt2_, pt4_ )
			|| isPointInTriangle( cmp_, tgt_, pt1_, pt2_, pt4_ )
			|| isPointInTriangle( cmp_, tgt_, pt2_, pt3_, pt4_ );
		}
	}//}}}
	public static boolean isPointInQuadrangle( KyVector2f tgt_, KyVector2f pt1_, KyVector2f pt2_, KyVector2f pt3_, KyVector2f pt4_ )
	{//{{{
		return isPointInQuadrangle( defaultComparator, tgt_, pt1_, pt2_, pt3_, pt4_ );
	}//}}}
	public static boolean isPointInPoints( KyFloatComparator cmp_, KyVector2f tgt_, KyVector2f... ptLs_ )
	{//{{{
		//http://www.hiramine.com/programming/graphics/2d_ispointinpolygon.html
		int sz_ = ptLs_.length;
		if( ptLs_[0].equals( ptLs_[ sz_ - 1 ] ) ){
			--sz_;
		}
		else{
			return false;
		}
		if( sz_ == 3 ){
			return isPointInTriangle( cmp_, tgt_, ptLs_[0], ptLs_[1], ptLs_[2] );
		}
		else if( sz_ == 4 ){
			return isPointInQuadrangle( cmp_, tgt_, ptLs_[0], ptLs_[1], ptLs_[2], ptLs_[3] );
		}
		int crossingCnt_ = 0;
		KyVector2f pt0_ = ptLs_[0];
		boolean flagx0_ = cmp_.compare(tgt_.x, pt0_.x ) <= 0;
		boolean flagy0_ = cmp_.compare(tgt_.y, pt0_.y ) <= 0;

		for( int i_ = 0; i_ < sz_;  ++i_ ){
			KyVector2f pt1_ = ptLs_[ (i_+1) % sz_ ];
			boolean flagx1_ = cmp_.compare( tgt_.x, pt1_.x) <= 0;
			boolean flagy1_ = cmp_.compare( tgt_.y, pt1_.y) <= 0;
			if( flagy0_ != flagy1_ ){
				if( flagx0_ == flagx1_ ){
					if( flagx0_ ){
						crossingCnt_ += flagy0_? -1: 1;
					}
				}
				else{
					float dx_ = pt1_.x - pt0_.x;
					float dy_ = pt1_.y - pt0_.y;
					float ty_ = tgt_.y - pt0_.y;
					float tx_ = pt0_.x + ty_ * dx_ / dy_;
					if( cmp_.compare( tgt_.x, tx_) <= 0 ){
						crossingCnt_ += flagy0_? -1: 1;
					}
				}
			}
			pt0_ = pt1_;
			flagx0_ = flagx1_;
			flagy0_ = flagy1_;
		}

		return crossingCnt_ != 0;
	}//}}}
	public static boolean isPointInPoints( KyVector2f tgt_, KyVector2f... ptLs_ )
	{//{{{
		return isPointInPoints( defaultComparator, tgt_, ptLs_ );
	}//}}}
	public static boolean isPointOnLine( KyFloatComparator cmp_, KyVector2f tgt_, KyVector2f pt1_, KyVector2f pt2_ )
	{//{{{
		if( pt1_.equals( pt2_ ) ){
			return pt1_.equals( tgt_ );
		}
		KyVector2f l1_ = pt2_.sub( pt1_ );
		KyVector2f t1_ = tgt_.sub( pt1_ );
		if( cmp_.equals( t1_.cross( l1_ ), 0f ) ){
			KyVector2f t2_ = tgt_.sub( pt2_ );
			KyVector2f l2_ = l1_.negative();
			return
				cmp_.moreEquals( t1_.dot(l1_), 0f )
				&& cmp_.moreEquals( t2_.dot(l2_), 0f );
		}
		return false;
	}//}}}
	public static boolean isPointOnLine( KyVector2f tgt_, KyVector2f pt1_, KyVector2f pt2_ )
	{//{{{
		return isPointOnLine( defaultComparator, tgt_, pt1_, pt2_ );
	}//}}}
	public static boolean isPointOnLines( KyFloatComparator cmp_, KyVector2f tgt_, KyVector2f... ptLs_ )
	{//{{{
		for( int i_ = 1; i_ < ptLs_.length; ++i_ ){
			KyVector2f pt1_ = ptLs_[ i_ - 1 ];
			KyVector2f pt2_ = ptLs_[ i_ ];
			if( isPointOnLine( cmp_, tgt_, pt1_, pt2_ ) ){
				return true;
			}
		}
		return false;
	}//}}}
	public static boolean isPointOnLines( KyVector2f tgt_, KyVector2f... ptLs_ )
	{//{{{
		return isPointOnLines( defaultComparator, tgt_, ptLs_ );
	}//}}}

	static float[] calcIntersection( KyFloatComparator cmp_, KyVector2f a_, KyVector2f b_, KyVector2f w_, float r_ )
	{//{{{
		double bax_ = b_.x-a_.x;
		double bay_ = b_.y-a_.y;
		double wax_ = w_.x-a_.x;
		double way_ = w_.y-a_.y;
		double aa_ = bax_*bax_+bay_*bay_;
		double bb_ = bax_*wax_+bay_*way_;
		double cc_ = wax_*wax_+way_*way_-r_*r_;

		if( cmp_.equals( aa_, 0f ) ){
			if( cmp_.equals( bb_, 0f ) ){
				return null;
			}
			else{
				return new float[]{(float)(cc_/(2.0*bb_))};
			}
		}
		else{
			double rt_ = Math.sqrt(bb_*bb_ -aa_*cc_);
			float k1_ = (float)((bb_ + rt_) / aa_);
			float k2_ = (float)((bb_ - rt_) / aa_);
			if( cmp_.less( k1_, k2_ ) ){
				return new float[]{ k1_, k2_ };
			}
			else if( cmp_.less( k2_, k1_ ) ){
				return new float[]{ k2_, k1_ };
			}
			else{
				return new float[]{ k1_ };
			}
		}
	}//}}}
	static boolean hasIntersection( KyFloatComparator cmp_, KyVector2f a_, KyVector2f b_, KyVector2f w_, float r_ )
	{//{{{
		float[] kLs_ = calcIntersection( cmp_, a_, b_, w_, r_ );
		if( kLs_ == null ){
			return false;
		}
		KyVector2f aw_ = w_.sub( a_ );
		KyVector2f ab_ = b_.sub( a_ );
		for( float k_ : kLs_ ){
			if( cmp_.less( k_, 0f ) ){
				continue;
			}
			if( cmp_.more( k_, 1f ) ){
				continue;
			}
			KyVector2f cw_ = aw_.sub( ab_.mul( k_ ) );
			if( cmp_.lessEquals( cw_.magnitudeSquared(), r_* r_ )
				&& !cmp_.equals( ab_.dot( cw_ ), 0f )
			){
				return true;
			}
		}
		return false;
	}//}}}
	static KyVector2f getSlipVector( KyFloatComparator cmp_, KyVector2f ballVec_, KyVector2f wallPt1_, KyVector2f wallPt2_ )
	{//{{{
		KyVector2f wallVec_ = wallPt2_.sub( wallPt1_ );
		if( cmp_.equals( ballVec_.dot( wallVec_ ), 0f ) ){
			return null;
		}
		return ballVec_.orth( wallVec_ );
	}//}}}
	static KyVector2f getBoundVector( KyFloatComparator cmp_, KyVector2f ballVec_, KyVector2f wallPt1_, KyVector2f wallPt2_, float elast_ )
	{//{{{
		KyVector2f wallVec_ = wallPt2_.sub( wallPt1_ );
		if( cmp_.equals( ballVec_.dot( wallVec_ ), 0f ) ){
			return ballVec_.mul( -elast_ );
		}
		float sign_ = 1f;
		KyVector2f owallVec_ = new KyVector2f( -wallVec_.y, wallVec_.x );
		if( cmp_.less( ballVec_.cross( wallVec_ ), 0f ) ){
			if( cmp_.less( owallVec_.cross( wallVec_ ), 0f ) ){
				sign_ = -1f;
			}
		}
		else{
			if( !cmp_.less( owallVec_.cross( wallVec_ ), 0f ) ){
				sign_ = -1f;
			}
		}
		KyVector2f xVec_ = ballVec_.orth( wallVec_ );
		KyVector2f yVec_ = ballVec_.orth( owallVec_ ).mulLocal( sign_ * elast_ );
		return xVec_.addLocal( yVec_ );
	}//}}}

	//Ball line vs Wall point
	static float[] calcBallLineProgress( KyFloatComparator cmp_, KyVector2f a1_, KyVector2f b1_, KyVector2f a2_, KyVector2f b2_, KyVector2f ww_ )
	{//{{{
		double p_ = ww_.x;
		double q_ = ww_.y;
		double a_ = a1_.x;
		double b_ = a1_.y;
		double c_ = b1_.x;
		double d_ = b1_.y;
		double e_ = a2_.x;
		double f_ = a2_.y;
		double g_ = b2_.x;
		double h_ = b2_.y;

		//aa_*k_*k_ + bb_*k_ + cc_ = 0
		double aa_ = (c_-g_)*(f_-b_) + (h_-d_)*(e_-a_);
		double bb_ = p_*(f_-b_+d_-h_) + b_*(2.0*c_-g_) -c_*f_ -a_*(d_-h_) - q_*(e_-a_+c_-g_) -d_*(a_-e_);
		double cc_ = p_*(b_-d_) + c_*(q_-b_) + a_*(d_-q_);

		double k_[] = null;
		if( cmp_.equals( aa_, 0f ) ){
			if( cmp_.equals( bb_, 0f ) ){
				return null;
			}
			else{
				k_ = new double[]{-(cc_/bb_)};
			}
		}
		else{
			double rt_ = Math.sqrt(bb_*bb_ -4.0*aa_*cc_);
			k_ = new double[] {
				(-bb_ + rt_) / (2.0 * aa_),
				(-bb_ - rt_) / (2.0 * aa_)
			};
		}
		int jg_ = 0;
		for( int i_ = 0; i_ < k_.length; ++i_ ){
			if( cmp_.less( k_[i_], 0f ) ){
				continue;
			}
			if( cmp_.more( k_[i_], 1f ) ){
				continue;
			}
			// dot check
			double ax_ = a_+(e_-a_)*k_[i_];
			double ay_ = b_+(f_-b_)*k_[i_];
			double bx_ = c_+(g_-c_)*k_[i_];
			double by_ = d_+(h_-d_)*k_[i_];
			
			double vabx_ = bx_ - ax_;
			double vaby_ = by_ - ay_;
			double vawx_ = p_ - ax_;
			double vawy_ = q_ - ay_;
			double dot1_ = vabx_*vawx_ + vaby_*vawy_;
			if( cmp_.less( (float)dot1_, 0f ) ){
				continue;
			}
			double vbax_ = -vabx_;
			double vbay_ = -vaby_;
			double vbwx_ = p_ - bx_;
			double vbwy_ = q_ - by_;
			double dot2_ = vbax_*vbwx_ + vbay_*vbwy_;
			if( cmp_.less( (float)dot2_, 0f ) ){
				continue;
			}
			jg_ |= (1 << i_);
		}
		if( (jg_ & 3 ) == 3 ){
			if( k_[0] < k_[1] ){
				return new float[] { (float)k_[0], (float)k_[1] };
			}
			else{
				return new float[] { (float)k_[1], (float)k_[0] };
			}
		}
		else if( (jg_ & 1) == 1 ){
			return new float[] { (float)k_[0] };
		}
		else if( (jg_ & 2) == 2 ){
			return new float[] { (float)k_[1] };
		}
		else{
			return null;
		}
	}//}}}
	static float[] calcBallPointProgress( KyFloatComparator cmp_, KyVector2f a_, KyVector2f b_, KyVector2f w_, float r_ )
	{//{{{
		float[] k_ = calcIntersection( cmp_, a_, b_, w_, r_ );
		int jg_ = 0;
		for( int i_ = 0; i_ < k_.length; ++i_ ){
			if( cmp_.less( k_[i_], 0f ) ){
				continue;
			}
			if( cmp_.more( k_[i_], 1f ) ){
				continue;
			}
			jg_ |= (1 << i_);
		}
		if( (jg_ & 3 ) == 3 ){
			if( k_[0] < k_[1] ){
				return new float[] { (float)k_[0], (float)k_[1] };
			}
			else{
				return new float[] { (float)k_[1], (float)k_[0] };
			}
		}
		else if( (jg_ & 1) == 1 ){
			return new float[] { (float)k_[0] };
		}
		else if( (jg_ & 2) == 2 ){
			return new float[] { (float)k_[1] };
		}
		else{
			return null;
		}
	}//}}}
	static float[] calcBallLineProgress( KyFloatComparator cmp_, KyVector2f a_, KyVector2f b_, KyVector2f c_, KyVector2f g_, float r_ )
	{//{{{
		double a1_ = a_.x;
		double a2_ = a_.y;
		double b1_ = b_.x;
		double b2_ = b_.y;
		double c1_ = c_.x;
		double c2_ = c_.y;
		double g1_ = g_.x;
		double g2_ = g_.y;
		double ba1_ = b1_ - a1_;
		double ba2_ = b2_ - a2_;
		double ca1_ = c1_ - a1_;
		double ca2_ = c2_ - a2_;

		if( cmp_.equals( ba1_, 0.0 ) && cmp_.equals( ba2_, 0.0 ) ){
			return null;
		}

		double kdiv_ = ca1_*ba2_ - ca2_*ba1_;

		if( cmp_.equals( kdiv_, 0.0 ) ){
			return null;
		}

		double pdiv_ = Math.sqrt( ba2_*ba2_ + ba1_*ba1_ );
		double p1_[] = new double[]{ g1_ + r_*ba2_/pdiv_, g1_ - r_*ba2_/pdiv_ };
		double p2_[] = new double[]{ g2_ + r_*ba1_/pdiv_, g2_ - r_*ba1_/pdiv_ };
		float[] k_ = new float[]{-1,-1,-1,-1};
		
		double r2_ = r_*r_;
		for( int i_ = 0; i_ < 2; ++i_ ){
			for( int j_ = 0; j_ < 2; ++j_ ){
				double gp1_ = g1_ - p1_[i_];
				double gp2_ = g2_ - p2_[j_];
				double pa1_ = p1_[i_] - a1_;
				double pa2_ = p2_[j_] - a2_;
				double kk_ = (pa1_*ba2_ - pa2_*ba1_)/kdiv_;
				if( !cmp_.more( (pa1_-kk_*ca1_)*ba1_+ (pa2_-kk_*ca2_)*ba2_, 0 ) ){
					continue;
				}
				if( !cmp_.more( -(pa1_-kk_*ca1_ -ba1_)*ba1_- (pa2_-kk_*ca2_-ba2_)*ba2_, 0 ) ){
					continue;
				}
				k_[i_*2+j_] = (float)kk_;
			}
		}

		for( int i_ = 0; i_ < 4; ++i_ ){
			for( int j_ = i_+1; j_ < 4; ++j_ ){
				if( k_[i_] > k_[j_] ){
					float tmp_ = k_[i_];
					k_[i_] = k_[j_];
					k_[j_] = tmp_;
				}
			}
		}
		int min_ = 0;
		for( ; min_ < 4; ++min_ ){
			if( cmp_.lessEquals( 0, k_[min_] ) ){
				break;
			}
		}
		int max_ = min_;
		for( ; max_ < 4; ++max_ ){
			if( cmp_.less( 1, k_[max_] ) ){
				break;
			}
		}
		int len_ = max_ - min_;
		if( len_ == 0 ){
			return null;
		}
		else if( len_ == 1 ){
			return new float[]{ k_[min_] };
		}
		else if( len_ == 2 ){
			return new float[]{ k_[min_], k_[min_+1] };
		}
		else if( len_ == 3 ){
			return new float[]{ k_[min_], k_[min_+1], k_[min_+2] };
		}
		else {
			assert( len_ == 4 );
			return new float[]{ k_[min_], k_[min_+1], k_[min_+2], k_[min_+3] };
		}
	}//}}}

	static float[] calcCircleProgress( KyFloatComparator cmp_, KyVector2f a1_, float r1_, KyVector2f b1_, float r2_, KyVector2f a2_ )
	{//{{{
		double a_ = a1_.x;
		double b_ = a1_.y;
		double c_ = b1_.x;
		double d_ = b1_.y;
		double p_ = a2_.x;
		double q_ = a2_.y;
		double l_ = c_ - a_;
		double m_ = d_ - b_;
		double n_ = p_ - a_;
		double o_ = q_ - b_;
		double vr_ = r2_ - r1_;
		double aa_ = l_*l_+m_*m_-vr_*vr_;
		double bb_ = -l_*n_-m_*o_-vr_*r1_;
		double cc_ = n_*n_+o_*o_-r1_*r1_;
		if( cmp_.equals( aa_, 0.0 ) ){
			if( cmp_.equals( bb_, 0.0 ) ){
				return null;
			}
			else{
				return new float[] { (float)(cc_/(2.0*bb_)) };
			}
		}
		double rt_ = Math.sqrt( bb_*bb_-aa_*cc_ );
		double[] k_ = new double[] { (-bb_-rt_)/aa_, (-bb_+rt_)/aa_ };
		int jg_ = 0;
		for( int i_ = 0; i_ < 2; ++i_ ){
			if( cmp_.less( k_[i_], 0.0 ) ){
				continue;
			}
			if( cmp_.more( k_[i_], 1.0 ) ){
				continue;
			}
			jg_ |= 1 << i_;
		}
		if( (jg_ & 3) == 3 ){
			if( k_[0] < k_[1] ){
				return new float[]{ (float)k_[0], (float)k_[1] };
			}
			else{
				return new float[]{ (float)k_[1], (float)k_[0] };
			}
		}
		else if( (jg_ & 1) == 1 ){
			return new float[]{ (float)k_[0] };
		}
		else if( (jg_ & 2) == 2 ){
			return new float[]{ (float)k_[1] };
		}
		else{
			return null;
		}
	}//}}}
	static float[] calcCircleProgress( KyFloatComparator cmp_, KyVector2f a_, float r1_, KyVector2f b_, float r2_, KyVector2f c_, KyVector2f d_ )
	{//{{{
		double a1_ = a_.x;
		double a2_ = a_.y;
		double b1_ = b_.x;
		double b2_ = b_.y;
		double c1_ = c_.x;
		double c2_ = c_.y;
		double d1_ = d_.x;
		double d2_ = d_.y;
		double s1_ = c1_ - a1_;
		double s2_ = c2_ - a2_;
		double t1_ = b1_ - a1_;
		double t2_ = b2_ - a2_;
		double u1_ = d1_ - c1_;
		double u2_ = d2_ - c2_;
		double uuuu_ = u1_*u1_+u2_*u2_;
		double tutu_ = t1_*u1_+t2_*u2_;
		double susu_ = s1_*u1_+s2_*u2_;
		double[] m_ = null;
		if( cmp_.equals( uuuu_, 0.0 ) ){
			if( cmp_.equals( tutu_, 0.0 ) ){
				return null;
			}
			m_ = new double[]{ (-susu_/tutu_) };
		}
		else{
			double vr_ = r2_-r1_;

			double aa_ = (t1_*t1_+t2_*t2_ -vr_*vr_)*uuuu_ - tutu_*tutu_;
			double bb_ = (t1_*s1_+t2_*s2_ +vr_*r1_)*uuuu_ - tutu_*susu_;
			double cc_ = (s1_*s1_+s2_*s2_ -r1_*r1_)*uuuu_ - susu_*susu_;

			if( cmp_.equals( aa_, 0.0 ) ){
				if( cmp_.equals( bb_, 0.0 ) ){
					return null;
				}
				else{
					m_ = new double[] { bb_/(cc_*2.0) };
				}
			}
			else{
				double rt_ = Math.sqrt( bb_*bb_-aa_*cc_ );
				double k1_ = (bb_-rt_)/aa_;
				double k2_ = (bb_+rt_)/aa_;
				m_ = new double[] { k1_, k2_ };
			}
		}
		int jg_ = 0;
		for( int i_ = 0; i_ < m_.length; ++i_ ){
			if( cmp_.less( m_[i_], 0.0 ) ){
				continue;
			}
			if( cmp_.more( m_[i_], 1.0 ) ){
				continue;
			}
			if( !cmp_.equals( uuuu_, 0.0 ) ){
				double n_ = (m_[i_]*tutu_-susu_)/uuuu_;
				if( cmp_.less( n_, 0.0 ) ){
					float[] prg_ = calcCircleProgress( cmp_, a_, r1_, b_, r2_, c_ );
					if( prg_ == null ){
						continue;
					}
					m_[i_] = prg_[0];
				}
				else if( cmp_.more( n_, 1.0 ) ){
					float[] prg_ = calcCircleProgress( cmp_, a_, r1_, b_, r2_, d_ );
					if( prg_ == null ){
						continue;
					}
					m_[i_] = prg_[0];
				}
			}
			jg_ |= 1 << i_;
		}
		if( (jg_ & 3) == 3 ){
			if( m_[0] < m_[1] ){
				return new float[]{ (float)m_[0], (float)m_[1] };
			}
			else{
				return new float[]{ (float)m_[1], (float)m_[0] };
			}
		}
		else if( (jg_ & 1) == 1 ){
			return new float[]{ (float)m_[0] };
		}
		else if( (jg_ & 2) == 2 ){
			return new float[]{ (float)m_[1] };
		}
		else{
			return null;
		}
	}//}}}
	static float[] calcCircleProgress( KyFloatComparator cmp_, KyVector2f a_, float r1_, KyVector2f b_, float r2_, KyVector2f c_, float r3_ )
	{//{{{
		double a1_ = a_.x;
		double a2_ = a_.y;
		double b1_ = b_.x;
		double b2_ = b_.y;
		double c1_ = c_.x;
		double c2_ = c_.y;
		double e_ = r1_;
		double f_ = r2_;
		double g_ = r3_;

		double b1a1_ = b1_ - a1_;
		double b2a2_ = b2_ - a2_;
		double a1c1_ = a1_ - c1_;
		double a2c2_ = a2_ - c2_;
		double fe_ = f_-e_;
		double eg_ = e_+g_;

		double aa_ = b1a1_*b1a1_ + b2a2_*b2a2_ - fe_*fe_;
		double bb_ = b1a1_*a1c1_ + b2a2_*a2c2_ - eg_*fe_;
		double cc_ = a1c1_*a1c1_ + a2c2_*a2c2_ - eg_*eg_;

		double mLs_[] = null;
		if( cmp_.equals( aa_, 0f ) ){
			if( cmp_.equals( bb_, 0f ) ){
				return null;
			}
			else{
				mLs_ = new double[]{(-cc_/(2.0*bb_))};
			}
		}
		else{
			double rt_ = Math.sqrt(bb_*bb_ -aa_*cc_);
			mLs_ = new double[] {
				(-bb_ + rt_) / aa_,
				(-bb_ - rt_) / aa_
			};
		}
		int jg_ = 0;
		for( int i_ = 0; i_ < mLs_.length; ++i_ ){
			if( cmp_.less( mLs_[i_], 0.0 ) ){
				continue;
			}
			if( cmp_.more( mLs_[i_], 1.0 ) ){
				continue;
			}
			jg_ |= (1 << i_);
		}
		if( jg_ == 0 ){
			return null;
		}
		else{
			if( (jg_ & 3 ) == 3 ){
				if( mLs_[0] < mLs_[1] ){
					return new float[]{ (float)mLs_[0], (float)mLs_[1] };
				}
				else{
					return new float[]{ (float)mLs_[1], (float)mLs_[0] };
				}
			}
			else if( (jg_ & 1) == 1 ){
				return new float[]{ (float)mLs_[0] };
			}
			else if( (jg_ & 2) == 2 ){
				return new float[]{ (float)mLs_[1] };
			}
			else{
				throw new AssertionError();
			}
		}
	}//}}}

	static ColShape collideWithShape( KyFloatComparator cmp_, IKyAction1<ColResult> scan_, ColShape ballShape_, ColShape nextShape_, ColShape wallShape_ )
	{//{{{
		assert( scan_ != null );
		assert( ballShape_ != null );
		assert( wallShape_ != null );
		assert( nextShape_ != null );

		if( ballShape_ instanceof ColCircle ){
			assert( nextShape_ instanceof ColCircle );
			if( wallShape_ instanceof ColCircle ){
				collideWith( cmp_, scan_, (ColCircle)ballShape_, (ColCircle)nextShape_, (ColCircle)wallShape_ );
			}
			else if( wallShape_ instanceof ColVectors ){
				collideWith( cmp_, scan_, (ColCircle)ballShape_, (ColCircle)nextShape_, (ColVectors)wallShape_ );
			}
			else{
				throw new AssertionError();
			}
		}
		else if( ballShape_ instanceof ColVectors ){
			assert( nextShape_ instanceof ColVectors );
			if( wallShape_ instanceof ColCircle ){
				collideWith( cmp_, scan_, (ColVectors)ballShape_, (ColVectors)nextShape_, (ColCircle)wallShape_ );
			}
			else if( wallShape_ instanceof ColVectors ){
				collideWith( cmp_, scan_, (ColVectors)ballShape_, (ColVectors)nextShape_, (ColVectors)wallShape_ );
			}
			else{
				throw new AssertionError();
			}
		}
		else{
			throw new AssertionError();
		}
		return nextShape_;
	}//}}}
	static ColShape collideWithShape( KyFloatComparator cmp_, IKyAction1<ColResult> scan_, ColShape ballShape_, ColShape wallShape_ )
	{//{{{
		assert( scan_ != null );
		assert( ballShape_ != null );
		assert( wallShape_ != null );

		ColShape nextShape_ = ballShape_.generateNext();
		return collideWithShape( cmp_, scan_, ballShape_, nextShape_, wallShape_ );
	}//}}}
	static void collideWith( KyFloatComparator cmp_, IKyAction1<ColResult> scan_, ColVectors ballShape1_, ColVectors ballShape2_, ColVectors wallShape_ )
	{//{{{
		ColResult res_ = new ColResult(ballShape1_, ballShape2_, wallShape_);
		KyVector2f[] ballPtLs1_ = ballShape1_.currentPoints;
		KyVector2f[] ballPtLs2_ = ballShape2_.currentPoints;
		KyVector2f[] wallPtLs_ = wallShape_.currentPoints;
		assert( ballPtLs1_.length == ballPtLs2_.length );

		int ballLnNum_ = ballPtLs1_.length;
		int ballPtNum_ = ballLnNum_;
		if( isClosedPoints( ballPtLs1_ ) ){
			--ballPtNum_;
		}
		int wallLnNum_ = wallPtLs_.length;
		int wallPtNum_ = wallLnNum_;
		if( isClosedPoints( wallPtLs_ ) ){
			--wallPtNum_;
		}

		//ball's point check
		for( int i_ = 0; i_ < ballPtNum_; ++i_ ){
			KyVector2f b1_ = ballPtLs1_[ i_ ];
			KyVector2f b2_ = ballPtLs2_[ i_ ];
			float bd_ = KyVector2f.getDistance(b1_, b2_);
			if( !isClosedPoints( ballPtLs1_ ) ){
				assert( !isClosedPoints( ballPtLs2_ ) );
				for( int j_ = 0; j_ < wallPtNum_; ++j_ ){
					KyVector2f w_ = wallPtLs_[ j_ ];
					if( !b2_.equals( cmp_, w_ ) && isPointOnLine( cmp_, w_, b1_, b2_ ) ){
						res_.ballIndex1 = i_;
						res_.ballIndex2 = -1;
						res_.wallIndex1 = j_;
						res_.wallIndex2 = -1;
						if( cmp_.equals( bd_, 0f ) ){
							res_.progress = -1f;
						}
						else{
							res_.progress = KyVector2f.getDistance( w_, b2_ )/bd_;
						}
						scan_.call( res_ );
					}
				}
			}
			for( int j_ = 1; j_ < wallLnNum_; ++j_ ){
				KyVector2f w1_ = wallPtLs_[ j_-1 ];
				KyVector2f w2_ = wallPtLs_[ j_ ];
				if( isPointOnLine( cmp_, b1_, w1_, w2_ ) ){
					if( isPointInPoints( cmp_, b2_, wallPtLs_ )
						|| KyVector2f.isIntersectedLines( cmp_, b1_, b2_, wallPtLs_ ) ){
						res_.ballIndex1 = i_;
						res_.ballIndex2 = -1;
						res_.wallIndex1 = j_-1;
						res_.wallIndex2 = j_;
						res_.progress = 0f;
						scan_.call( res_ );
						continue;
					}
				}
				if( KyVector2f.isIntersected( cmp_, b1_, b2_, w1_, w2_ ) ){
					res_.ballIndex1 = i_;
					res_.ballIndex2 = -1;
					res_.wallIndex1 = j_-1;
					res_.wallIndex2 = j_;
					if( cmp_.equals( bd_, 0f ) ){
						res_.progress = -1f;
					}
					else{
						KyVector2f m_ = KyVector2f.getIntersection( b1_, b2_, w1_, w2_ );
						res_.progress = KyVector2f.getDistance( b1_, m_ )/bd_;
					}
					scan_.call( res_ );
					continue;
				}
			}
		}

		//ball's edge check
		for( int i_ = 1; i_ < ballLnNum_; ++i_ ){
			KyVector2f b1_ = ballPtLs1_[ i_ - 1 ];
			KyVector2f b2_ = ballPtLs1_[ i_ ];
			KyVector2f a1_ = ballPtLs2_[ i_ - 1 ];
			KyVector2f a2_ = ballPtLs2_[ i_ ];
			KyVector2f a12_ = a2_.sub( a1_ );
			KyVector2f b12_ = b2_.sub( b1_ );
			KyVector2f b21_ = b1_.sub( b2_ );
			KyVector2f b1a1_ = a1_.sub( b1_ );
			KyVector2f b2a2_ = a2_.sub( b2_ );
			int c1_ = cmp_.compare0( b1a1_.cross( b12_ ) );
			int c2_ = cmp_.compare0( b2a2_.cross( b12_ ) );
			if( c1_ == 0 && c2_ == 0 ){
				float dab1_ = KyVector2f.getDistance( a1_, b1_ );
				KyVector2f d1_ = a2_;
				KyVector2f d2_ = b2_;
				KyVector2f d3_ = a1_;
				KyVector2f d4_ = b1_;
				if( b1a1_.dot( b12_ ) < 0 ){
					d1_ = a1_;
					d2_ = b1_;
					d3_ = a2_;
					d4_ = b2_;
				}
				for( int j_ = 1; j_ < wallLnNum_; ++j_ ){
					KyVector2f w1_ = wallPtLs_[j_-1];
					KyVector2f w2_ = wallPtLs_[j_];
					if( KyVector2f.isIntersected( cmp_, w1_, w2_, d1_, d4_ ) ){
						res_.ballIndex1 = i_-1;
						res_.ballIndex2 = i_;
						res_.wallIndex1 = j_-1;
						res_.wallIndex2 = j_;
						if( cmp_.equals( dab1_, 0f ) ){
							res_.progress = -1f;
						}
						else{
							KyVector2f m_ = KyVector2f.getIntersection( w1_, w2_, d1_, d4_ );
							res_.progress = KyVector2f.getDistance( m_, d2_ ) / dab1_;
						}
						scan_.call( res_ );
					}
				}
				/*
				for( int j_ = 0; j_ < wallPtNum_; ++j_ ){
					KyVector2f w_ = wallPtLs_[j_];
					if( !d1_.equals( w_) && KyVector2f.isPointOnLine( w_, d1_, d2_ ) ){
						ColResult res_ = new ColResult( ballShape1_, ballShape2_, wallShape_ );
						res_.ballIndex1 = i_-1;
						res_.ballIndex2 = i_;
						res_.wallIndex1 = j_;
						resLs_.add( res_ );
					}
				}
				*/
			}
			else{
				KyVector2f[] bLs_ = new KyVector2f[]{ a1_, a2_, b2_, b1_, a1_ };
				if( c1_*c2_ < 0 ){
					bLs_[1] = b2_;
					bLs_[2] = a2_;
				}
				BALL_SQUARE_WALL_EDGE_CHECK:
				for( int j_ = 1; j_ < wallLnNum_; ++j_ ){
					KyVector2f w1_ = wallPtLs_[j_-1];
					KyVector2f w2_ = wallPtLs_[j_];
					for( int k_ = 0; k_ < 2; ++k_ ){
						//<KyVector2f#isPointOnLine's algorithm>
						KyVector2f ww1_ = wallPtLs_[j_ + k_ - 1];
						KyVector2f ww2_ = wallPtLs_[j_ + (k_ + 1)%2 - 1];
						KyVector2f b1w1_ = ww1_.sub( b1_ );
						if( cmp_.equals( b1w1_.cross( b12_ ), 0f ) ){
							KyVector2f b2w1_ = ww1_.sub( b2_ );
							if( !cmp_.isNegative( b1w1_.dot( b12_ ) )
							 && !cmp_.isNegative( b2w1_.dot( b21_ ) ) ){
								//<KyVector2f#isPointOnLine return true>
								float blen_ = b21_.magnitude();
								if( cmp_.equals( blen_, 0f ) ){
									continue;
								}
								float r_ = b1w1_.magnitude() / blen_;
								KyVector2f vv_ = a1_.add( a12_.mul( r_ ) );
								KyVector2f w1v_ = vv_.sub( ww1_ );
								KyVector2f w1w2_ = ww2_.sub( ww1_ );
								int cv_ = cmp_.compare0( b12_.cross( w1v_ ) );
								int cw_ = cmp_.compare0( b12_.cross( w1w2_ ) );
								if( cv_*cw_ > 0 ){
									res_.ballIndex1 = i_-1;
									res_.ballIndex2 = i_;
									res_.wallIndex1 = j_-1;
									res_.wallIndex2 = j_;
									res_.progress = 0f;
									scan_.call( res_ );
									continue BALL_SQUARE_WALL_EDGE_CHECK;
								}
								//</KyVector2f#isPointOnLine return true>
							}
						}
						//</KyVector2f#isPointOnLine's algorithm>
					}
					boolean reg_ = false;
					float p_ = 2f;
					for( int k_ = 1; k_ < bLs_.length; ++k_ ){
						KyVector2f d1_ = bLs_[ k_ - 1];
						KyVector2f d2_ = bLs_[ k_ ];
						if( KyVector2f.isIntersected( cmp_, w1_, w2_, d1_, d2_ ) ){
							reg_ = true;
							KyVector2f m_ = KyVector2f.getIntersection( w1_, w2_, d1_, d2_ );
							float[] pLs_ = calcBallLineProgress( cmp_, b1_, b2_, a1_, a2_, m_ );
							if( pLs_ != null && cmp_.less( pLs_[0], p_) ){
								p_ = pLs_[0];
							}
						}
					}
					if( reg_ ){
						res_.ballIndex1 = i_-1;
						res_.ballIndex2 = i_;
						res_.wallIndex1 = j_-1;
						res_.wallIndex2 = j_;
						if( cmp_.more(p_, 1f) ){
							res_.progress = -1;
						}
						else{
							res_.progress = p_;
						}
						scan_.call( res_ );
					}
				}
				if( b1a1_.equals( cmp_, b2a2_ ) ){
					KyVector2f wVec_ = b1a1_.mul( -1f );
					for( int j_ = 0; j_ < wallPtNum_; ++j_ ){
						KyVector2f w1_ = wallPtLs_[j_];
						KyVector2f w2_ = w1_.add( wVec_ );
						float wd_ = KyVector2f.getDistance(w1_, w2_);
						if( isPointOnLine( cmp_, w1_, b1_, b2_ ) ){
							if( isPointInPoints( cmp_, w2_, ballPtLs1_ )
								|| KyVector2f.isIntersectedLines( cmp_, w1_, w2_, ballPtLs1_ )
							){
								res_.ballIndex1 = i_-1;
								res_.ballIndex2 = i_;
								res_.wallIndex1 = j_;
								res_.wallIndex2 = -1;
								res_.progress = 0f;
								scan_.call( res_ );
								continue;
							}
						}
						if( KyVector2f.isIntersected( cmp_, w1_, w2_, b1_, b2_ ) ){
							res_.ballIndex1 = i_-1;
							res_.ballIndex2 = i_;
							res_.wallIndex1 = j_;
							res_.wallIndex2 = -1;
							if( cmp_.equals( wd_, 0f ) ){
								res_.progress = -1f;
							}
							else{
								KyVector2f m_ = KyVector2f.getIntersection( w1_, w2_, b1_, b2_ );
								res_.progress = KyVector2f.getDistance( w1_, m_ )/wd_;
							}
							scan_.call( res_ );
							continue;
						}
					}
				}
				else{
					for( int j_ = 0; j_ < wallPtNum_; ++j_ ){
						KyVector2f w_ = wallPtLs_[j_];
						if( isPointInQuadrangle( cmp_, w_, bLs_[0], bLs_[1], bLs_[2], bLs_[3] ) ){
							if( !isPointOnLine( cmp_, w_, a1_, a2_ )
							&& !isPointOnLine( cmp_, w_, b1_, b2_ ) ){
								float[] p_ = calcBallLineProgress( cmp_, b1_, b2_, a1_, a2_, w_ );
								res_.ballIndex1 = i_-1;
								res_.ballIndex2 = i_;
								res_.wallIndex1 = j_;
								res_.wallIndex2 = -1;
								if( p_ == null ){
									res_.progress = -1f;
								}
								else{
									res_.progress = p_[0];
								}
								scan_.call( res_ );
							}
						}
					}
				}
			}
		}
	}//}}}
	static void collideWith( KyFloatComparator cmp_, IKyAction1<ColResult> scan_, ColCircle ballShape1_, ColCircle ballShape2_, ColVectors wallShape_ )
	{//{{{
		ColResult res_ = new ColResult(ballShape1_, ballShape2_, wallShape_);
		KyVector2f a_ = ballShape1_.transform.position;
		KyVector2f b_ = ballShape2_.transform.position;
		float aRad_ = ballShape1_.getRadius();
		float bRad_ = ballShape2_.getRadius();
		KyVector2f[] wallPtLs_ = wallShape_.currentPoints;

		int wallLnNum_ = wallPtLs_.length;
		int wallPtNum_ = wallLnNum_;
		if( isClosedPoints( wallPtLs_ ) ){
			--wallPtNum_;
		}

		KyVector2f ab_ = b_.sub( a_ );
		KyVector2f uv_ = ab_.normalize();
		KyVector2f nv1_ = new KyVector2f( -uv_.y, uv_.x );
		KyVector2f nv2_ = new KyVector2f( uv_.y, -uv_.x );
		KyVector2f loc1_ = a_.add( nv1_.mul( aRad_ ) );
		KyVector2f loc2_ = a_.add( nv2_.mul( aRad_ ) );
		KyVector2f loc3_ = b_.add( nv1_.mul( bRad_ ) );
		KyVector2f loc4_ = b_.add( nv2_.mul( bRad_ ) );
		KyVector2f[] locLs_ = new KyVector2f[]{ loc1_, loc2_, loc4_, loc3_, loc1_ };
		KyVector2f loc34_ = loc4_.sub( loc3_ );
		float bRad2_ = bRad_*bRad_;

		for( int i_ = 0; i_ < wallPtNum_; ++i_ ){
			KyVector2f w_ = wallPtLs_[i_];
			boolean reg_ = false;
			if( cmp_.less( w_.magnitudeSquared( b_ ), bRad2_ ) ){
				reg_ = true;
			}
			if( !reg_ && isPointInQuadrangle( cmp_, w_, locLs_ )
				&& !isPointOnLines( cmp_, w_, locLs_ )
			){
				reg_ = true;
			}
			if( reg_ ){
				float[] prog_ = calcCircleProgress( cmp_, a_, aRad_, b_, bRad_, w_ );
				if( prog_ != null ){
					res_.ballIndex1 = 0;
					res_.ballIndex2 = -1;
					res_.wallIndex1 = i_;
					res_.wallIndex2 = -1;
					res_.progress = prog_[0];
					scan_.call( res_ );
				}
			}
		}

		WALL_EDGE_LOOP:
		for( int i_ = 1; i_ < wallLnNum_; ++i_ ){
			KyVector2f w1_ = wallPtLs_[i_-1];
			KyVector2f w2_ = wallPtLs_[i_];
			boolean reg_ = false;
			for( int j_ = 1; j_ < locLs_.length; ++j_ ){
				KyVector2f l1_ = locLs_[j_-1];
				KyVector2f l2_ = locLs_[j_];
				if( KyVector2f.isIntersected( cmp_, l1_, l2_, w1_, w2_ ) ){
					reg_ = true;
					break;
				}
			}
			if( !reg_ && hasIntersection( cmp_, w1_, w2_, b_, bRad_ ) ){
				reg_ = true;
			}
			if( !reg_ &&(
				( isPointOnLines( cmp_, w1_, locLs_)
				|| isPointInQuadrangle( cmp_, w1_, locLs_ ) )
				&& ( isPointOnLines( cmp_, w2_, locLs_)
				|| isPointInQuadrangle( cmp_, w2_, locLs_ ) )
			)){
				reg_ = true;
			}
			if( reg_ ){
				float[] prog_ = calcCircleProgress( cmp_, a_, aRad_, b_, bRad_, w1_, w2_ );
				if( prog_ != null ){
					res_.ballIndex1 = 0;
					res_.ballIndex2 = -1;
					res_.wallIndex1 = i_-1;
					res_.wallIndex2 = i_;
					res_.progress = prog_[0];
					scan_.call( res_ );
				}
			}
		}
	}//}}}
	static void collideWith( KyFloatComparator cmp_, IKyAction1<ColResult> scan_, ColVectors ballShape1_, ColVectors ballShape2_, ColCircle wallShape_ )
	{//{{{
		ColResult res_ = new ColResult(ballShape1_, ballShape2_, wallShape_);
		KyVector2f[] ballPtLs1_ = ballShape1_.currentPoints;
		KyVector2f[] ballPtLs2_ = ballShape2_.currentPoints;
		KyVector2f wallPt_ = wallShape_.transform.position;
		float wRad_ = wallShape_.getRadius();

		assert( ballPtLs1_.length == ballPtLs2_.length );

		int ballLnNum_ = ballPtLs1_.length;
		int ballPtNum_ = ballLnNum_;
		if( isClosedPoints( ballPtLs1_ ) ){
			--ballPtNum_;
		}

		//ball's point check
		for( int i_ = 0; i_ < ballPtNum_; ++i_ ){
			KyVector2f b1_ = ballPtLs1_[ i_ ];
			KyVector2f b2_ = ballPtLs2_[ i_ ];
			boolean reported_ = false;
			float[] prog_ = calcBallPointProgress( cmp_, b1_, b2_, wallPt_, wRad_ );
			KyVector2f b1b2_ = b2_.sub( b1_ );
			if( prog_ != null ){
				KyVector2f b1b3_ = b1b2_.mul( prog_[0] );
				KyVector2f b1w_ = wallPt_.sub( b1_ );
				KyVector2f wb3_ = b1b3_.sub( b1w_ );
				if( !cmp_.equals( wb3_.dot( b1b2_ ), 0f ) ){
					res_.ballIndex1 = i_;
					res_.wallIndex1 = 0;
					res_.ballIndex2 = -1;
					res_.wallIndex2 = -1;
					res_.progress = prog_[0];
					scan_.call( res_ );
					reported_ = true;
				}
			}
			if( !reported_ ){
				if( cmp_.less( b2_.magnitudeSquared( wallPt_ ), wRad_*wRad_ ) ){
					res_.ballIndex1 = i_;
					res_.wallIndex1 = 0;
					res_.ballIndex2 = -1;
					res_.wallIndex2 = -1;
					res_.progress = -1f;
					scan_.call( res_ );
				}
			}
		}

		//ball's edge check
		BALL_EDGE_CHECK:
		for( int i_ = 1; i_ < ballLnNum_; ++i_ ){
			KyVector2f b1_ = ballPtLs1_[ i_ - 1 ];
			KyVector2f b2_ = ballPtLs1_[ i_ ];
			KyVector2f a1_ = ballPtLs2_[ i_ - 1 ];
			KyVector2f a2_ = ballPtLs2_[ i_ ];
			KyVector2f a12_ = a2_.sub( a1_ );
			KyVector2f b12_ = b2_.sub( b1_ );
			KyVector2f b21_ = b1_.sub( b2_ );
			KyVector2f b1a1_ = a1_.sub( b1_ );
			KyVector2f b2a2_ = a2_.sub( b2_ );
			int c1_ = cmp_.compare0( b1a1_.cross( b12_ ) );
			int c2_ = cmp_.compare0( b2a2_.cross( b12_ ) );
			if( c1_ == 0 && c2_ == 0 ){
				float dab1_ = KyVector2f.getDistance( a1_, b1_ );
				KyVector2f d1_ = a2_;
				KyVector2f d2_ = b2_;
				KyVector2f d3_ = a1_;
				KyVector2f d4_ = b1_;
				KyVector2f d2d1_ = b2a2_;
				if( cmp_.less( b1a1_.dot( b12_ ), 0f ) ){
					d1_ = a1_;
					d2_ = b1_;
					d3_ = a2_;
					d4_ = b2_;
					d2d1_ = b1a1_;
				}
				float progLs_[] = calcBallPointProgress( cmp_, d2_, d1_, wallPt_, wRad_ );
				if( progLs_ == null ){
					continue;
				}
				KyVector2f d2d5_ = d2d1_.mul( progLs_[0] );
				KyVector2f d2w_ = wallPt_.sub( d2_ );
				KyVector2f wd5_ = d2d5_.sub( d2w_ );
				if( cmp_.equals( wd5_.dot( d2d1_ ), 0f ) ){
					continue;
				}
				res_.ballIndex1 = i_-1;
				res_.ballIndex2 = i_;
				res_.wallIndex1 = 0;
				res_.wallIndex2 = -1;
				res_.progress = progLs_[0];
				scan_.call( res_ );
			}
			else{
				boolean calc_ = false;
				KyVector2f[] dLs_ = new KyVector2f[]{ a1_, a2_, b2_, b1_, a1_ };
				if( c1_*c2_ < 0 ){
					dLs_[1] = b2_;
					dLs_[2] = a2_;
				}
				for( int j_ = 0; j_ < 4; ++j_ ){
					float progLs_[] = calcBallPointProgress( cmp_, dLs_[j_], dLs_[j_+1], wallPt_, wRad_ );
					if( progLs_ == null ){
						continue;
					}
					KyVector2f d1d2_ = dLs_[j_+1].sub( dLs_[j_] );
					for( float prog_ : progLs_ ){
						KyVector2f d1d3_ = d1d2_.mul( prog_ );
						KyVector2f d1w_ = wallPt_.sub( dLs_[j_] );
						KyVector2f wd3_ = d1w_.sub( d1d3_ );
						if( cmp_.equals( wd3_.dot( d1d3_ ), 0f ) ){
							continue;
						}
						calc_ = true;
						break;
					}
					if( calc_ ){
						break;
					}
				}
				if( !calc_ && isPointInQuadrangle( cmp_, wallPt_, dLs_ ) ){
					calc_ = true;
				}
				if( !calc_ ){
					continue BALL_EDGE_CHECK;
				}
				KyVector2f e1_ = a1_.sub( b1_ );
				if( e1_.equals( cmp_, a2_.sub( b2_ ) ) ){
					float[] progLs_ = calcBallLineProgress( cmp_, b1_, b2_, a1_, wallPt_, wRad_ );
					if( progLs_ == null ){
						continue BALL_EDGE_CHECK;
					}
					KyVector2f b1w_ = wallPt_.sub( b1_ );
					for( float prog_ : progLs_ ){
						KyVector2f b1k1_ = b1a1_.mul( prog_ );
						KyVector2f wk1_ = b1w_.sub( b1k1_ );
						if( cmp_.equals( wk1_.dot( b1a1_ ), 0f ) ){
							continue;
						}
						res_.ballIndex1 = i_-1;
						res_.ballIndex2 = i_;
						res_.wallIndex1 = 0;
						res_.wallIndex2 = -1;
						res_.progress = prog_;
						scan_.call( res_ );
						break;
					}
				}
				else{
					res_.ballIndex1 = i_;
					res_.ballIndex2 = i_-1;
					res_.wallIndex1 = 0;
					res_.wallIndex2 = -1;
					res_.progress = -1;
					scan_.call( res_ );
				}
			}
		}
	}//}}}
	static void collideWith( KyFloatComparator cmp_, IKyAction1<ColResult> scan_, ColCircle ballShape1_, ColCircle ballShape2_, ColCircle wallShape_ )
	{//{{{
		ColResult res_ = new ColResult(ballShape1_, ballShape2_, wallShape_);
		KyVector2f a_ = ballShape1_.transform.position;
		KyVector2f b_ = ballShape2_.transform.position;
		KyVector2f c_ = wallShape_.transform.position;
		float aRad_ = ballShape1_.getRadius();
		float bRad_ = ballShape2_.getRadius();
		float cRad_ = wallShape_.getRadius();

		KyVector2f ab_ = b_.sub( a_ );
		KyVector2f uv_ = ab_.normalize();
		KyVector2f nv1_ = new KyVector2f( -uv_.y, uv_.x );
		KyVector2f nv2_ = new KyVector2f( uv_.y, -uv_.x );
		KyVector2f loc1_ = a_.add( nv1_.mul( aRad_ ) );
		KyVector2f loc2_ = a_.add( nv2_.mul( aRad_ ) );
		KyVector2f loc3_ = b_.add( nv1_.mul( bRad_ ) );
		KyVector2f loc4_ = b_.add( nv2_.mul( bRad_ ) );
		KyVector2f[] locLs_ = new KyVector2f[]{ loc1_, loc2_, loc4_, loc3_, loc1_ };
		KyVector2f loc34_ = loc4_.sub( loc3_ );

		float[] kLs_ = null;
		if( isPointInQuadrangle( cmp_, c_, locLs_ ) ){
			kLs_ = calcCircleProgress( cmp_, a_, aRad_, b_, bRad_, c_, cRad_ );
		}
		if( kLs_ == null ){
			float dist_ = bRad_+cRad_;
			if( cmp_.less( c_.magnitudeSquared( b_ ), dist_*dist_) ){
				kLs_ = calcCircleProgress( cmp_, a_, aRad_, b_, bRad_, c_, cRad_ );
			}
		}
		if( kLs_ == null ){
			for( int i_ = 1; i_ < 4; ++i_ ){
				KyVector2f lo1_ = locLs_[i_-1];
				KyVector2f lo2_ = locLs_[i_];
				if( hasIntersection( cmp_, lo1_, lo2_, c_, cRad_ ) ){
					kLs_ = calcCircleProgress( cmp_, a_, aRad_, b_, bRad_, c_, cRad_ );
					break;
				}
			}
		}
		if( kLs_ == null ){
			return;
		}
		res_.ballIndex1 = 0;
		res_.wallIndex1 = 0;
		res_.ballIndex2 = -1;
		res_.wallIndex2 = -1;
		res_.progress = kLs_[0];
		scan_.call( res_ );
		return;
	}//}}}
	public static boolean isSharedShape( KyFloatComparator cmp_, ColShape pShape_, ColShape qShape_ )
	{//{{{
		if( pShape_ instanceof ColCircle ){
			if( qShape_ instanceof ColCircle ){
				return isShared( cmp_, (ColCircle)pShape_, (ColCircle)qShape_ );
			}
			else if( qShape_ instanceof ColVectors ){
				return isShared( cmp_, (ColCircle)pShape_, (ColVectors)qShape_ );
			}
		}
		else if( pShape_ instanceof ColVectors ){
			if( qShape_ instanceof ColCircle ){
				return isShared( cmp_, (ColCircle)qShape_, (ColVectors)pShape_ );
			}
			else if( qShape_ instanceof ColVectors ){
				return isShared( cmp_, (ColVectors)pShape_, (ColVectors)qShape_ );
			}
		}
		throw new AssertionError();
	}//}}}
	public static boolean isShared( KyFloatComparator cmp_, ColCircle pShape_, ColCircle qShape_ )
	{//{{{
		KyVector2f pPos_ = pShape_.transform.position;
		KyVector2f qPos_ = qShape_.transform.position;
		float pRad_ = pShape_.getRadius();
		float qRad_ = qShape_.getRadius();
		float d_ = pRad_ + qRad_;
		return cmp_.less( pPos_.magnitudeSquared( qPos_ ), d_*d_ );
	}//}}}
	public static boolean isShared( KyFloatComparator cmp_, ColCircle pShape_, ColVectors qShape_ )
	{//{{{
		KyVector2f pPos_ = pShape_.transform.position;
		float pRad_ = pShape_.getRadius();
		KyVector2f[] qLs_ = qShape_.currentPoints;

		if( isClosedPoints( qLs_ ) ){
			if( isPointInPoints( cmp_, pPos_, qLs_ ) ){
				return true;
			}
		}
		for( int i_ = 1; i_ < qLs_.length; ++i_ ){
			KyVector2f q1_ = qLs_[i_-1];
			KyVector2f q2_ = qLs_[i_];
			float[] kLs_ = calcIntersection( cmp_, q1_, q2_, pPos_, pRad_ );
			if( kLs_ == null ){
				continue;
			}
			KyVector2f q1p_ = pPos_.sub( q1_ );
			KyVector2f q1q2_ = q2_.sub( q1_ );
			for( float k_ : kLs_ ){
				if( cmp_.less( k_, 0f ) ){
					continue;
				}
				if( cmp_.more( k_, 1f ) ){
					continue;
				}
				KyVector2f q1q3_ = q1q2_.mul( k_ );
				KyVector2f q3p_ = q1p_.sub( q1q3_ );
				if( !cmp_.equals( q1q2_.dot( q3p_ ), 0f )
					&& cmp_.lessEquals( q1q3_.magnitudeSquared( q1p_ ), pRad_*pRad_ )
				){
					return true;
				}
			}
		}
		return false;
	}//}}}
	public static boolean isShared( KyFloatComparator cmp_, ColVectors pShape_, ColVectors qShape_ )
	{//{{{
		KyVector2f[] pLs_ = pShape_.currentPoints;
		KyVector2f[] qLs_ = qShape_.currentPoints;
		if( isClosedPoints( pLs_ ) ){
			for( KyVector2f pt_ : qLs_ ){
				if( isPointInPoints( cmp_, pt_, pLs_ ) ){
					if( !isPointOnLines( cmp_, pt_, pLs_ ) ){
						return true;
					}
				}
			}
		}
		if( isClosedPoints( qLs_ ) ){
			for( KyVector2f pt_ : pLs_ ){
				if( isPointInPoints( cmp_, pt_, qLs_ ) ){
					if( !isPointOnLines( cmp_, pt_, qLs_ ) ){
						return true;
					}
				}
			}
		}
		for( int i_ = 1; i_ < pLs_.length; ++i_ ){
			KyVector2f p1_ = pLs_[i_-1];
			KyVector2f p2_ = pLs_[i_];
			for( int j_ = 1; i_ < qLs_.length; ++i_ ){
				KyVector2f q1_ = qLs_[j_-1];
				KyVector2f q2_ = qLs_[j_];
				if( KyVector2f.isIntersected( cmp_, p1_, p2_, q1_, q2_ ) ){
					return true;
				}
			}
		}
		return false;
	}//}}}
}//}}}
