package com.tryjava.datetime;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.Period;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.TextStyle;
import java.time.temporal.ChronoUnit;
import java.time.temporal.TemporalAdjusters;
import java.util.Date;
import java.util.Locale;
import java.util.concurrent.TimeUnit;

import com.tryjava.util.AppUtil;

/**
 * 日時を扱う練習
 *
 * 参考
 * <ul>
 * <li>Java日付処理メモ(Hishidama's Java Date Memo)<br>
 * http://www.ne.jp/asahi/hishidama/home/tech/java/date.html
 * <li>日時に関する情報を取得する(get) - Calendarクラス<br>
 * http://www.javadrive.jp/start/calendar/index2.html
 * <li>Java日付時刻APIメモ(Hishidama's Java8 Date and Time API Memo)<br>
 * http://www.ne.jp/asahi/hishidama/home/tech/java/datetime.html
 * <li>Java日付時刻フォーマッターメモ(Hishidama's Java8 DateTimeFormatter Memo)<br>
 * http://www.ne.jp/asahi/hishidama/home/tech/java/DateTimeFormatter.html
 * </ul>
 *
 * @since 2016/5/2
 */
public class DateTime01 {
	long msec1 = System.currentTimeMillis();
	Instant instant1 = Instant.now();
	LocalDateTime localDt1 = LocalDateTime.now();
	ZonedDateTime zonedDt1 = ZonedDateTime.now(ZoneId.systemDefault());

	public static void main(String[] args) {
		DateTime01 app = new DateTime01();
		app.run();
	}

	void run() {
		createDateTime();
		getDateTimePart();
		calcDateTime();
		useTimeUnit();
	}

	/**
	 * 現在/指定日時を作成する。
	 *
	 * Java8では、Java7までのDate,Calendarの代替として、次のクラスなどが導入された。
	 *
	 * <ul>
	 * <li>Instant - 日時（エポック秒）
	 * <li>LocalDateTime - タイムゾーンなし日時
	 * <li>ZonedDateTime - タイムゾーンあり日時
	 * </ul>
	 */
	void createDateTime() {
		System.out.println("--- " + AppUtil.getMethod() + " ---");

		// Instant
		// 現在日時
		Instant nowInstant = Instant.now();
		// long→Instant
		Instant instant2 = Instant.ofEpochMilli(msec1);
		// LocalDateTime→Instant
		Instant instant3b = localDt1.toInstant(ZoneId.systemDefault().getRules().getOffset(Instant.EPOCH));
		// ZonedDateTime→Instant
		Instant instant4ca = zonedDt1.toInstant();
		Instant instant4cb = Instant.from(zonedDt1);
		// 文字列→Instant
		Instant instant5 = Instant.parse("2007-12-03T10:15:30.00Z");

		System.out.println("Instant");
		System.out.println(nowInstant);
		System.out.println(instant2);
		System.out.println(instant3b);
		System.out.println(instant4ca);
		System.out.println(instant4cb);
		System.out.println(instant5);

		// LocalDateTime
		// 現在日時
		LocalDateTime nowLocalDt = LocalDateTime.now();
		// 指定日時
		LocalDateTime localDt2 = LocalDateTime.of(2016, 5, 2, 15, 0, 0);
		// long→LocalDateTime
		LocalDateTime localDt3 = LocalDateTime.ofInstant(Instant.ofEpochMilli(msec1), ZoneId.systemDefault());
		// Instant→LocalDateTime
		LocalDateTime localDt4 = LocalDateTime.ofInstant(instant1, ZoneId.systemDefault());
		// ZonedDateTime→LocalDateTime
		LocalDateTime localDt5a = zonedDt1.toLocalDateTime();
		LocalDateTime localDt5b = LocalDateTime.from(zonedDt1);
		// 文字列→LocalDateTime
		LocalDateTime localDt6a = LocalDateTime.parse("2007-12-03T10:15:30.123");
		LocalDateTime localDt6b = LocalDateTime.parse("2007/12/03 10:15:30.123",
				DateTimeFormatter.ofPattern("yyyy/MM/dd HH:mm:ss.SSS"));

		System.out.println("LocalDateTime");
		System.out.println(nowLocalDt);
		System.out.println(localDt2);
		System.out.println(localDt3);
		System.out.println(localDt4);
		System.out.println(localDt5a);
		System.out.println(localDt5b);
		System.out.println(localDt6a);
		System.out.println(localDt6b);

		// ZonedDateTime
		// 現在日時
		ZonedDateTime nowZonedDt = ZonedDateTime.now(ZoneId.of("Asia/Tokyo"));
		// 指定日時
		ZonedDateTime zonedDt2 = ZonedDateTime.of(2016, 5, 2, 15, 0, 0, 0, ZoneId.of("Asia/Tokyo"));
		// long→ZonedDateTime
		ZonedDateTime zonedDt3 = ZonedDateTime.ofInstant(Instant.ofEpochMilli(msec1), ZoneId.systemDefault());
		// Instant→ZonedDateTime
		ZonedDateTime zonedDt4a = instant1.atZone(ZoneId.systemDefault());
		ZonedDateTime zonedDt4b = ZonedDateTime.ofInstant(instant1, ZoneId.systemDefault());
		// LocalDateTime→ZonedDateTime
		ZonedDateTime zonedDt5a = localDt1.atZone(ZoneId.systemDefault());
		ZonedDateTime zonedDt5b = ZonedDateTime.ofLocal(localDt1, ZoneId.systemDefault(), null);
		// 文字列→ZonedDateTime型
		ZonedDateTime zonedDt6a = ZonedDateTime.parse("2007-12-03T10:15:30+01:00[Europe/Paris]");
		ZonedDateTime zonedDt6b = ZonedDateTime.parse("2016/05/02 10:15:30 Asia/Tokyo",
				DateTimeFormatter.ofPattern("yyyy/MM/dd HH:mm:ss VV"));
		ZonedDateTime zonedDt6c = ZonedDateTime.parse("2016/05/02 10:15:30 JST",
				DateTimeFormatter.ofPattern("yyyy/MM/dd HH:mm:ss zzz"));

		System.out.println("ZonedDateTime");
		System.out.println(nowZonedDt);
		System.out.println(zonedDt2);
		System.out.println(zonedDt3);
		System.out.println(zonedDt4a);
		System.out.println(zonedDt4b);
		System.out.println(zonedDt5a);
		System.out.println(zonedDt5b);
		System.out.println(zonedDt6a);
		System.out.println(zonedDt6b);
		System.out.println(zonedDt6c);

		// 文字列
		// Instant→文字列
		String instantStr1 = DateTimeFormatter.ISO_INSTANT.format(instant1);
		String instantStr2 = DateTimeFormatter.ofPattern("yyyy/MM/dd HH:mm:ss.SSS")
				.format(LocalDateTime.ofInstant(instant1, ZoneId.systemDefault()));
		// LocalDateTime→文字列
		String localStr1 = DateTimeFormatter.ofPattern("yyyy/MM/dd HH:mm:ss.SSS").format(localDt1);
		// ZonedDateTime→文字列
		// "xxxxx VV" - （例）"+09:00 Asia/Tokyo"
		// "xxxx zzz" - （例）"+0900 JST"
		String zonedStr1a = DateTimeFormatter.ofPattern("yyyy/MM/dd HH:mm:ss.SSSxxxxx VV").format(zonedDt1);
		String zonedStr1b = DateTimeFormatter.ofPattern("yyyy/MM/dd HH:mm:ss.SSSxxxx zzz").format(zonedDt1);

		System.out.println("文字列");
		System.out.println(instantStr1);
		System.out.println(instantStr2);
		System.out.println(localStr1);
		System.out.println(zonedStr1a);
		System.out.println(zonedStr1b);
	}

	/**
	 * 日時から一部を取得する。
	 */
	void getDateTimePart() {
		System.out.println("--- " + AppUtil.getMethod() + " ---");

		//
		// 日付/時刻
		//

		// Instant 日時→日付
		Instant ymdInstant1 = instant1.truncatedTo(ChronoUnit.DAYS);

		// LocalDateTime 日時→日付、時刻
		LocalDate localDate1 = localDt1.toLocalDate();
		LocalTime localTime1 = localDt1.toLocalTime();

		// ZonedDateTime 日時→日付、時刻
		ZonedDateTime ymdZoned1 = zonedDt1.truncatedTo(ChronoUnit.DAYS);
		LocalDate localDate2 = zonedDt1.toLocalDate();
		LocalTime localTime2 = zonedDt1.toLocalTime();

		System.out.println(ymdInstant1);
		System.out.println(localDate1);
		System.out.println(localTime1);
		System.out.println(ymdZoned1);
		System.out.println(localDate2);
		System.out.println(localTime2);

		//
		// 曜日
		//

		// Instant→曜日
		String instantWeek1 = DateTimeFormatter.ofPattern("E, EEEE", Locale.JAPANESE)
				.format(LocalDateTime.ofInstant(instant1, ZoneId.systemDefault()));
		String instantWeek2 = LocalDateTime.ofInstant(instant1, ZoneId.systemDefault()).getDayOfWeek()
				.getDisplayName(TextStyle.FULL, Locale.JAPANESE);

		// LocalDateTime→曜日
		// ※"E","EEEE"は、タイムゾーンによって曜日の表記が変わる。
		String localWeek1 = DateTimeFormatter.ofPattern("E, EEEE", Locale.JAPANESE).format(localDt1);
		String localWeek2 = localDt1.getDayOfWeek().getDisplayName(TextStyle.FULL, Locale.JAPANESE);

		// ZonedDateTime→曜日
		// ※"E","EEEE"は、タイムゾーンによって曜日の表記が変わる。
		String zonedWeek1 = DateTimeFormatter.ofPattern("E, EEEE", Locale.JAPANESE).format(zonedDt1);
		String zonedWeek2 = zonedDt1.getDayOfWeek().getDisplayName(TextStyle.FULL, Locale.JAPANESE);

		System.out.println(instantWeek1);
		System.out.println(instantWeek2);
		System.out.println(localWeek1);
		System.out.println(localWeek2);
		System.out.println(zonedWeek1);
		System.out.println(zonedWeek2);
	}

	/**
	 * 日時を計算する。
	 */
	void calcDateTime() {
		System.out.println("--- " + AppUtil.getMethod() + " ---");

		//
		// 計算
		//

		// 加算/減算
		// LocalDateTime 1秒後、1日後、１ヶ月後
		LocalDateTime localNextSec = localDt1.plusSeconds(1);
		LocalDateTime localNextDay = localDt1.plusDays(1);
		LocalDateTime localNextMonth = localDt1.plusMonths(1);
		// ZonedDateTime 1秒後、1日後、１ヶ月後
		ZonedDateTime zonedNextSec = zonedDt1.plusSeconds(1);
		ZonedDateTime zonedNextDay = zonedDt1.plusDays(1);
		ZonedDateTime zonedNextMonth = zonedDt1.plusMonths(1);

		System.out.println("1秒後：" + localNextSec);
		System.out.println("1日後：" + localNextDay);
		System.out.println("1ヶ月後：" + localNextMonth);
		System.out.println("1秒後：" + zonedNextSec);
		System.out.println("1日後：" + zonedNextDay);
		System.out.println("1ヶ月後：" + zonedNextMonth);

		// LocalDateTime 月初、月末
		LocalDateTime localMonthBeginning = localDt1.with(TemporalAdjusters.firstDayOfMonth());
		LocalDateTime localMonthEnd = localDt1.with(TemporalAdjusters.lastDayOfMonth());
		// ZonedDateTime 月初、月末
		ZonedDateTime zonedMonthBeginning = zonedDt1.with(TemporalAdjusters.firstDayOfMonth());
		ZonedDateTime zonedMonthEnd = zonedDt1.with(TemporalAdjusters.lastDayOfMonth());

		System.out.println("月初：" + localMonthBeginning);
		System.out.println("月末：" + localMonthEnd);
		System.out.println("月初：" + zonedMonthBeginning);
		System.out.println("月末：" + zonedMonthEnd);

		// 差
		// LocalDateTime 日時の差 ミリ秒単位、日数単位（切り捨て）
		long localDiffMsec = ChronoUnit.MILLIS.between(localDt1, localNextDay);
		long localDiffDays1 = ChronoUnit.DAYS.between(localDt1, localNextDay);
		long localDiffDays2 = ChronoUnit.DAYS.between(localDt1.toLocalDate(), localNextDay.toLocalDate());
		// LocalDateTime 日時の差 時分秒単位、年月日単位
		Duration localDuration = Duration.between(localDt1, localNextDay);
		Period localPeriod = Period.between(localDt1.toLocalDate(), localNextDay.toLocalDate());
		// ZonedDateTime 日時の差 ミリ秒単位、日数単位（切り捨て）
		long zonedDiffMsec = ChronoUnit.MILLIS.between(zonedDt1, zonedNextDay);
		long zonedDiffDays1 = ChronoUnit.DAYS.between(zonedDt1, zonedNextDay);
		long zonedDiffDays2 = ChronoUnit.DAYS.between(zonedDt1.toLocalDate(), zonedNextDay.toLocalDate());
		// ZonedDateTime 日時の差 時分秒単位、年月日単位
		Duration zonedDuration = Duration.between(zonedDt1, zonedNextDay);
		Period zonedPeriod = Period.between(zonedDt1.toLocalDate(), zonedNextDay.toLocalDate());

		System.out.println(localDiffMsec);
		System.out.println(localDiffDays1);
		System.out.println(localDiffDays2);
		System.out.println(localDuration);
		System.out.println(localPeriod);
		System.out.println(zonedDiffMsec);
		System.out.println(zonedDiffDays1);
		System.out.println(zonedDiffDays2);
		System.out.println(zonedDuration);
		System.out.println(zonedPeriod);

		//
		// 比較
		//

		// LocalDateTime
		boolean localAfterFlag = localNextDay.isAfter(localDt1);
		// ZonedDateTime
		boolean zonedAfterFlag = zonedNextDay.isAfter(zonedDt1);

		System.out.println("比較：" + localAfterFlag);
		System.out.println("比較：" + zonedAfterFlag);
	}

	/**
	 * Java5のTimeUnitを使ってみる。
	 */
	void useTimeUnit() {
		System.out.println("--- " + AppUtil.getMethod() + " ---");
		DateFormat format = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss.SSS");

		long nextSec = msec1 + TimeUnit.SECONDS.toMillis(1);
		long nextDay = msec1 + TimeUnit.DAYS.toMillis(1);

		System.out.println("1秒後：" + format.format(new Date(nextSec)));
		System.out.println("1日後：" + format.format(new Date(nextDay)));

		try {
			System.out.println("スリープ前：" + format.format(new Date()));
			TimeUnit.SECONDS.sleep(1);
			System.out.println("スリープ後：" + format.format(new Date()));
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
	}
}
