package com.tryjava.net;

import java.io.IOException;

import org.apache.http.HttpEntity;
import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.StatusLine;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;

import com.tryjava.AppConfig;
import com.tryjava.util.AppUtil;

/**
 * HTTPクライアントを作成してみる。（Apache HttpComponents HttpClient 4.3以降）
 *
 * プロキシを使ってみる。Basic認証あり。
 *
 * 参考<br>
 * JavaでのHttpClientサンプル - CLOVER
 * http://d.hatena.ne.jp/Kazuhira/20131026/1382796711<br>
 * Apache HttpComponents - HttpClient Quick Start
 * http://hc.apache.org/httpcomponents-client-4.5.x/quickstart.html<br>
 * Javaで認証が必要なProxyサーバー経由でHTTP接続する - Qiita
 * http://qiita.com/tag1216/items/24a78d0b4bf2ca641d97<br>
 *
 * @since 2016/5/26
 */
public class ApacheHttpClient02Proxy {
	private static final String PROXY_HOST = AppConfig.getString("proxy.host");
	private static final int PROXY_PORT = AppConfig.getInt("proxy.port");
	private static final String PROXY_USER = AppConfig.getString("proxy.user");
	private static final String PROXY_PASSWORD = AppConfig.getString("proxy.password");

	public void run() {
		doGet();
	}

	void doGet() {
		System.out.println("--- " + AppUtil.getMethod() + " ---");

		// プロキシ設定
		HttpHost proxy = new HttpHost(PROXY_HOST, PROXY_PORT);
		CredentialsProvider provider = new BasicCredentialsProvider();
		provider.setCredentials( //
				new AuthScope(proxy), //
				new UsernamePasswordCredentials(PROXY_USER, PROXY_PASSWORD));
		RequestConfig config = RequestConfig.custom().setProxy(proxy).build();
		HttpClientBuilder builder = HttpClients.custom() //
				.setDefaultCredentialsProvider(provider) //
				.setDefaultRequestConfig(config);

		CloseableHttpClient client = null;
		try {
			client = builder.build();

			// リクエスト作成
			HttpGet request = new HttpGet("http://www.google.co.jp/");

			// 接続
			CloseableHttpResponse response = null;
			try {
				response = client.execute(request);

				// レスポンスコード取得
				StatusLine statusLine = response.getStatusLine();
				System.out.println(statusLine);
				if (statusLine.getStatusCode() != HttpStatus.SC_OK) {
					throw new RuntimeException("エラーレスポンスを受信しました。statusLine=" + statusLine);
				}

				// レスポンスボディ取得
				HttpEntity entity = response.getEntity();
				String content = EntityUtils.toString(entity);
				System.out.println(content);
			} finally {
				if (response != null) {
					response.close();
				}
			}

		} catch (Exception e) {
			throw new RuntimeException("HTTP通信に失敗しました。", e);
		} finally {
			if (client != null) {
				try {
					client.close();
				} catch (IOException e) {
					throw new RuntimeException(e);
				}
			}
		}
	}
}
