package com.tryjava.io.file;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.List;
import java.util.Scanner;
import java.util.stream.Stream;

import org.apache.commons.lang3.StringUtils;

/**
 * テキストファイル読み書きの練習
 *
 * 参考
 * <ul>
 * <li>Javaで1行ずつテキストデータを読み込むイディオムの変遷 - argius note<br>
 * http://argius.hatenablog.jp/entry/2015/06/06/234315
 * </ul>
 *
 * @sine 2016/5/2
 */
public class TextFile01 {
	static final String path = "./work/file01.txt";

	public static void main(String[] args) {
		TextFile01 app = new TextFile01();
		app.run();
	}

	void run() {
		writeLines();
		readLines();
		readTokens();
		writeAll();
		readAll();
	}

	/**
	 * Java7～8でのテキストファイル書き込み（順次）
	 *
	 * Java7から、try-with-resources文が使えるようになった。<br>
	 * それ以外は、Java6までと同様。<br>
	 */
	void writeLines() {
		try (PrintWriter pw = new PrintWriter(
				Files.newBufferedWriter(Paths.get(path), Charset.forName("UTF-8"), StandardOpenOption.CREATE))) {
			pw.println("Hello World");
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Java7～8でのテキストファイル読み込み（順次）
	 *
	 * Java7から、try-with-resources文が使えるようになった。<br>
	 * それ以外は、Java6までと同様。<br>
	 *
	 * Java8から、ストリームAPIも使えるようになった。<br>
	 */
	void readLines() {
		try (BufferedReader br = Files.newBufferedReader(Paths.get(path), Charset.forName("UTF-8"))) {
			String line;
			while ((line = br.readLine()) != null) {
				System.out.println(line);
			}
		} catch (IOException e) {
			e.printStackTrace();
		}

		// ストリームAPI、ラムダ式を使用する。
		try (Stream<String> stream = Files.lines(Paths.get(path), Charset.forName("UTF-8"))) {
			stream.forEach(System.out::println);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Java5～8でのテキストファイル読み込み（トークンごと）
	 *
	 * Java5では、Scannerが使えるようになった。<br>
	 * 空白や改行などでトークン分割された文字列を順次読み込む。
	 */
	void readTokens() {
		Scanner scanner = null;
		try {
			scanner = new Scanner(new File(path), "UTF-8");
			while (scanner.hasNext()) {
				String line = scanner.next();
				System.out.println(line);
			}
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} finally {
			if (scanner != null) {
				scanner.close();
			}
		}
	}

	/**
	 * Java7～8でのテキストファイル書き込み（一括）
	 *
	 * Java7から、Files,Pathsが使えるようになった。
	 */
	void writeAll() {
		List<String> lines = new ArrayList<String>();
		lines.add("Hello World");
		try {
			Files.write(Paths.get(path), lines, Charset.forName("UTF-8"), StandardOpenOption.CREATE);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Java7～8でのテキストファイル読み込み（一括）
	 *
	 * Java7から、Files,Pathsが使えるようになった。
	 */
	void readAll() {
		try {
			List<String> lines = Files.readAllLines(Paths.get(path), Charset.forName("UTF-8"));
			System.out.println(StringUtils.join(lines, System.lineSeparator()));
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
}
