package example.tryspring.c4_db;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.BeanPropertyRowMapper;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.namedparam.BeanPropertySqlParameterSource;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.stereotype.Repository;

/**
 * DAO実装クラスです。
 * <p>
 * Spring JDBCで、データベースにアクセスします。
 * JdbcTemplateとNamedParameterJdbcTemplateのインスタンス変数は、Spring DIでインジェクションされます。
 * </p>
 */
@Repository
public class ProductDaoImpl implements ProductDao {
	@Autowired
	private JdbcTemplate jdbcTemplate;

	@Autowired
	private NamedParameterJdbcTemplate npJdbcTemplate;

	/**
	 * レコード件数を取得します。
	 *
	 * @return 件数
	 */
	@Override
	public int count() {
		return count_1();
	}

	/**
	 * レコード件数を取得します。（実装方法１）
	 *
	 * @return 件数
	 */
	protected int count_1() {
		return this.jdbcTemplate
				.queryForInt("select count(*) from APP.PRODUCT");
	}

	/**
	 * 1レコードを検索します。
	 *
	 * @param name
	 *            検索する名前
	 * @return Productインスタンス
	 */
	@Override
	public Product find(String name) {
		// return find_1(name);
		// return find_2(name);
		// return find_3(name);
		return find_4(name);
	}

	/**
	 * 1レコードを検索します。（実装方法１）
	 * <p>
	 * JDBC操作：JdbcTemplateクラス<br/>
	 * SQLパラメータ指定：引数渡し<br/>
	 * レコード/インスタンス変換：RowMapperクラス<br/>
	 * </p>
	 *
	 * @param name
	 *            検索する名前
	 * @return Productインスタンス
	 */
	protected Product find_1(String name) {
		return this.jdbcTemplate.queryForObject(
				"select * from APP.PRODUCT where NAME=?",
				new String[] { name }, new RowMapper<Product>() {
					@Override
					public Product mapRow(ResultSet rs, int num)
							throws SQLException {
						Product product = new Product();
						product.setName(rs.getString("NAME"));
						product.setPrice(rs.getInt("PRICE"));
						return product;
					}
				});
	}

	/**
	 * 1レコードを検索します。（実装方法２）
	 * <p>
	 * JDBC操作：JdbcTemplateクラス<br/>
	 * SQLパラメータ指定：引数渡し<br/>
	 * レコード/インスタンス変換：BeanPropertyRowMapperクラス<br/>
	 * </p>
	 *
	 * @param name
	 *            検索する名前
	 * @return Productインスタンス
	 */
	protected Product find_2(String name) {
		return this.jdbcTemplate.queryForObject(
				"select * from APP.PRODUCT where NAME=?",
				new Object[] { name }, new BeanPropertyRowMapper<Product>(
						Product.class));
	}

	/**
	 * 1レコードを検索します。（実装方法３）
	 * <p>
	 * JDBC操作：NamedParameterJdbcTemplateクラス<br/>
	 * SQLパラメータ指定：MapSqlParameterSourceクラス<br/>
	 * レコード/インスタンス変換：BeanPropertyRowMapperクラス<br/>
	 * </p>
	 *
	 * @param name
	 *            検索する名前
	 * @return Productインスタンス
	 */
	protected Product find_3(String name) {
		return this.npJdbcTemplate.queryForObject(
				"select * from APP.PRODUCT where NAME=:name",
				new MapSqlParameterSource().addValue("name", name),
				new BeanPropertyRowMapper<Product>(Product.class));
	}

	/**
	 * 1レコードを検索します。（実装方法４）
	 * <p>
	 * JDBC操作：NamedParameterJdbcTemplateクラス<br/>
	 * SQLパラメータ指定：BeanPropertySqlParameterSourceクラス<br/>
	 * レコード/インスタンス変換：BeanPropertyRowMapperクラス<br/>
	 * </p>
	 *
	 * @param name
	 *            検索する名前
	 * @return Productインスタンス
	 */
	protected Product find_4(String name) {
		Product product = new Product();
		product.setName(name);
		return this.npJdbcTemplate.queryForObject(
				"select * from APP.PRODUCT where NAME=:name",
				new BeanPropertySqlParameterSource(product),
				new BeanPropertyRowMapper<Product>(Product.class));
	}

	/**
	 * 全レコードを検索します。
	 *
	 * @return Productインスタンスのリスト
	 */
	@Override
	public List<Product> findAll() {
		// return findAll_1();
		return findAll_2();
	}

	/**
	 * 全レコードを検索します。（実装方法１）
	 * <p>
	 * JDBC操作：JdbcTemplateクラス<br/>
	 * レコード/インスタンス変換：RowMapperクラス<br/>
	 * </p>
	 *
	 * @return Productインスタンスのリスト
	 */
	protected List<Product> findAll_1() {
		return this.jdbcTemplate.query(
				"select * from APP.PRODUCT order by NAME",
				new RowMapper<Product>() {
					@Override
					public Product mapRow(ResultSet rs, int num)
							throws SQLException {
						Product product = new Product();
						product.setName(rs.getString("NAME"));
						product.setPrice(rs.getInt("PRICE"));
						return product;
					}
				});
	}

	/**
	 * 全レコードを検索します。（実装方法２）
	 * <p>
	 * JDBC操作：JdbcTemplateクラス<br/>
	 * レコード/インスタンス変換：BeanPropertyRowMapperクラス<br/>
	 * </p>
	 *
	 * @return Productインスタンスのリスト
	 */
	protected List<Product> findAll_2() {
		return this.jdbcTemplate.query(
				"select * from APP.PRODUCT order by NAME",
				new BeanPropertyRowMapper<Product>(Product.class));
	}

	/**
	 * レコードを登録します。
	 *
	 * @param product
	 *            登録対象
	 * @return 登録結果
	 */
	@Override
	public int insert(Product product) {
		// return insert_1(product);
		return insert_2(product);
	}

	/**
	 * レコードを登録します。（実装方法１）
	 * <p>
	 * JDBC操作：JdbcTemplateクラス<br/>
	 * SQLパラメータ指定：引数渡し<br/>
	 * </p>
	 *
	 * @param product
	 *            登録対象
	 * @return 登録結果
	 */
	protected int insert_1(Product product) {
		return this.jdbcTemplate.update(
				"insert into APP.PRODUCT (NAME, PRICE) values (?, ?)",
				product.getName(), product.getPrice());
	}

	/**
	 * レコードを登録します。（実装方法２）
	 * <p>
	 * JDBC操作：NamedParameterJdbcTemplateクラス<br/>
	 * SQLパラメータ指定：BeanPropertySqlParameterSourceクラス<br/>
	 * </p>
	 *
	 * @param product
	 *            登録対象
	 * @return 登録結果
	 */
	protected int insert_2(Product product) {
		return this.npJdbcTemplate.update(
				"insert into APP.PRODUCT (NAME, PRICE) values (:name, :price)",
				new BeanPropertySqlParameterSource(product));
	}

	/**
	 * レコードを削除します。
	 *
	 * @param product
	 *            削除対象
	 * @return 削除結果
	 */
	@Override
	public int delete(Product product) {
		// return delete_1(product);
		return delete_2(product);
	}

	/**
	 * レコードを削除します。（実装方法１）
	 * <p>
	 * JDBC操作：JdbcTemplateクラス<br/>
	 * SQLパラメータ指定：引数渡し<br/>
	 * </p>
	 *
	 * @param product
	 *            削除対象
	 * @return 削除結果
	 */
	protected int delete_1(Product product) {
		return this.jdbcTemplate.update("delete from APP.PRODUCT where NAME=?",
				product.getName());
	}

	/**
	 * レコードを削除します。（実装方法１）
	 * <p>
	 * JDBC操作：JdbcTemplateクラス<br/>
	 * SQLパラメータ指定：BeanPropertySqlParameterSourceクラス<br/>
	 * </p>
	 *
	 * @param product
	 *            削除対象
	 * @return 削除結果
	 */
	protected int delete_2(Product product) {
		return this.npJdbcTemplate.update(
				"delete from APP.PRODUCT where NAME=:name",
				new BeanPropertySqlParameterSource(product));
	}

}
