﻿using System;
using System.Text;
using System.IO;
using System.Linq;


using Vintagestory.API.Common;
using Vintagestory.API.Config;
using Vintagestory.API.Server;

using Vintagestory.Server;

namespace AdminToolkit
{
	public class BackupCycleCommand : AdminModCommand
	{
		private long backupTickerID;
		private const string _timeFormat = @"d\D\ hh\H\ mm\M\ ss\S";


		private DateTimeOffset _lastBackupTime = DateTimeOffset.MinValue;

		private string fileFilter;

		public BackupCycleCommand(ICoreServerAPI _serverAPI) : base(_serverAPI)
		{
			this.Command = "backups";
			this.Description = "Control server automatic backup cycle.";
			this.handler += BackupCommandParser; 
			this.RequiredPrivilege = Privilege.controlserver;//"Backup" ?
			this.Syntax = "enable / disable /  delay [hours] / stats";



			ServerAPI.Event.ServerRunPhase(EnumServerRunPhase.RunGame, ServerStarts);
		}

		public bool Enabled {

			get { return this.backupTickerID != 0; }
		}

		private void BackupCommandParser(IServerPlayer player, int groupId, CmdArgs args)
		{
			//Enable/Disable, time rate, count of files & status

			StringBuilder msgLine = new StringBuilder( );
			bool success = false;
			int? delay, count;

			if (args.Length > 0) {
				string command = args.PopWord( );

				if (command.Length > 0) {
					command = command.ToLowerInvariant( );
				}

				switch (command) {
				case "enable":
					success = ToggleBackupCycle(true); 
					if (success) msgLine.AppendFormat("Enabled, with delay: {0}, next: {1}\n", this.CachedConfiguration.BackupDelay.ToString(_timeFormat), DateTimeOffset.UtcNow.Add(this.CachedConfiguration.BackupDelay).ToString("u"));
					break;

				case "disable":
					success = ToggleBackupCycle(false); 
					if (success) msgLine.AppendLine("Disabled.");
					break;

				case "delay":
					delay = args.PopInt( );

					if (delay.HasValue && delay.Value >= 1) {
						AlterTimeDelay(delay.Value);
						msgLine.AppendFormat("Changed delay to: {0} ", this.CachedConfiguration.BackupDelay.ToString(_timeFormat));
						success = true;
					} else {
						msgLine.AppendLine("Wrong time delay value; must be positive integer 1 or more (hours) ");
					}

					break;

				case "count":
					count = args.PopInt( );

					if (count.HasValue && count.Value >= 4) {
						this.CachedConfiguration.BackupCount = (uint)count.Value;
						msgLine.AppendFormat("Changed count to: {0} ", this.CachedConfiguration.BackupCount);
						success = true;
					} else {
						msgLine.AppendLine("Wrong quantity value; must be positive integer 4 or more (count) ");
					}

					break;

				case "state":
				case "status":
				case "stats":
					StatusLine(msgLine);
					success = true;
					break;

				default:
					msgLine.AppendLine("Unknown parameter");
					break;
				}



			} else {
				msgLine.AppendLine("supply more command arguments; enable / disable / stats / delay ");
			}




			player.SendMessage(groupId, msgLine.ToString( ), success ? EnumChatType.CommandSuccess : EnumChatType.CommandError);
		}

		/// <summary>
		/// Toggles the backup cycle.
		/// </summary>
		/// <returns>The backup cycle.</returns>
		/// <param name="enable">Enable.</param>
		private bool ToggleBackupCycle(bool enable)
		{
			if (enable) 
			{
				if (!this.Enabled) 
				{
					this.backupTickerID = ServerAPI.Event.RegisterCallback(BackupCycleHandler, (int)this.CachedConfiguration.BackupDelay.TotalMilliseconds);
					Logger.Notification("BackupCycle enabled, delay: {0}", this.CachedConfiguration.BackupDelay.ToString(_timeFormat));
					return true;
				}
			} 
			else 
			{				
				if (this.Enabled) 
				{
					ServerAPI.Event.UnregisterCallback(backupTickerID);
					this.backupTickerID = 0;
					Logger.Notification("BackupCycle disabled");
					return true;
				}
			}

			return false;
		}


		private void BackupCycleHandler(float elapsed)
		{			
			Logger.Event("Backup cycle triggered, elasped {0} @{1} Uptime", elapsed, ServerAPI.Server.ServerUptimeSeconds);
			//Peek at backup DIR, count files of the backup type
			var backupDir = new DirectoryInfo(GamePaths.Backups);

			if (backupDir.Exists) {
				
				var files = backupDir.GetFiles(fileFilter);

				if (files.Length > this.CachedConfiguration.BackupCount) {
					Logger.VerboseDebug("There are {0} backup files, over set limit", files.Length);
					var oldest = files.OrderBy(fi => fi.CreationTimeUtc).FirstOrDefault( );

					if (oldest != null) {
						Logger.VerboseDebug("Oldest backup file: {0}", oldest.FullName);
						oldest.Delete( );
						Logger.Notification("Deleted old Backup: {0}", oldest.Name);
					}
				}
			} else {
				Logger.VerboseDebug("Could not open backup directory");
			}


			ServerAPI.InjectConsole(@"/genbackup");
			_lastBackupTime = DateTimeOffset.UtcNow;

			this.backupTickerID = ServerAPI.Event.RegisterCallback(BackupCycleHandler, ( int )this.CachedConfiguration.BackupDelay.TotalMilliseconds);
			Logger.Notification("BackupCycle re-engaged, delay: {0} next time: {1}", this.CachedConfiguration.BackupDelay.ToString(_timeFormat), DateTimeOffset.UtcNow.Add(this.CachedConfiguration.BackupDelay).ToString("u"));

			//GlobalConstants.ConsoleGroup
			/*
			backupFileName = Path.GetFileName(server.Config.WorldConfig.SaveFileLocation).Replace(GlobalConstants.WorldSaveExtension, "") + "-" + string.Format("{0:yyyy-MM-dd_HH-mm-ss}", DateTime.Now) + GlobalConstants.WorldSaveExtension;
			 */

		}

		private void AlterTimeDelay(int delay)
		{
			delay = Math.Max(delay, 1);
			delay = Math.Min(delay, 500);//Limited by callback set in milliseconds

			this.CachedConfiguration.BackupDelay = new TimeSpan(delay, 0, 0);

			if (this.Enabled) 
			{
				ToggleBackupCycle(false);
				ToggleBackupCycle(true);
			}
		}

		private void StatusLine(StringBuilder msgLine)
		{
			msgLine.AppendFormat("Delay: {0}, Max Backups:{1}, Auto-start:{2}, ", CachedConfiguration.BackupDelay.ToString(_timeFormat),CachedConfiguration.BackupCount,CachedConfiguration.Autobackup );

			if (this.Enabled) {
				msgLine.AppendFormat("<font color='lime'>Currently enabled: YES</font>\n");
			} else {
				msgLine.AppendFormat("<font color='red'>Currently enabled: NO</font>\n");
			}

			if (_lastBackupTime != DateTimeOffset.MinValue) 
			{
				msgLine.AppendFormat("Last Backup attempt: {0} ",  _lastBackupTime.ToString("u") );
				msgLine.AppendFormat("Next backup due: {0}\n",_lastBackupTime.Add(CachedConfiguration.BackupDelay).ToString("u") );
			}

			var backupDir = new DirectoryInfo(GamePaths.Backups);
			long totalBackupSize = 0;

			if (backupDir.Exists) 
			{
				var backups = backupDir.GetFiles(fileFilter);

				foreach (FileInfo backFile in backups) 
				{
					msgLine.AppendFormat("<font color='white'> '{0}' Size: {1} Written: {2} </font>\n", backFile.Name, FileSizeReadable(backFile.Length), backFile.LastWriteTimeUtc.ToString("u"));
					totalBackupSize += backFile.Length;
				}

				msgLine.AppendFormat("Total size: {0} in {1} files.\n", FileSizeReadable(totalBackupSize),backups.Length);

			}
		}

		public static string FileSizeReadable(long i)
		{
			// Get absolute value
			long absolute_i = (i < 0 ? -i : i);
			// Determine the suffix and readable value
			string suffix;
			double readable;
			if (absolute_i >= 0x1000000000000000) // Exabyte
			{
				suffix = "EB";
				readable = (i >> 50);
			} else if (absolute_i >= 0x4000000000000) // Petabyte
			  {
				suffix = "PB";
				readable = (i >> 40);
			} else if (absolute_i >= 0x10000000000) // Terabyte
			  {
				suffix = "TB";
				readable = (i >> 30);
			} else if (absolute_i >= 0x40000000) // Gigabyte
			  {
				suffix = "GB";
				readable = (i >> 20);
			} else if (absolute_i >= 0x100000) // Megabyte
			  {
				suffix = "MB";
				readable = (i >> 10);
			} else if (absolute_i >= 0x400) // Kilobyte
			  {
				suffix = "KB";
				readable = i;
			} else {
				return i.ToString("0 B"); // Byte
			}
			// Divide by 1024 to get fractional value
			readable = (readable / 1024);
			// Return formatted number with suffix
			return readable.ToString("0.### ") + suffix;
		}

		private void ServerStarts( )
		{
			//ServerAPI.WorldManager.CurrentWorldName
			//ServerAPI.WorldManager.SaveGame.WorldName
			string worldNameTrimmed = Path.GetFileNameWithoutExtension(ServerAPI.WorldManager.CurrentWorldName);

			fileFilter = $"{worldNameTrimmed}-*{GlobalConstants.WorldSaveExtension}";
			Logger.VerboseDebug("File Filter: {0}", fileFilter);

			if (CachedConfiguration.Autobackup) { ToggleBackupCycle(true); }
		}
	}


}

