﻿using System;
using System.Collections.Generic;
using System.Linq;

using Vintagestory.API.Common;

namespace ArmourMod
{
	public class EntityArmourPlayer : EntityPlayer
	{
		protected ICoreAPI CoreAPI;
		protected ILogger Logger;

		//Damage Filter list: BY Inventory SLOT#
		private Dictionary<int,DamageFilter> DamageFilters;
		//Pre-cook values 
		private float bluntPercentReduction = 0F;
		private float piercingPercentReduction = 0F;
		private float slashingPercentReduction = 0F;
		private float crushingPercentReduction = 0F;
		private float firePercentReduction = 0F;

		private long client_callback;

		public Dictionary<EnumCharacterDressType,Item> WornClothesItems { get; private set;}//Needs to be client side

		public override void Initialize(Vintagestory.API.Common.Entities.EntityProperties properties, ICoreAPI api, long chunkindex3d)
		{
			base.Initialize(properties, api, chunkindex3d);

			CoreAPI = api;
			Logger = api.World.Logger;
		}

		public override void OnEntitySpawn()
		{
			base.OnEntitySpawn();

			Logger.VerboseDebug( "OnEntitySpawn - EntityArmourPlayer" );

			if ( CoreAPI.Side == EnumAppSide.Server ) {
				//Attach to Inventory events
				#if DEBUG
				Logger.VerboseDebug( "Side:{0}, gearInv == null?{1}", CoreAPI.Side.ToString( ), base.GearInventory == null );
				#endif

				base.GearInventory.SlotModified += ServerWatchSlotModified;
				//this.LeftHandItemSlot.//Check also for shields

				//base.GearInventory.SlotNotified += WatchSlotNotified;

				//Walk Inventory FIRST TIME ONLY
				PopulateDamageFilters();
			}

			if ( CoreAPI.Side == EnumAppSide.Client ) {
				this.WornClothesItems = new Dictionary<EnumCharacterDressType, Item>();

				//Attach to Inventory events
				#if DEBUG
				Logger.VerboseDebug( "Side:{0}, gearInv == null?{1}", CoreAPI.Side.ToString( ), base.GearInventory == null );
				#endif

				//base.GearInventory.SlotModified += ClientWatchSlotModified;

				client_callback = CoreAPI.Event.RegisterCallback( ClientDelayHook, 300 );
			}
		}

		public override void Die(EnumDespawnReason reason = EnumDespawnReason.Death, DamageSource damageSourceForDeath = null)
		{
			base.Die(reason, damageSourceForDeath);
			//Clear Armour filters?
			base.Api.World.Logger.VerboseDebug( "Player Died" );
		}

		public override void Revive()
		{
			base.Revive();
			base.Api.World.Logger.VerboseDebug( "Respawned, alive again..." );
		}

		private void ServerWatchSlotModified (int slotId)
		{
			var watchedSlot = base.GearInventory[slotId];

			Logger.VerboseDebug( "WatchSlotModified:{0}", slotId );

			if ( !watchedSlot.Empty ) {
				
				if ( watchedSlot.StorageType == EnumItemStorageFlags.Outfit && watchedSlot.Itemstack.Class == EnumItemClass.Item ) {

					Logger.VerboseDebug( "Equiped a clothing item:{0}", watchedSlot.Itemstack.Item.Code.ToString() );
					//replace / add damage filter when applicable armor item in slot (which ever that is?)

					if (CheckIfClothesIsArmour(watchedSlot)) 
					{
					DamageFilter dmgFilt = ConstructFilter( watchedSlot );

					Logger.VerboseDebug( "Armour Filter Props: B{0} P{1} S{2} C{3} F{4}", dmgFilt.bluntPercent, dmgFilt.piercingPercent, dmgFilt.slashingPercent, dmgFilt.crushingPercent, dmgFilt.firePercent );

					DamageFilters[slotId] = dmgFilt;
					RecomputeDamageFilterPercents( );											
					} 
				}
			} else if (watchedSlot.StorageType == EnumItemStorageFlags.Outfit){
				//Clear out filters for what was in slot# X here - if applicable
				Logger.VerboseDebug( "Removed a clothing item from SlotId:{0}",slotId );
				DamageFilters.Remove( slotId );
				RecomputeDamageFilterPercents( );
			}
		}

		private void ClientWatchSlotModified (int slotId)
		{
			Logger.VerboseDebug( "(Client) WatchSlotModified:{0}", slotId );
			var watchedSlot = base.GearInventory[slotId];

			if ( !watchedSlot.Empty ) {

				if ( watchedSlot.StorageType == EnumItemStorageFlags.Outfit && watchedSlot.Itemstack.Class == EnumItemClass.Item ) {					
					if (CheckIfClothesIsArmour(watchedSlot)) 
					{						
						WearArmorModel( slotId, watchedSlot.Itemstack.Item );
					} 
				}
			} else if (watchedSlot.StorageType == EnumItemStorageFlags.Outfit){
				
				UnwearArmorModel( slotId );
			}
		}

		/*
		private void WatchSlotNotified(int slotId)
		{
			var watchedSlot = base.GearInventory[slotId];

			Logger.VerboseDebug( "WatchSlotNotified:{0}", slotId );
		}
		*/

		protected void ClientDelayHook(float delayed )
		{
			#if DEBUG
			Logger.VerboseDebug( "Checking {0} later, gearInv == null?{1}; HOOK EVENTS", delayed,  base.GearInventory == null );
			#endif
			CoreAPI.Event.UnregisterCallback( client_callback );
			base.GearInventory.SlotModified += ClientWatchSlotModified;
			PopulateWornArmourModels( );
		}

		private void PopulateDamageFilters()
		{
			DamageFilters = new Dictionary<int, DamageFilter>( 10 );

			if ( this.GearInventory.Count > 0 ) {
				foreach(var itemSlot in this.GearInventory ) {

					int slotId = this.GearInventory.GetSlotId( itemSlot );
					if ( !itemSlot.Empty ) {

						if ( itemSlot.StorageType == EnumItemStorageFlags.Outfit && itemSlot.Itemstack.Class == EnumItemClass.Item ) {

							if ( CheckIfClothesIsArmour(itemSlot) ) 
							{
							CoreAPI.World.Logger.VerboseDebug( "DF: Consider clothing item:{0}", itemSlot.Itemstack.Item.Code.ToString( ) );

							DamageFilter dmgFilt = ConstructFilter( itemSlot);
							CoreAPI.World.Logger.VerboseDebug( "Armour Filter Props: B{0} P{1} S{2} C{3} F{4}", dmgFilt.bluntPercent, dmgFilt.piercingPercent, dmgFilt.slashingPercent, dmgFilt.crushingPercent, dmgFilt.firePercent );

							DamageFilters.Add(slotId,dmgFilt);							
							} 
						}
					}	
				}				
			}

			if ( DamageFilters.Count > 0 ) {
				RecomputeDamageFilterPercents( );
			}
		}

		private void RecomputeDamageFilterPercents()
		{
			if (DamageFilters != null && DamageFilters.Count > 0)
			{				
			bluntPercentReduction = DamageFilters.Sum( df => df.Value.bluntPercent );
			bluntPercentReduction = Math.Abs( bluntPercentReduction );
			bluntPercentReduction = Math.Min( 0.95F, bluntPercentReduction );
			
			piercingPercentReduction = DamageFilters.Sum( df => df.Value.piercingPercent );
			piercingPercentReduction = Math.Abs( piercingPercentReduction );
			piercingPercentReduction = Math.Min( 0.95F, piercingPercentReduction );
						
			slashingPercentReduction = DamageFilters.Sum( df => df.Value.slashingPercent );
			slashingPercentReduction = Math.Abs( slashingPercentReduction );
			slashingPercentReduction = Math.Min( 0.95F, slashingPercentReduction );
							
			crushingPercentReduction = DamageFilters.Sum( df => df.Value.crushingPercent );
			crushingPercentReduction = Math.Abs( crushingPercentReduction );
			crushingPercentReduction = Math.Min( 0.95F, crushingPercentReduction );
			
			firePercentReduction = DamageFilters.Sum( df => df.Value.firePercent );
			firePercentReduction = Math.Abs( firePercentReduction );
			firePercentReduction = Math.Min( 0.95F, firePercentReduction );			
			}
		}

		private void WearArmorModel(int slotId, Item wornArmour)
		{		
			if ( !WornClothesItems.ContainsKey( (EnumCharacterDressType)slotId ) ) {
				Logger.VerboseDebug( "Wearing: {0}", wornArmour.Code.ToString( ) );
				WornClothesItems.Add( (EnumCharacterDressType)slotId, wornArmour );
			} else {
				Logger.VerboseDebug( "Duplicate of SlotId {0} worn Armour: {1}",slotId ,wornArmour.Code.ToString( ) );
			}
		}

		private void UnwearArmorModel(int slotId)
		{
			if ( WornClothesItems.ContainsKey( (EnumCharacterDressType)slotId ) ) {
				var armourItem = WornClothesItems[(EnumCharacterDressType)slotId];
				Logger.VerboseDebug( "UN-Wearing: {0}", armourItem.Code.ToString( ) );
				WornClothesItems.Remove( (EnumCharacterDressType)slotId );
			} else {
				Logger.VerboseDebug( "Non-armour removed, slotID{0}", slotId );
			}
		}

		private void PopulateWornArmourModels()
		{			
			if ( this.GearInventory != null && this.GearInventory.Count > 0 ) {
				foreach(var itemSlot in this.GearInventory ) {

					int slotId = this.GearInventory.GetSlotId( itemSlot );
					if ( !itemSlot.Empty ) {

						if ( itemSlot.StorageType == EnumItemStorageFlags.Outfit && itemSlot.Itemstack.Class == EnumItemClass.Item ) {

							if ( CheckIfClothesIsArmour(itemSlot) ) 
							{
								Logger.VerboseDebug( "Already wearing armour:{0}", itemSlot.Itemstack.Item.Code.ToString( ) );
								WornClothesItems.Add( (EnumCharacterDressType)slotId, itemSlot.Itemstack.Item );
							} 
						}
					}	
				}				
			}
		}

		/// <summary>
		/// Where the Protection happens
		/// </summary>
		/// <returns><c>true</c>, if damage was received, <c>false</c> otherwise.</returns>
		/// <param name="damageSource">Damage source.</param>
		/// <param name="damage">Damage.</param>
		public override bool ReceiveDamage(DamageSource damageSource, float damage)
		{
			float beforeDamage = damage;
			//TODO: evaluate postion vector of strike to determine IF armour offers any protection

			if ( DamageFilters.Count > 0 ) {				
				//Evaluate damge type
				//Reduce by any applicable filter(s)
				//Pass along

				#if DEBUG
				Logger.VerboseDebug("{0} has Damage Filters#{1}", this.PlayerUID,DamageFilters.Count);
				#endif

				switch(damageSource.Type) {

				case EnumDamageType.BluntAttack:					
					damage -= damage * bluntPercentReduction;
					break;

				case EnumDamageType.PiercingAttack:					
					damage -= damage * piercingPercentReduction;
					break;

				case EnumDamageType.SlashingAttack:					
					damage -= damage * slashingPercentReduction;
					break;
			
				case EnumDamageType.Crushing:					
					damage -= damage * crushingPercentReduction;
					break;
			
				case EnumDamageType.Fire:					
					damage -= damage * firePercentReduction;
					break;							

				default:

					break;
				}				
			}

			#if DEBUG
			Logger.VerboseDebug( "Reduced: {0} Dmg from {2}; [{1}]", damage, damageSource.Type.ToString(),beforeDamage );
			#endif

			return base.ReceiveDamage(damageSource, damage);
		}

		public static bool CheckIfClothesIsArmour(ItemSlot possibleArmour)
		{
			if (possibleArmour.Itemstack != null && possibleArmour.Itemstack.Item.Code.BeginsWith( "armourmod", "clothes" ) ) {
				
				if ( possibleArmour.Itemstack.ItemAttributes.Exists
				     && ( 
				         possibleArmour.Itemstack.ItemAttributes.KeyExists( "BluntProtection" ) ||
				         possibleArmour.Itemstack.ItemAttributes.KeyExists( "PiercingProtection" ) ||
				         possibleArmour.Itemstack.ItemAttributes.KeyExists( "SlashingProtection" ) ||
				         possibleArmour.Itemstack.ItemAttributes.KeyExists( "CrushingProtection" ) ||
				         possibleArmour.Itemstack.ItemAttributes.KeyExists( "FireProtection" ) 
				     ) ) {
					return true;
				}
			}

			return false;
		}

		public static DamageFilter ConstructFilter(ItemSlot armourInSlot)
		{
			var blunt = armourInSlot.Itemstack.ItemAttributes["BluntProtection"].AsFloat( );
			var piercing = armourInSlot.Itemstack.ItemAttributes["PiercingProtection"].AsFloat( );
			var slashing = armourInSlot.Itemstack.ItemAttributes["SlashingProtection"].AsFloat( );
			var crushing = armourInSlot.Itemstack.ItemAttributes["CrushingProtection"].AsFloat( );
			var flame = armourInSlot.Itemstack.ItemAttributes["FireProtection"].AsFloat( );

			return new DamageFilter( blunt, piercing, slashing, crushing, flame );
		}

	}
}

/*
 	public enum EnumCharacterDressType
	{
		Head,
		Shoulder,
		UpperBody,
		UpperBodyOver = 11,
		LowerBody = 3,
		Foot,
		Neck = 6,
		Emblem,
		Ring,
		Arm = 10,
		Hand = 5,
		Waist = 9
	}

BluntProtection:0.1,
PiercingProtection:0.1,
SlashingProtection:0.1,
CrushingProtection:0.1,
FireProtection:0.1

"attachmentpoints": [
{
"code": "Back",
 
 
 AttachmentPointAndPose apap = entity.AnimManager.Animator.GetAttachmentPointPose("Back"); 
 
 EntityShapeRenderer.cs -  void RenderHeldItem(bool isShadowPass)
 
.json patch in attachment points
	*/