/****************************************************************************
Copyright (c) 2010-2012 cocos2d-x.org
Copyright (c) 2011      Ricardo Quesada
Copyright (c) 2011      Zynga Inc.

http://www.cocos2d-x.org

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
****************************************************************************/
package org.cocos2d.shaders;

import org.cocos2d.cocoa.CCDictionary;
import org.cocos2d.include.ccMacros;

/**
 * @addtogroup shaders
 * @{
 */

/** CCShaderCache
 * Singleton that stores manages GL shaders
 * @since v2.0
 */
public class CCShaderCache {
	private static final String TAG = CCShaderCache.class.getSimpleName();

	public CCShaderCache() {
		m_pPrograms = null;
	}

	/** returns the shared instance */
	public static CCShaderCache sharedShaderCache() {
		synchronized(CCShaderCache.class) {
			if (_sharedShaderCache == null) {
				_sharedShaderCache = new CCShaderCache();
				if (_sharedShaderCache.init() == false) {
					_sharedShaderCache = null;
				}
			}
		}
		return _sharedShaderCache;
	}

	/** purges the cache. It releases the retained instance. */
	public static void purgeSharedShaderCache() {
		_sharedShaderCache = null;
	}

	/** loads the default shaders */
	public void loadDefaultShaders() {
		// Position Texture Color shader
		CCGLProgram p = new CCGLProgram();
		loadDefaultShader(p, kCCShaderType_PositionTextureColor);

		m_pPrograms.setObject(p, CCGLProgram.kCCShader_PositionTextureColor);

		// Position Texture Color alpha test
		p = new CCGLProgram();
		loadDefaultShader(p, kCCShaderType_PositionTextureColorAlphaTest);

		m_pPrograms.setObject(p, CCGLProgram.kCCShader_PositionTextureColorAlphaTest);

		//
		// Position, Color shader
		//
		p = new CCGLProgram();
		loadDefaultShader(p, kCCShaderType_PositionColor);

		m_pPrograms.setObject(p, CCGLProgram.kCCShader_PositionColor);

		//
		// Position Texture shader
		//
		p = new CCGLProgram();
		loadDefaultShader(p, kCCShaderType_PositionTexture);

		m_pPrograms.setObject(p, CCGLProgram.kCCShader_PositionTexture);

		//
		// Position, Texture attribs, 1 Color as uniform shader
		//
		p = new CCGLProgram();
		loadDefaultShader(p, kCCShaderType_PositionTexture_uColor);

		m_pPrograms.setObject(p, CCGLProgram.kCCShader_PositionTexture_uColor);

		//
		// Position Texture A8 Color shader
		//
		p = new CCGLProgram();
		loadDefaultShader(p, kCCShaderType_PositionTextureA8Color);

		m_pPrograms.setObject(p, CCGLProgram.kCCShader_PositionTextureA8Color);

		//
		// Position and 1 color passed as a uniform (to simulate glColor4ub )
		//
		p = new CCGLProgram();
		loadDefaultShader(p, kCCShaderType_Position_uColor);
		
		m_pPrograms.setObject(p, CCGLProgram.kCCShader_Position_uColor);

		//
		// Position, Legth(TexCoords, Color (used by Draw Node basically )
		//
		p = new CCGLProgram();
		loadDefaultShader(p, kCCShaderType_PositionLengthTexureColor);

		m_pPrograms.setObject(p, CCGLProgram.kCCShader_PositionLengthTexureColor);
	}

	/** reload the default shaders */
	public void reloadDefaultShaders() {
		// reset all programs and reload them

		// Position Texture Color shader
		CCGLProgram p = programForKey(CCGLProgram.kCCShader_PositionTextureColor);
		p.reset();
		loadDefaultShader(p, kCCShaderType_PositionTextureColor);

		// Position Texture Color alpha test
		p = programForKey(CCGLProgram.kCCShader_PositionTextureColorAlphaTest);
		p.reset();
		loadDefaultShader(p, kCCShaderType_PositionTextureColorAlphaTest);

		//
		// Position, Color shader
		//
		p = programForKey(CCGLProgram.kCCShader_PositionColor);
		p.reset();
		loadDefaultShader(p, kCCShaderType_PositionColor);

		//
		// Position Texture shader
		//
		p = programForKey(CCGLProgram.kCCShader_PositionTexture);
		p.reset();
		loadDefaultShader(p, kCCShaderType_PositionTexture);

		//
		// Position, Texture attribs, 1 Color as uniform shader
		//
		p = programForKey(CCGLProgram.kCCShader_PositionTexture_uColor);
		p.reset();
		loadDefaultShader(p, kCCShaderType_PositionTexture_uColor);

		//
		// Position Texture A8 Color shader
		//
		p = programForKey(CCGLProgram.kCCShader_PositionTextureA8Color);
		p.reset();
		loadDefaultShader(p, kCCShaderType_PositionTextureA8Color);

		//
		// Position and 1 color passed as a uniform (to simulate glColor4ub )
		//
		p = programForKey(CCGLProgram.kCCShader_Position_uColor);
		p.reset();
		loadDefaultShader(p, kCCShaderType_Position_uColor);

		//
		// Position, Legth(TexCoords, Color (used by Draw Node basically )
		//
		p = programForKey(CCGLProgram.kCCShader_PositionLengthTexureColor);
		p.reset();
		loadDefaultShader(p, kCCShaderType_Position_uColor);
	}

	/** returns a GL program for a given key */
	public CCGLProgram programForKey(final String key) {
		return (CCGLProgram)m_pPrograms.objectForKey(key);
	}

	/** adds a CCGLProgram to the cache for a given name */
	public void addProgram(CCGLProgram program, final String key) {
		m_pPrograms.setObject(program, key);
	}

	private boolean init() {
		m_pPrograms = new CCDictionary();
		loadDefaultShaders();
		return true;
	}

	private void loadDefaultShader(CCGLProgram p, int type) {
		switch (type) {
		case kCCShaderType_PositionTextureColor:
			p.initWithVertexShaderByteArray(ccShaders.ccPositionTextureColor_vert, ccShaders.ccPositionTextureColor_frag);

			p.addAttribute(CCGLProgram.kCCAttributeNamePosition, CCGLProgram.kCCVertexAttrib_Position);
			p.addAttribute(CCGLProgram.kCCAttributeNameColor, CCGLProgram.kCCVertexAttrib_Color);
			p.addAttribute(CCGLProgram.kCCAttributeNameTexCoord, CCGLProgram.kCCVertexAttrib_TexCoords);

			break;
		case kCCShaderType_PositionTextureColorAlphaTest:
			p.initWithVertexShaderByteArray(ccShaders.ccPositionTextureColor_vert, ccShaders.ccPositionTextureColorAlphaTest_frag);

			p.addAttribute(CCGLProgram.kCCAttributeNamePosition, CCGLProgram.kCCVertexAttrib_Position);
			p.addAttribute(CCGLProgram.kCCAttributeNameColor, CCGLProgram.kCCVertexAttrib_Color);
			p.addAttribute(CCGLProgram.kCCAttributeNameTexCoord, CCGLProgram.kCCVertexAttrib_TexCoords);

			break;
		case kCCShaderType_PositionColor:
			p.initWithVertexShaderByteArray(ccShaders.ccPositionColor_vert, ccShaders.ccPositionColor_frag);

			p.addAttribute(CCGLProgram.kCCAttributeNamePosition, CCGLProgram.kCCVertexAttrib_Position);
			p.addAttribute(CCGLProgram.kCCAttributeNameColor, CCGLProgram.kCCVertexAttrib_Color);

			break;
		case kCCShaderType_PositionTexture:
			p.initWithVertexShaderByteArray(ccShaders.ccPositionTexture_vert, ccShaders.ccPositionTexture_frag);

			p.addAttribute(CCGLProgram.kCCAttributeNamePosition, CCGLProgram.kCCVertexAttrib_Position);
			p.addAttribute(CCGLProgram.kCCAttributeNameTexCoord, CCGLProgram.kCCVertexAttrib_TexCoords);

			break;
		case kCCShaderType_PositionTexture_uColor:
			p.initWithVertexShaderByteArray(ccShaders.ccPositionTexture_uColor_vert, ccShaders.ccPositionTexture_uColor_frag);

			p.addAttribute(CCGLProgram.kCCAttributeNamePosition, CCGLProgram.kCCVertexAttrib_Position);
			p.addAttribute(CCGLProgram.kCCAttributeNameTexCoord, CCGLProgram.kCCVertexAttrib_TexCoords);

			break;
		case kCCShaderType_PositionTextureA8Color:
			p.initWithVertexShaderByteArray(ccShaders.ccPositionTextureA8Color_vert, ccShaders.ccPositionTextureA8Color_frag);

			p.addAttribute(CCGLProgram.kCCAttributeNamePosition, CCGLProgram.kCCVertexAttrib_Position);
			p.addAttribute(CCGLProgram.kCCAttributeNameColor, CCGLProgram.kCCVertexAttrib_Color);
			p.addAttribute(CCGLProgram.kCCAttributeNameTexCoord, CCGLProgram.kCCVertexAttrib_TexCoords);

			break;
		case kCCShaderType_Position_uColor:
			p.initWithVertexShaderByteArray(ccShaders.ccPosition_uColor_vert, ccShaders.ccPosition_uColor_frag);

			p.addAttribute("aVertex", CCGLProgram.kCCVertexAttrib_Position);

			break;
		case kCCShaderType_PositionLengthTexureColor:
			p.initWithVertexShaderByteArray(ccShaders.ccPositionColorLengthTexture_vert, ccShaders.ccPositionColorLengthTexture_frag);

			p.addAttribute(CCGLProgram.kCCAttributeNamePosition, CCGLProgram.kCCVertexAttrib_Position);
			p.addAttribute(CCGLProgram.kCCAttributeNameTexCoord, CCGLProgram.kCCVertexAttrib_TexCoords);
			p.addAttribute(CCGLProgram.kCCAttributeNameColor, CCGLProgram.kCCVertexAttrib_Color);

			break;
		default:
			ccMacros.CCLOG(TAG, "cocos2d:error shader type");
			return;
		}

		p.link();
		p.updateUniforms();

		ccMacros.CHECK_GL_ERROR_DEBUG();
	}

	private CCDictionary m_pPrograms;

	private static final int kCCShaderType_PositionTextureColor = 0;
	private static final int kCCShaderType_PositionTextureColorAlphaTest = 1;
	private static final int kCCShaderType_PositionColor = 2;
	private static final int kCCShaderType_PositionTexture = 3;
	private static final int kCCShaderType_PositionTexture_uColor = 4;
	private static final int kCCShaderType_PositionTextureA8Color = 5;
	private static final int kCCShaderType_Position_uColor = 6;
	private static final int kCCShaderType_PositionLengthTexureColor = 7;
	private static final int kCCShaderType_MAX = 8;

	private static CCShaderCache _sharedShaderCache = null;
}

// end of shaders group
/// @}
