//
//  LogicalLine.h
//  Fumizuki
//
//  Created by 二鏡 on 11/10/30.
//  Copyright 2011年 二鏡庵. All rights reserved.
//


#import "GlyphCluster.h"
#import "CFRangeExt.h"
#import <vector>

using namespace std;

// 行の論理構造を担当する。
class LogicalLine
{
    static NSImage *_return_icon;
    vector<GlyphCluster*> _clusters;
    
    CFRange _character_range;
    bool _is_closed; // 改行終了するlineは閉じているという
    vector<CGFloat> _border_offsets; // 末尾を含んでいるのに注意。よって文字数より1つ多い
    CGFloat _box_size; // 行幅
    
public:
    LogicalLine(const CFRange &range, const CGFloat box_size,const bool is_closed, const vector<CGFloat>& borders)
    {
        _character_range = range;
        _box_size = box_size;
        _is_closed = is_closed;
        
        size_t border_count = range.length+1;
        _border_offsets.resize(border_count);
        copy(borders.begin(), borders.begin()+border_count, _border_offsets.begin());
    }
    
    ~LogicalLine()
    {
        for(vector<GlyphCluster*>::iterator it = _clusters.begin();
            it != _clusters.end();
            ++it)
        {
            delete *it;
        }
    }
    
    static void initialize_state()
    {
        id bundle = [NSBundle mainBundle];
        id URL = [bundle URLForImageResource: @"return"];
        _return_icon = [[NSImage alloc] initWithContentsOfURL: URL];
        [_return_icon setFlipped: YES];
    }
    
#pragma mark Property
    inline bool get_is_closed() const
    {
        return _is_closed;
    }
    
    inline CFRange get_character_range() const
    {
        return _character_range;
    }
    
    inline CFIndex get_character_count() const
    {
        return _character_range.length;
    }
    
    inline CGFloat get_border(CFIndex i) const
    {
        return _border_offsets[i];
    }
    
    inline CGFloat get_border_tail() const
    {
        return *(_border_offsets.end()-1);
    }
    
#pragma mark Predicate
    inline bool is_empty_line() const
    {
        return _character_range.length == 1 && _is_closed;
    }
    
    inline CFIndex convert_index_to_local(CFIndex index) const
    {
        return index - _character_range.location;
    }
    
    inline CFIndex convert_index_to_block(CFIndex index) const
    {
        return index + _character_range.location;
    }
#pragma mark Query
    // glyphのイメージボックス上かどうかを判定してindexを返す
    // 両端を超えた領域はkCFNotFoundとみなす
    CFIndex query_box_index(CGFloat offset) const
    {
        if(_border_offsets[0] > offset)
            return kCFNotFound;
        if(*(_border_offsets.end()-1) < offset)
            return kCFNotFound;
        
        for(size_t i=0;i<_border_offsets.size()-1;i++)
        {
            CGFloat border = _border_offsets[i+1];
            if(border > offset)
                return i;
        }
        return kCFNotFound;
    }
    
    // ある点からもっとも近いborderを返す。超えていた場合も必ず境界値を返す
    // is_tailは末端かどうかを返し、改行終端時の補正に利用する
    CFIndex query_nearest_border_index(CGFloat offset, bool &is_tail) const
    {
        CFIndex tail = _border_offsets.size()-1;
        if(offset < _border_offsets[0])
        {
            is_tail = false;
            return 0;
        }
        if(offset > _border_offsets[tail])
        {
            is_tail = true;
            return tail;
        }
        
        for(CFIndex i=0;i<tail;i++)
        {
            CGFloat bottom = _border_offsets[i+1];
            // まず挟み込みを検出
            // +---top---+  i
            // | offset  |
            // +--bottom-+  i+1
            if(bottom >= offset)
            {
                CGFloat top = _border_offsets[i];
                CGFloat delta_bottom = bottom-offset;
                CGFloat delta_top    = offset-top;
                if(delta_bottom > delta_top)
                {
                    is_tail = false;
                    return i;
                }
                else
                {
                    CFIndex ret = i+1;
                    is_tail = ret == tail ? true : false;
                    return ret;
                }
            }
        }
        return kCFNotFound;
    }
    
#pragma mark Operator
    inline NSRect make_caret_rect(CGFloat offset, CGPoint line_origin) const
    {
        CGFloat x = line_origin.x-_box_size/2.0;
        CGFloat y = line_origin.y+offset;
        CGFloat w = _box_size;
        CGFloat h = 1.0;
        return NSMakeRect(x,y,w,h);
    }
    
    // rangeは行内のrangeで受ける
    // actual_rangeも行内のrangeで返す
    inline NSRect make_bounding_rect(NSRange range, NSRangePointer actual_range, CGPoint line_origin) const
    {
        // locationは有効な行内ローカル(from 0)と仮定している
        // この時、末端までの距離と与えられた距離の短い方を新lengthとして採用する
        range.length = MIN(range.length, _character_range.length-range.location);
        if(actual_range != NULL)
            *actual_range = range;
        
        CFIndex top = range.location;
        CFIndex bottom = NSMaxRange(range);
        CGFloat x = line_origin.x-_box_size/2.0;
        CGFloat y = line_origin.y+_border_offsets[top];
        CGFloat w = _box_size;
        CGFloat h = _border_offsets[bottom]-_border_offsets[top]; 
        return NSMakeRect(x,y,w,h);
    }
    
    inline void add(GlyphCluster* cluster)
    {
        _clusters.push_back(cluster);
    }
    void draw_return_mark_if_need(CGContextRef context, CGPoint lineOrigin, CGFloat border) const
    {
        if(_is_closed == false)
            return ;
        
        CGFloat location = *(_border_offsets.end()-2);
        if(location < border)
            return;
        
        // image sizeを16x16と仮定している
        NSPoint point = NSMakePoint(lineOrigin.x-8,lineOrigin.y+location+1);
        [_return_icon drawAtPoint: point
                         fromRect: NSZeroRect
                        operation: NSCompositeSourceOver
                         fraction: 1.0];
    }
    
#pragma mark Render
    void draw(CGContextRef context, CGPoint line_origin) const
    {
        CGContextSetTextPosition(context, line_origin.x, line_origin.y);
        vector<GlyphCluster*>::const_iterator it = _clusters.begin();
        for(;it != _clusters.end();++it)
            (*it)->draw(context);
    }    
    
    // fill colorは外側でセットする。
    void fill_background(CGContextRef context, CGPoint line_origin, CFRange range) const
    {
        CGFloat top = _border_offsets[range.location];
        CGFloat bottom = _border_offsets[_CFRangeLimit(range)];
        
        CGFloat x = line_origin.x-_box_size/2.0;
        CGFloat y = line_origin.y+top;
        
        CGFloat w = _box_size;
        CGFloat h = bottom-top;
        if(range.location == 0)
        {
            y -= _border_offsets[0];
            h += _border_offsets[0];
        }
        NSRectFill(NSMakeRect(x,y,w,h));
    }
};

