//
//  MSLayoutBlockManager.m
//  Okusa
//
//  Created by 木谷 洋 on 12/03/02.
//  Copyright (c) 2012年 二鏡庵. All rights reserved.
//


#import "MSLayoutBlockManager.h"
#import "MSLayoutBlock.h"

NSString *MSLayoutBlockManagerWillChangeBlocksNotification = @"MSLayoutBlockManagerWillChangeBlocks";
NSString *MSLayoutBlockManagerDidChangeBlocksNotification = @"MSLayoutBlockManagerDidChangeBlocks";

static inline NSRange
_outdatedCharacterRange(NSTextStorage *storage)
{
    NSRange newRange = storage.editedRange;
    NSRange oldRange = ({
        NSUInteger changeInLength = storage.changeInLength;
        NSMakeRange(newRange.location,newRange.length-changeInLength);   
    });
    return oldRange;  
}

@implementation MSLayoutBlockManager
@synthesize editedRange = editedRange_;
- (id)init
{
    textStorage_ = [[NSTextStorage alloc] init];
    textStorage_.delegate = self;
    blocks_ = [NSMutableArray array];
    return self;
}

- (NSTextStorage*)textStorage
{
    return textStorage_;
}

- (NSArray*)blocks
{
    return [blocks_ copy];
}

- (NSArray*)subblocksWithRange:(NSRange)blockRange
{
    return [blocks_ subarrayWithRange: blockRange];
}

- (MSLayoutBlock*)lastBlock
{
    return [blocks_ lastObject];
}

// range内のblockについて、行が正しい範囲を指すことを保証する
- (void)_ensureValidLineOffsetsToLimit:(NSUInteger)limit
{
    id (*objAtIndex)(id,CFIndex) = (id(*)(id,CFIndex))CFArrayGetValueAtIndex;
    NSUInteger head; // range head/tail for test
    NSUInteger offset; // line offset

    // -1 -> 信用できるブロックはない
    if( mapping_.has_valid_line_mapping() == false)
    {
        // offset 0から初めてlimitまでループ
        offset = 0;
        head = 0;
    }
    else
    {
        // もし十分信用できるならば何もしない
        if( mapping_.is_line_mapped(limit-1))
            return ;
        
        NSUInteger validMapping = mapping_.get_valid_line_mapping();
        MSLayoutBlock *block = objAtIndex(blocks_,validMapping);
        offset = block.lineOffset + [block.lines count];
        head = validMapping + 1;
    }
    
    // last valid
    for(NSUInteger i=head;i<limit;i++)
    {
        MSLayoutBlock *block = objAtIndex(blocks_, i);
        block.lineOffset = offset;
        offset += [block.lines count];
    }
    mapping_.validate_line_mapping_to_limit(limit);
}

// ブロックのcharacterLocationをきちんと保証する
// ブロック内の文字数のみで決めるから、storageと一致するかどうかはここでは見ない
- (void)_ensureValidCharacterMappingToLimit:(NSUInteger)blockLimit
{
    id (*objAtIndex)(id,CFIndex) = (id(*)(id,CFIndex))CFArrayGetValueAtIndex;
    NSUInteger offset;
    NSUInteger head;
    
    if( mapping_.has_valid_character_mapping() == false)
    {
        // offset 0
        offset = 0;
        head = 0;
    }
    else
    {
        if( mapping_.is_character_mapped(blockLimit-1) )
            return;
        
        NSUInteger validMapping = mapping_.get_valid_character_mapping();
        MSLayoutBlock *block = objAtIndex(blocks_, validMapping);
        NSRange range = block.characterRange;
        offset = NSMaxRange(range);
        head = validMapping + 1;
    }
    
    for(NSUInteger i=head;i<blockLimit;i++)
    {
        MSLayoutBlock *block = objAtIndex(blocks_, i);
        block.characterLocation = offset;
        offset += block.characterLength;
    }
    mapping_.validate_character_mapping_to_limit(blockLimit);
}

- (NSRange)characterRangeOfBlock:(MSLayoutBlock*)aBlock
{
    NSUInteger idx = [blocks_ indexOfObject: aBlock];
    if(idx == NSNotFound)
        return NSMakeRange(NSNotFound,0);
    
    [self _ensureValidCharacterMappingToLimit: idx+1];
    return aBlock.characterRange;
}

- (NSRange)lineRangeOfBlock:(MSLayoutBlock*)aBlock
{
    NSUInteger idx = [blocks_ indexOfObject: aBlock];
    if(idx == NSNotFound)
        return NSMakeRange(NSNotFound, 0);
    [self _ensureValidLineOffsetsToLimit: idx+1];
    return NSMakeRange(aBlock.lineOffset, [aBlock.lines count]);
}

- (MSLayoutBlock*)blockAtCharacterIndex:(NSUInteger)index
{
    NSUInteger offset = 0;
    NSUInteger searchHead  = 0;
    NSUInteger limit = [blocks_ count];
    if( limit == 0 )
        return nil;
    
    if( mapping_.has_valid_character_mapping())
    {
        NSUInteger validMapping = mapping_.get_valid_character_mapping();
        MSLayoutBlock *block = [blocks_ objectAtIndex: validMapping];
        NSRange range = block.characterRange;
        if(NSLocationInRange(index, range))
            return block;
        
        if(range.location > index)
        {            
            // すでに計算された範囲に含まれている
            // この中でサーチ
            for(NSUInteger i=0;i<validMapping;i++)
            {
                block = [blocks_ objectAtIndex: i];
                NSRange range = block.characterRange;
                if(NSLocationInRange(index, range))
                    return block;
            }
            
            // ここに来たらバグ！
            NSAssert(NO, @"invalid character mapping!");
        }
        else
        {
            // 含まれないのでoffsetを設定して次へ
            offset = NSMaxRange(range);
            searchHead = validMapping+1;
        }
    }
    
    // 存在しないので、順次レンジ計算しつつ発見次第return
    // validMapped+1から初めて見つかるまで
    for( NSUInteger i=searchHead;i<limit;i++)
    {
        MSLayoutBlock *block = [blocks_ objectAtIndex: i];
        {
            block.characterLocation = offset;
            mapping_.validate_character_mapping(i);
        }
        CFIndex length = block.characterLength;
        offset += length;
        if(offset > index)
            return block;
    }
    
    return nil;
}

- (NSRange)lineRangeForBlockRange:(NSRange)aBlockRange;
{
    NSUInteger location = 0, length = 0;

    id (*objAtIndex)(id,CFIndex) = (id(*)(id,CFIndex))CFArrayGetValueAtIndex;
    NSUInteger limit = [blocks_ count];
    NSUInteger tail = NSMaxRange(aBlockRange);
    NSUInteger head = aBlockRange.location;
    
    // invalid range
    if(limit < tail)
        return NSMakeRange(NSNotFound, 0);
    
    if(aBlockRange.location == limit && aBlockRange.length == 0)
    {
        return NSMakeRange(NSNotFound,0);
    }
    
    // validを保証
    [self _ensureValidLineOffsetsToLimit: NSMaxRange(aBlockRange)];
    
    MSLayoutBlock *headBlock = objAtIndex(blocks_, head);
    MSLayoutBlock *tailBlock = objAtIndex(blocks_, tail-1);
    location = headBlock.lineOffset;
    length = tailBlock.lineOffset + [tailBlock.lines count] - location;

    return NSMakeRange(location,length);
}

// 自動シンクロ.
@end


@implementation MSLayoutBlockManager (TextStorageProxy)
- (void)_testNewlineInsertion:(id)aString
             replacementRange:(NSRange)replacementRange
{    
    // 改行入力のチェック
    // 新規/置換の文字列を調べ、改行状態によって
    // content生成に指示を出す
    
    // 前準備
    NSUInteger length = [aString length];
    if(length == 0)
        return ;
    
    // 文字列に変換
    if([aString isKindOfClass: [NSAttributedString class]])
        aString = [aString string];
    
    // 新改行のテスト
    BOOL hasNewInsertion,hasOldNewline = NO;
    CFCharacterSetRef newlineSet = CFCharacterSetGetPredefined(kCFCharacterSetNewline);
    
    // 旧改行のテスト
    unichar newLast = [aString characterAtIndex: length-1];
    hasNewInsertion = CFCharacterSetIsCharacterMember(newlineSet, newLast);
    if(replacementRange.length != 0)
    {        
        unichar oldLast = [[textStorage_ mutableString] characterAtIndex: NSMaxRange(replacementRange)-1];
        hasOldNewline = CFCharacterSetIsCharacterMember(newlineSet, oldLast);
    }
    
    // 共に改行状態ならばそのままでよい
    if(hasNewInsertion != hasOldNewline)
    {
        // 対応する側のフラグを立てる
        if(hasNewInsertion)
        {
            synchronizeAffection_.removalAffect = 0;
            synchronizeAffection_.creationAffect = 1;            
        }
        else
        {
            synchronizeAffection_.removalAffect = 1;
            synchronizeAffection_.creationAffect = 0;
        }
    }
}

- (void)_stringWillReplace:(id)string
                 withRange:(NSRange)range
{
    if(string == nil)
    {
        // delete
        if(NSMaxRange(range) != [textStorage_ length])
        {
            unichar last = [[textStorage_ mutableString] characterAtIndex: NSMaxRange(range)-1];
            CFCharacterSetRef newlineSet = CFCharacterSetGetPredefined(kCFCharacterSetNewline);
            if(CFCharacterSetIsCharacterMember(newlineSet, last))
            {
                synchronizeAffection_.removalAffect = 1;
                synchronizeAffection_.creationAffect = 0;
            }
        }
    }
    else
    {
        // 入力
        [self _testNewlineInsertion: string
                   replacementRange: range];
    }
}

- (void)replaceCharactersInRange:(NSRange)range
            withAttributedString:(NSAttributedString*)aStr
{
    [self _stringWillReplace: aStr withRange: range];
    
    [textStorage_ beginEditing];
    if(aStr == nil)
    {
        [textStorage_ deleteCharactersInRange: range];
    }
    else
    {
        [textStorage_ replaceCharactersInRange: range
                          withAttributedString: aStr];
    }
    [textStorage_ endEditing];
}

- (void)loadString:(NSAttributedString*)aString
{
    [self replaceCharactersInRange: NSMakeRange(0,[textStorage_ length])
              withAttributedString: aString];
}

- (NSData*)dataWithEncoding:(NSStringEncoding)aEncoding
{
    id string = [textStorage_ mutableString];
    return [string dataUsingEncoding: aEncoding];
}

// 現状のパラグラフにおいて、文字列の範囲に関わるパラグラフの範囲を返す
// 直前までの情報はキャッシュされる
- (NSRange)blockRangeForCharacterRange:(NSRange)dirtyRange
{
    MSLayoutBlock *block = nil;
    id (*objAtIndex)(id,CFIndex) = (id(*)(id,CFIndex))CFArrayGetValueAtIndex;

    CFIndex first = dirtyRange.location;
    CFIndex last = dirtyRange.length == 0 ? first : NSMaxRange(dirtyRange)-1;
    CFIndex bLimit = [blocks_ count];
    
    if(bLimit == 0)
        return NSMakeRange(NSNotFound,0);
    
    CFIndex location = 0;
    CFIndex idx = 0; // block index
    CFIndex length;
    
    if(syncCache_.valid())
    {
        CFIndex b,c;
        syncCache_.get_state(b, c);
        if(first > c)
        {
            idx = b+1;
            location = c;
        }
        else
        {
            syncCache_.invalidate();
        }
    }
    
    // 直前までを保証する
    [self _ensureValidCharacterMappingToLimit: idx];

    // find first character
    for(;idx<bLimit;idx++)
    {
        block = objAtIndex(blocks_,idx); //[paragraphs objectAtIndex: idx];
        length = block.characterLength;
        block.characterLocation = location; // 再配置
        mapping_.validate_character_mapping(idx);
        CFRange range = CFRangeMake(location,length);
        location+=length;

        // 末尾追加はこの条件では脱出しない
        if(_CFLocationInRange(first,range))
            break;
        
        // 未到達の場合に限ってキャッシュするので
        // 編集途上のhotspotはキャッシュを破壊しない
        syncCache_.cache(idx, location);
    }
    
    location -= length;
    // special case
    // 末尾に文字追加が入った時
    if(idx == bLimit)
    {
        if([block isClosed])
        {
            return NSMakeRange(bLimit,0);
        }
        else
        {
            syncCache_.cache(idx-2,location);
            return NSMakeRange(bLimit-1,1);
        }
    }
    CFIndex firstIndex = idx;
    
    // find last character
    for(;idx<bLimit;idx++)
    {
        MSLayoutBlock *block = objAtIndex(blocks_,idx); //[paragraphs objectAtIndex: idx];
        length = block.characterLength;
        CFRange range = CFRangeMake(location,length);
        location+=length;
        if(_CFLocationInRange(last,range))
        {
            break;        
        }
    }
    // overflow backtrack
    CFIndex lastIndex = MIN(idx,bLimit-1);
    
    // range = first ~ last
    NSRange ret = NSMakeRange(firstIndex,lastIndex-firstIndex+1);
    return ret;
}

- (NSArray*)_makeParagraphsInCharacterRange:(NSRange)referenceRange
{
    id tmp = [NSMutableArray array];
    
    id string = [textStorage_ mutableString];
    NSRange refLocation = NSMakeRange(referenceRange.location, 0);
    NSUInteger limit = NSMaxRange(referenceRange);
    
    NSUInteger start,end;
    [string getLineStart: &start
                     end: &end
             contentsEnd: NULL
                forRange: refLocation];
    
    for(;;)
    {
        NSRange paragraphRange = NSMakeRange(start,end-start);
        if(paragraphRange.length == 0)
            break;
        id aStr = [textStorage_ attributedSubstringFromRange: paragraphRange];
        
        id block = [[MSLayoutBlock alloc] initWithAttributedString: aStr];
        [tmp addObject: block];
        if(end >= limit)
            break;
        refLocation = NSMakeRange(end,0);
        [string getLineStart: &start
                         end: &end
                 contentsEnd: NULL
                    forRange: refLocation];
        
    }
    return [tmp copy];
}

- (NSRange)outdatedCharacterRange
{
    NSRange newRange = textStorage_.editedRange;
    NSRange oldRange = ({
        NSUInteger changeInLength = textStorage_.changeInLength;
        NSMakeRange(newRange.location,newRange.length-changeInLength);   
    });
    oldRange.length += synchronizeAffection_.removalAffect;
    return oldRange;
}

- (void)textStorageDidProcessEditing:(id)notif
{
    id center = [NSNotificationCenter defaultCenter];
    
    NSRange newCharacterRange = textStorage_.editedRange;
    NSRange oldCharacterRange = ({
        NSUInteger changeInLength = textStorage_.changeInLength;
        NSMakeRange(newCharacterRange.location,
                    newCharacterRange.length-changeInLength);   
    });
    
    // 改行の増減による更新範囲補正
    oldCharacterRange.length += synchronizeAffection_.removalAffect;
    newCharacterRange.length += synchronizeAffection_.creationAffect;
    
    // 新旧のブロック情報を構築
    NSRange oldBlockRange = [self blockRangeForCharacterRange: oldCharacterRange];
    id newBlocks = [self _makeParagraphsInCharacterRange: newCharacterRange];
    
    // notificationを吐く
    editedRange_ = oldBlockRange;
    [center postNotificationName: MSLayoutBlockManagerWillChangeBlocksNotification
                          object: self];
    
    // 入れ替え
    if(oldBlockRange.location == NSNotFound)
    {
        [blocks_ addObjectsFromArray: newBlocks];
        oldBlockRange.location = 0;
    }
    else
    {
        [blocks_ replaceObjectsInRange: oldBlockRange
                  withObjectsFromArray: newBlocks];
        if( oldBlockRange.location == 0 )
        {
            // 根こそぎ変えたので無効化
            mapping_.invalidate_line_mapping();
            mapping_.invalidate_character_mapping();
        }
        else
        {
            // すでに無効でない場合は入替えの直前まで有効値
            if(mapping_.has_valid_line_mapping())
                mapping_.cut_line_mapping_to_limit(oldBlockRange.location);
            if(mapping_.has_valid_character_mapping())
                mapping_.cut_character_mapping_to_limit(oldBlockRange.location);
        }
    }
    
    // 情報を更新
    editedRange_ = NSMakeRange(oldBlockRange.location, [newBlocks count]);
    
    // 再初期化
    synchronizeAffection_.removalAffect = 0;
    synchronizeAffection_.creationAffect = 0;

    // notificationを吐く
    [center
     postNotificationName: MSLayoutBlockManagerDidChangeBlocksNotification
     object: self];
}

@end
