//
//  OpenType.h
//  VTKit
//
//  Created by 木谷 洋 on 12/01/30.
//  Copyright (c) 2012年 二鏡庵. All rights reserved.
//

#ifndef VTKit_OpenType_h
#define VTKit_OpenType_h
#include <ext/hash_map>
#include <CoreFoundation/CoreFoundation.h>

using namespace std;
using namespace __gnu_cxx;

typedef enum
{
    OT_LookupFlag_RightToLeft = 0x0001,
    OT_LookupFlag_IgnoreBaseGlyphs = 0x0002,
    OT_LookupFlag_IgnoreLigatures = 0x0004,
    OT_LookupFlag_IgnoreMarks = 0x0008,
    OT_LookupFlag_UseMarkFilteringSet = 0x0010,
    OT_LookupFlag_MarkAttachmentType = 0xFF00,
} OT_LookupFlag_t;

// feature別lookupリストのテーブル
class OT_Feature
{
private:
    uint32_t tag_; // 4bytesコードをパックしたもの。'code'とバイナリ比較できる
    uint16_t count_;
    uint16_t *lookup_indexes_;
    
public:
    OT_Feature(CFDataRef table, uint16_t offset, uint32_t tag)
    {
        tag_ = tag;
        
        // layout:
        //  offset+0 param: uint16(NULL)
        //        +2 count: uint16
        //        +4 index: uint16[count]
        
        // countの読み出し
        CFDataGetBytes(table, CFRangeMake(offset+2, 2), (UInt8*)&count_);
        count_ = CFSwapInt16BigToHost(count_);
        
        // メモリ確保
        lookup_indexes_ = new uint16_t[count_];
        
        // データ読み出し
        CFDataGetBytes(table, CFRangeMake(offset+4, 2*count_), (UInt8*)lookup_indexes_);
        
        // スワップ
        for(int i=0;i<count_;i++)
            lookup_indexes_[i] = CFSwapInt16BigToHost(lookup_indexes_[i]);
    }
    
    ~OT_Feature()
    {
        delete [] lookup_indexes_;
    }
    
    inline uint32_t get_tag() const
    {
        return tag_;
    }
    
    inline uint16_t get_count() const
    {
        return count_;
    }
    
    inline uint16_t get_lookup_index(int i) const
    {
        return lookup_indexes_[i];
    }
};

class OT_Lookup
{
private:
    uint16_t offset_; // self offset.ここに各サブテーブルへのオフセットを足して実体を得る
    uint16_t type_;
    uint16_t flag_;
    uint16_t table_count_;
    uint16_t *table_offsets_;
    uint16_t mark_filtering_; // 未使用でいく
    
public:
    OT_Lookup(CFDataRef table, uint16_t offset)
    {
        offset_ = offset;
        // 読み出し
        
        // layout
        //  +0    type: uint16_t
        //  +2    flag: uint16_t 
        //  +4   count: uint16_t
        //  +6 offsets: uint16_t
        // optional:
        //  +6+2*count filter: uint16_t
        //  filterはflagがある場合に存在しているが、縦置換では不要
        
        // 基本情報の読み出し
        CFDataGetBytes(table, CFRangeMake(offset_+0,2), (UInt8*)&type_);
        CFDataGetBytes(table, CFRangeMake(offset_+2,2), (UInt8*)&flag_);
        CFDataGetBytes(table, CFRangeMake(offset_+4,2), (UInt8*)&table_count_);
        
        // スワップ
        type_ = CFSwapInt16BigToHost(type_);
        flag_ = CFSwapInt16BigToHost(flag_);
        table_count_ = CFSwapInt16BigToHost(table_count_);
        
        // メモリ準備
        table_offsets_ = new uint16_t[table_count_];
        
        // オフセットの読み出し
        CFDataGetBytes(table, CFRangeMake(offset+6, 2*table_count_), (UInt8*)table_offsets_);
        
        // スワップ
        for(uint16_t i=0;i<table_count_;i++)
            table_offsets_[i] = CFSwapInt16BigToHost(table_offsets_[i]);
    }
    
    ~OT_Lookup()
    {
        delete[] table_offsets_;
    }
    
    inline uint16_t get_subtable_count() const
    {
        return table_count_;
    }
    
    inline uint16_t get_subtable_offset_to_index(int i) const
    {
        return offset_+table_offsets_[i];
    }
};

typedef
void (*lookup_single_iterator_callback)(uint16_t src, uint16_t dst, void* context);

// 関係するグリフの定義表。GSUB format1/2のsubstitutionについて評価可能
class OT_Coverage
{
    // layout 
    //  +0 type: uint16_t
    //  .....
public:
    static inline uint16_t type_of_coverage(CFDataRef table, uint16_t offset)
    {
        uint16_t type;
        CFDataGetBytes(table, CFRangeMake(offset,2), (UInt8*)&type);
        return CFSwapInt16BigToHost(type);
    }
    
    virtual void iterate_coverage_GSUB_format1(uint16_t delta,  lookup_single_iterator_callback callback, void *context) const = 0;
    virtual void iterate_coverage_GSUB_format2(uint16_t *table, lookup_single_iterator_callback callback, void *context) const = 0;
};

class OT_Coverage_Format1 : public OT_Coverage
{
    // 直接列挙方式
    // layout
    //  +0   type: uint16_t
    //  +2  count: uint16_t
    //  +4 glyphs: uint16_t[count]
private:
    uint16_t count_;
    uint16_t *glyphs_;
    
public:
    OT_Coverage_Format1(CFDataRef table, uint16_t offset)
    {
        // countを読み出し
        CFDataGetBytes(table, CFRangeMake(offset+2, 2), (UInt8*)&count_);
        count_ = CFSwapInt16BigToHost(count_);
        
        // メモリ確保
        glyphs_ = new uint16_t[count_];
        
        // glyph読み出し
        CFDataGetBytes(table, CFRangeMake(offset+4, 2*count_), (UInt8*)glyphs_);
        for(uint16_t i=0;i<count_; i++)
            glyphs_[i] = CFSwapInt16BigToHost(glyphs_[i]);
    }
    
    ~OT_Coverage_Format1()
    {
        delete[] glyphs_;
    }
    
    void iterate_coverage_GSUB_format1(uint16_t delta, lookup_single_iterator_callback callback, void *context) const
    {
        for(uint16_t i=0;i<count_;i++)
        {
            callback(glyphs_[i], glyphs_[i]+delta, context);
        }
    }
    
    void iterate_coverage_GSUB_format2(uint16_t *table, lookup_single_iterator_callback callback, void *context) const
    {
        for(uint16_t i=0;i<count_;i++)
        {
            callback(glyphs_[i], table[i], context);
        }
    }
};

class OT_Coverage_Format2 : public OT_Coverage
{
    // レンジ方式
    // layout
    //  +0   type: uint16
    //  +2  count: uint16
    //  +4 record: Record[]
    //    Record(6bytes)
    //      +0 start: uint16
    //      +2   end: uint16
    //      +4   SCI: uint16
    
    struct Range {
        uint16_t start;
        uint16_t end;
        uint16_t SCI; // StartCoverageIndex
    } ;
    
    uint16_t count_;
    Range *ranges_;
    
public:
    OT_Coverage_Format2(CFDataRef table, uint16_t offset)
    {
        // countを読み出し
        CFDataGetBytes(table, CFRangeMake(offset+2, 2), (UInt8*)&count_);
        count_ = CFSwapInt16BigToHost(count_);
        
        ranges_ = new Range[count_];
        
        // 一端直列で読み出し
        size_t array_size = count_*6;
        UInt8 *buf = (UInt8*)alloca(array_size);
        CFDataGetBytes(table, CFRangeMake(offset+4 , array_size), buf);
        
        // スワップ代入
        for(int i=0;i<count_;i++)
        {
            ranges_[i].start = CFSwapInt16BigToHost(*(uint16_t*)(buf+i*6+0));
            ranges_[i].end   = CFSwapInt16BigToHost(*(uint16_t*)(buf+i*6+2));
            ranges_[i].SCI   = CFSwapInt16BigToHost(*(uint16_t*)(buf+i*6+4));
        }
    }
    
    ~OT_Coverage_Format2()
    {
        delete[] ranges_;
    }
    
    void iterate_coverage_GSUB_format1(uint16_t delta, lookup_single_iterator_callback callback, void *context) const
    {
        for(int i=0;i<count_;i++)
        {
            for(uint16_t g=ranges_[i].start;g<=ranges_[i].end; g++)
            {
                callback(g,g+delta,context);
            }
        }
    }
    void iterate_coverage_GSUB_format2(uint16_t *table, lookup_single_iterator_callback callback, void *context) const
    {
        int index=0;
        for(int i=0;i<count_;i++)
        {
            for(uint16_t g=ranges_[i].start;g<=ranges_[i].end; g++)
            {
                callback(g,table[index++],context);
            }
        }        
    }
};

class OT_Coverage_Factory
{
public:
    static OT_Coverage*
    load_coverage(CFDataRef table, uint16_t offset)
    {
        uint16_t type = OT_Coverage::type_of_coverage(table, offset);
        if(type == 1)
        {
            return new OT_Coverage_Format1(table, offset);
        }
        else
        {
            return new OT_Coverage_Format2(table, offset);
        }
    }
};

class OT_Substitution
{
protected:
    OT_Coverage *coverage;
    OT_Substitution(CFDataRef table, uint16_t offset)
    {
        // layout
        //  +0            type: uint16
        //  +2 coverage offset: uint16
        //  ....
        uint16_t cov_offset;
        CFDataGetBytes(table, CFRangeMake(offset+2,2), (UInt8*)&cov_offset);
        cov_offset = offset+CFSwapInt16BigToHost(cov_offset);
        coverage = OT_Coverage_Factory::load_coverage(table, cov_offset);
    }
    
public:
    ~OT_Substitution()
    {
        delete coverage;
    }
    
public:
    virtual uint16_t get_type() const = 0;
    virtual void iterate_single_lookup(lookup_single_iterator_callback callback, void *context) const = 0;
};

class OT_Substitution1 : public OT_Substitution
{
private:
    uint16_t delta_;
public:
    OT_Substitution1(CFDataRef table, uint16_t offset) : OT_Substitution(table, offset)
    {
        // layout
        //  +0   format: uint16_t
        //  +2 coverage: uint16_t
        //  +4    delta: uint16_t
        CFDataGetBytes(table, CFRangeMake(offset+4,2), (UInt8*)&delta_);
        delta_ = CFSwapInt16BigToHost(delta_);
    }
    
    uint16_t get_type() const
    {
        return 1;
    }
    
    void iterate_single_lookup(lookup_single_iterator_callback callback, void *context) const
    {
        coverage->iterate_coverage_GSUB_format1(delta_, callback, context);
    }
};

class OT_Substitution2 : public OT_Substitution
{
private:
    uint16_t *glyphs_;
public:
    OT_Substitution2(CFDataRef table, uint16_t offset) : OT_Substitution(table, offset)
    {
        // layout
        //  +0   format: uint16_t
        //  +2 coverage: uint16_t
        //  +4    count: uint16_t
        //  +6   glyphs: uint16_t[count]
        uint16_t count;
        CFDataGetBytes(table, CFRangeMake(offset+4,2), (UInt8*)&count);
        count = CFSwapInt16BigToHost(count);
        glyphs_ = new uint16_t[count];
        CFDataGetBytes(table, CFRangeMake(offset+6, 2*count), (UInt8*)glyphs_);
        for(int i=0;i<count;i++)
            glyphs_[i] = CFSwapInt16BigToHost(glyphs_[i]);
    }
    
    ~OT_Substitution2()
    {
        delete glyphs_;
    }
    
    uint16_t get_type() const
    {
        return 2;
    }
    
    void iterate_single_lookup(lookup_single_iterator_callback callback, void *context) const
    {
        coverage->iterate_coverage_GSUB_format2(glyphs_, callback, context);
    }
};

class OT_Substitution_Factory
{
public:
    static OT_Substitution*
    load_substitution(CFDataRef table, uint16_t offset)
    {
        uint16_t type;
        CFDataGetBytes(table, CFRangeMake(offset,2), (UInt8*)&type);
        type = CFSwapInt16BigToHost(type);
        switch(type)
        {
            case 1:
                // single 1
                return new OT_Substitution1(table, offset);
            case 2:
                // single 2
                return new OT_Substitution2(table, offset);
            default:
                // 他は未サポート
                return NULL;
        }
    }
};

// 総合リーダー
class OT_GSUBReader 
{
private:
    CFDataRef table_;
    // 各listへのオフセット値
    uint16_t script_offset_;
    uint16_t features_offset_;
    uint16_t lookup_offset_;
    
public:
    OT_GSUBReader(CFDataRef table)
    {
        table_ = (CFDataRef)CFRetain(table);
        // layout
        // +0     version: int32
        // +4  ScriptList: int16
        // +6 FeatureList: int16
        // +8  LookupList: int16
        
        // read
        CFDataGetBytes(table_, CFRangeMake(4,2), (UInt8*)&script_offset_);
        CFDataGetBytes(table_, CFRangeMake(6,2), (UInt8*)&features_offset_);
        CFDataGetBytes(table_, CFRangeMake(8,2), (UInt8*)&lookup_offset_);
        
        // swap
        script_offset_ = CFSwapInt16BigToHost(script_offset_);
        features_offset_ = CFSwapInt16BigToHost(features_offset_);
        lookup_offset_ = CFSwapInt16BigToHost(lookup_offset_);
    }
    
    ~OT_GSUBReader()
    {
        CFRelease(table_);
    }
    
    // Feature layout
    // features_offset_ +0 count: uint16
    //                  +2 Record[count]
    //  Record(6bytes)
    //   +0    tag: uint8[4] (uint32_t)
    //   +4 offset: uint16_t 
    // すなわち、6bytesのレコードが+2よりcount個続く
    OT_Feature *
    load_fature_by_index(uint32_t index) const
    {
        // インデックスのチェック
        uint16_t count;
        CFDataGetBytes(table_, CFRangeMake(features_offset_,2), (UInt8*)&count);
        count = CFSwapInt16BigToHost(count);
        if(index >= count)
            return NULL;
        
        // レコードの読み出し
        uint32_t tag;
        uint16_t offset;
        uint16_t record_offset = features_offset_+2+6*index;
        CFDataGetBytes(table_, CFRangeMake(record_offset+0,4), (UInt8*)&tag);
        CFDataGetBytes(table_, CFRangeMake(record_offset+4,2), (UInt8*)&offset);
        
        // スワップ
        tag = CFSwapInt32BigToHost(tag);
        offset = CFSwapInt16BigToHost(offset);
        
        return new OT_Feature(table_, features_offset_+offset, tag);
    }
    
    OT_Feature*
    load_feature_by_tag(uint32_t key) const
    {
        uint16_t count;
        CFDataGetBytes(table_, CFRangeMake(features_offset_,2), (UInt8*)&count);
        count = CFSwapInt16BigToHost(count);
        
        for(uint16_t i=0; i<count; i++)
        {
            // i番目のrecord
            uint16_t record_offset = features_offset_+2+6*i;
            
            // タグの読み出し
            uint32_t tag;
            CFDataGetBytes(table_, CFRangeMake(record_offset+0,4), (UInt8*)&tag);
            tag = CFSwapInt32BigToHost(tag);
            if(tag != key)
                continue;
            
            // タグが一致
            uint16_t offset;
            CFDataGetBytes(table_, CFRangeMake(record_offset+4,2), (UInt8*)&offset);
            offset = CFSwapInt16BigToHost(offset);
            
            return new OT_Feature(table_, features_offset_+offset, tag);
        }
        
        // 発見できない
        return NULL;
    }
    
    OT_Lookup*
    load_lookup_by_index(int i)
    {
        // layout
        //  +0   count: uint16_t
        //  +2 offsets: uint16_t[count]
        
        // index test
        uint16_t count;
        CFDataGetBytes(table_, CFRangeMake(lookup_offset_,2), (UInt8*)&count);
        count = CFSwapInt16BigToHost(count);
        if(i >= count)
            return NULL;
        
        uint16_t offset;
        CFDataGetBytes(table_, CFRangeMake(lookup_offset_+2+i*2,2), (UInt8*)&offset);
        offset = CFSwapInt16BigToHost(offset);
        return new OT_Lookup(table_, lookup_offset_+offset);
    }
    
    OT_Substitution*
    load_substitution(const OT_Lookup *lookup, uint16_t index) const
    {
        uint16_t offset = lookup->get_subtable_offset_to_index(index);
        return OT_Substitution_Factory::load_substitution(table_, offset);
    }
};


// GlyphID -> GlyphIDのマップ
class OT_GSUBSingleGlyphMap
{
private:
    hash_map<uint16_t,uint16_t> table;
public:
    void insert(uint16_t src, uint16_t dst)
    {
        table.insert(make_pair(src,dst));
    }
    unsigned long size() const
    {
        return table.size();
    }
    inline uint16_t find(uint16_t src) const // return src if not found
    {
        hash_map<uint16_t,uint16_t>::const_iterator p;
        
        p=table.find(src);
        if( p == table.end())
            return src;
        else
            return p->second;
    }
};


#endif

