//
//  VerticalGlyphLocator.h
//  Manuscript
//
//  Created by 二鏡 on 11/07/29.
//  Copyright 2011年 二鏡庵. All rights reserved.
//

#ifndef VTKit_VerticalGlyphLocator_h
#define VTKit_VerticalGlyphLocator_h

#include "_GlyphLocator.h"
#include "_CharacterAttributePredicate.h"
#include <Cocoa/Cocoa.h>

extern NSString *ParserNoEnoughSpaceException;

static void inline
raiseNoEnoughSpaceException()
{
    [[NSException exceptionWithName: ParserNoEnoughSpaceException
                             reason: @"There is no enough box space to layout a glyph."
                           userInfo: nil] raise];
}
namespace _BoxLayoutSupport 
{
    class _ChainState
    {
        CFRange range;
        CGFloat headPosition;
    public:
        _ChainState()
        {
            range = CFRangeMake(0,0);
            headPosition = 0.0;
        }
        
        inline void clear()
        {
            range = CFRangeMake(0,0);
            headPosition = 0.0;
        }
        
        inline void init(CFIndex loc, CGFloat startOffset)
        {
            range = CFRangeMake(loc,1);
            headPosition = startOffset;
        }
        
        inline bool isChaining()
        {
            return range.length != 0;
        }
        
        inline void cutChainToIndex(CFIndex i)
        {
            range.length = i-range.location+1;
        }
        
        inline _ChainState& operator++()
        {
            range.length++;
            return *this;
        }
        
        inline _ChainState& operator++(int)
        {
            range.length++;
            return *this;
        }
        
        inline CGFloat getHeadPosition()
        {
            return headPosition;
        }
        
        inline CFRange getRange()
        {
            return range;
        }
    };
    
    class _BoxState
    {
        // 現在の行の開始インデックス。
        CFIndex anchor;
        // 計算中のボックスの基準サイズ。
        CGFloat boxWidth;
        // 二通りの解釈による残りのフィールド状態。行単位で再ロードを行う。
        /*
         * discrete文脈は残りの完全に開いているboxの離散数
         * continuous文脈は残りの開いている空間連続量
         * プロポーショナル文字は連続文脈で処理するが、その後のboxは余り分を飛ばした数を取る
         */
        CGFloat continuousRemain;
        CFIndex discreteRemain;
        CGFloat topBoxOffset;
    public:
        _BoxState()
        {
            boxWidth = 0.0;
            continuousRemain = 0.0;
            discreteRemain = 0;
            topBoxOffset = 0.0;
        }
        inline CGFloat getBoxWidth() { return boxWidth; }
        inline CGFloat getTopBoxOffset() { return topBoxOffset; }
        inline CFIndex getAnchor() { return anchor; }
        inline void loadBoxState(CGFloat aBoxWidth, CFIndex count, CFIndex anchorLocation)
        {
            boxWidth = aBoxWidth;
            continuousRemain = boxWidth*count;
            discreteRemain = count;
            anchor = anchorLocation;
            topBoxOffset = 0.0;
        }
        inline bool isBoxRemaining() { return discreteRemain != 0; }
        inline bool canPutPropotionalGlyph(CGFloat width) { return continuousRemain >= width; }
        inline bool isAnchorLocation(CFIndex index) { return index == anchor; }
        
        // 1box分だけ消費して、連続文脈は残りのボックス分とする。
        inline void take1Box()
        {
            discreteRemain--;
            continuousRemain = boxWidth*discreteRemain;
            topBoxOffset -= boxWidth;
        }
        
        // 幅widthのグリフを連続文脈で詰めた時、連続文脈は残りそのままであり、
        // 離散文脈は完全に開いているboxの数を表すので残り領域をボックス幅で割って0向きに丸めればよい
        inline void take1Glyph(CGFloat width)
        {
            continuousRemain -= width;
            CFIndex nextDiscreteRemain = continuousRemain/boxWidth;
            topBoxOffset -= boxWidth*(discreteRemain-nextDiscreteRemain);
            discreteRemain = nextDiscreteRemain;
        }
    };
}


#pragma mark -

class VerticalGlyphLocator : public _GlyphLocatorBase, public CharacterAttributePredicate
{
    _BoxLayoutSupport::_ChainState chainState;
    _BoxLayoutSupport::_BoxState boxState;
    CFRange stabilizedRange; // 確定レンジ
    bool shouldHyphenation;
    bool shouldStabilizeCentering;
    
    uint32_t locateGlyphsPropotionallyByCentering(CFRange range, CGFloat x, CGFloat y);
    
    inline void advanceStabilizedRangeToIndex(CFIndex i)
    {
        CFIndex anchor = boxState.getAnchor();
        stabilizedRange = CFRangeMake(anchor, i-anchor+1);
    }
    
    inline void startChainAtIndex(CFIndex i)
    {
        chainState.init(i, boxState.getTopBoxOffset());
    }
    
    // chain中の半角文字を全て確定
    inline void stabilizeCurrentChain()
    {
        CFRange range = chainState.getRange();
        CGFloat datum = chainState.getHeadPosition();
        if(shouldStabilizeCentering)
            locateGlyphsPropotionallyByCentering(range,0,datum);
        else
            _locateGlyphsPropotionally(range,0,datum,kCTFontVerticalOrientation);    
        
        chainState.clear();
        advanceStabilizedRangeToIndex(range.location+range.length-1);    
    }
    
    // 全角文字一つを確定
    inline void stabilizeFullGlyph(CFIndex i)
    {
        CGFloat y = boxState.getTopBoxOffset();
        _locateSingleGlyph(i, 0, y);
        boxState.take1Box();
        advanceStabilizedRangeToIndex(i);
    }
    
    // 改行文字を確定
    inline void stabilizeNewline(CFIndex i)
    {
        const vector<CGPoint>& positions = positions_();
        const vector<CGSize>& advances = advances_();
        CFIndex anchor = boxState.getAnchor();
        if(i == anchor)
            _locateSingleGlyph(i,0,0);
        else
        {
            CGFloat lastPos = positions[i-1].y;
            CGFloat lastWidth = -advances[i-1].width;
            _locateSingleGlyph(i,0,lastPos+lastWidth);
        }
        advanceStabilizedRangeToIndex(i);
    }
public:
    
    VerticalGlyphLocator(_GlyphBufferSpace* buf) :_GlyphLocatorBase(buf)
    {
        shouldStabilizeCentering = true;
    }
    
    inline void stabilizeCurrentChainIfNeed()
    {
        if(chainState.isChaining())
        {
            stabilizeCurrentChain();
        }
    }
    
    inline CFRange getStabilizedRange()
    {
        return stabilizedRange;
    }
    
    inline void reloadLine(CGFloat boxWidth, CFIndex count, CFIndex anchor)
    {
        boxState.loadBoxState(boxWidth, count, anchor);
        chainState.clear();
        stabilizedRange = CFRangeMake(anchor,0);
    }
    
    bool analyzeGlyphsInRange(CFRange range);
    
    bool tryExtraLayoutIfNeed()
    {
        const vector<uint8>& characterAttribute = characterAttribute_();
        // 1. 確定済み基本行の次の文字が存在し
        CFIndex next = stabilizedRange.location+stabilizedRange.length;
        if(next >= characterAttribute.size())
            return false;
        
        // 2. 行頭禁則であるなら1文字取り出す
        uint8_t cAttr = characterAttribute[next];
        if(isHeadProhibition(cAttr))
        {
            stabilizeFullGlyph(next);
            // 3. さらに次の文字を調べ、
            next++;
            if(next < characterAttribute.size())
            {
                // 4. 改行文字であるならそこまで含む
                cAttr = characterAttribute[next];
                if(isStrongBreak(cAttr))
                    stabilizeFullGlyph(next);
            }
            return true;
        }
        else
        {
            return false;
        }
    }
    
    bool tryWrapToNextIfNeed()
    {
        const vector<uint8>& characterAttribute = characterAttribute_();
        
        CFIndex last = stabilizedRange.location + stabilizedRange.length-1;
        uint8_t cAttr = characterAttribute[last];
        
        if(isTailProhibition(cAttr))
        {
            advanceStabilizedRangeToIndex(last-1);
            return true;
        }
        else
        {
            return false;
        }
    }
    
    inline void finalizeStabilizedRange()
    {
        finalizeVerticalLine(stabilizedRange);
    }
    
    inline bool hasHyphenation() const
    {
        return shouldHyphenation;
    }
    
    inline bool shouldCenteringStabilize() const
    {
        return shouldStabilizeCentering;
    }
    
    inline void setShouldCenteringStabilize(bool val)
    {
        shouldStabilizeCentering = val;
    }
};

#endif
