//
//  MSTextView+Moving.m
//  Manuscript
//
//  Created by 二鏡 on 12/03/20.
//  Copyright (c) 2012年 二鏡庵. All rights reserved.
//


#import "MSTextView.h"
#import "MSTextView_Private.h"

inline static id
readObjectFromPasteboard(NSPasteboard *pboard, Class aClass)
{
    id array = [NSArray arrayWithObject: aClass];
    id objs = [pboard readObjectsForClasses: array options: nil];
    return [objs objectAtIndex: 0];
}

@implementation MSTextView (Action)
#pragma mark Internal
- (void)_moveToPreviousCharacter
{
    caretState.clear_height();
    if(inputState.is_selected_range_nonzero())
    {
        inputState.reset_selection_anchor();
        self.selectedRange = inputState.get_zero_range_from_selection_head();
        [self scrollToReferenceIndex: inputState.get_head_of_selection()];
        [self setNeedsDisplay: YES];
        return ;
    }
    
    if(inputState.is_selected_range_at_head())
    {
        [self scrollToReferenceIndex: 0];
        return ;
    }
    
    CFIndex prevIndex = inputState.selectedRange.location-1;
    self.selectedRange = NSMakeRange(prevIndex,0);
    inputState.reset_selection_anchor();
    [self scrollToReferenceIndex: prevIndex];
    [self setNeedsDisplay: YES]; 
}

- (void)_moveToNextCharacter
{
    caretState.clear_height();
    if(inputState.is_selected_range_nonzero())
    {
        inputState.reset_selection_anchor();
        self.selectedRange = inputState.get_zero_range_from_selection_tail();
        [self scrollToReferenceIndex: inputState.get_tail_of_selection()];
        [self setNeedsDisplay: YES];
        return ;
    }
    
    NSTextStorage *storage = pager_.content.textStorage;
    NSRange range = inputState.selectedRange;
    // 末端でなければ前進
    if([storage length] != inputState.selectedRange.location)
        range.location++;
    range.length = 0;
    
    self.selectedRange = range;
    inputState.reset_selection_anchor();
    [self scrollToReferenceIndex: range.location];
    [self setNeedsDisplay: YES];
}

- (void)_moveToPreviousLine
{
    [self _memoryCaretHeightIfNeed];
    NSRect rect = [self _referenceCaretRect];
    NSPoint lineOrigin = [pager_ lineOriginAtIndex: 0];
    
    CGFloat x = NSMidX(rect);
    CGFloat y = lineOrigin.y+caretState.get_height();
    NSUInteger index = [pager_ indexOfCharacterOfPreviousLineFromPoint: NSMakePoint(x,y)];
    
    if(index == NSNotFound)
        index = 0;
    
    inputState.reset_selection_anchor();
    self.selectedRange = NSMakeRange(index,0);
    
    if( caretState.is_extended() )
        caretState.set_rect([pager_ caretRectForCharacterIndexWithExtend: index]);
    else
        caretState.set_rect([pager_ caretRectForCharacterIndex: index]);
    
    [self scrollToReferenceIndex: index];
    [self setNeedsDisplay: YES];
}

- (void)_moveToNextLine
{
    [self _memoryCaretHeightIfNeed];
    NSRect rect = [self _referenceCaretRect];
    NSPoint lineOrigin = [pager_ lineOriginAtIndex: 0];
    
    CGFloat x = NSMidX(rect);
    CGFloat y = lineOrigin.y+caretState.get_height();
    NSUInteger index = [pager_ indexOfCharacterOfNextLineFromPoint: NSMakePoint(x,y)];
    
    if(index == NSNotFound)
        index = pager_.content.textStorage.length;
    
    inputState.reset_selection_anchor();
    self.selectedRange = NSMakeRange(index,0);
    
    if( caretState.is_extended() )
        caretState.set_rect([pager_ caretRectForCharacterIndexWithExtend: index]);
    else
        caretState.set_rect([pager_ caretRectForCharacterIndex: index]);
    
    [self scrollToReferenceIndex: index];
    [self setNeedsDisplay: YES];
}

- (void)_moveAndModifySelectionToPreviousLine
{
    [self _memoryCaretHeightIfNeed];
    NSRect rect = [self _referenceCaretRect];
    NSPoint lineOrigin = [pager_ lineOriginAtIndex: 0];
    
    CGFloat x = NSMidX(rect);
    CGFloat y = lineOrigin.y+caretState.get_height();
    NSUInteger index = [pager_ indexOfCharacterOfPreviousLineFromPoint: NSMakePoint(x,y)];
    
    if(index == NSNotFound)
        index = 0;
    
    // selection
    if(inputState.is_selected_range_zero())
    {
        inputState.set_anchor_to_head_of_selection();
        inputState.selectionLocator = index;
    }
    else
    {
        if(inputState.has_valid_selection_anchor() == false)
            inputState.set_anchor_to_tail_of_selection();
        inputState.selectionLocator = index;
    }
    self.selectedRange = inputState.get_anchored_range();
    
    [self scrollToReferenceIndex: index];
    [self setNeedsDisplay: YES];    
}

- (void)_moveAndModifySelectionToNextLine
{
    [self _memoryCaretHeightIfNeed];
    NSRect rect = [self _referenceCaretRect];
    NSPoint lineOrigin = [pager_ lineOriginAtIndex: 0];
    
    CGFloat x = NSMidX(rect);
    CGFloat y = lineOrigin.y+caretState.get_height();
    NSUInteger index = [pager_ indexOfCharacterOfNextLineFromPoint: NSMakePoint(x,y)];
    
    if(index == NSNotFound)
        index = pager_.content.textStorage.length;
    
    // selection
    if(inputState.is_selected_range_zero())
    {
        inputState.set_anchor_to_head_of_selection();
        inputState.selectionLocator = index;
    }
    else
    {
        if(inputState.has_valid_selection_anchor() == false)
            inputState.set_anchor_to_tail_of_selection();
        inputState.selectionLocator = index;
    }
    self.selectedRange = inputState.get_anchored_range();
    
    [self scrollToReferenceIndex: index];
    [self setNeedsDisplay: YES];    
}

- (void)_moveAndModifySelectionToPreviousCharacter
{
    caretState.clear_height();
    if(inputState.is_selected_range_zero())
    {
        if(inputState.is_selected_range_at_head())
        {
            [self scrollToReferenceIndex: 0];
            return ;
        }
        inputState.set_anchor_to_head_of_selection();
        inputState.selectionLocator = inputState.selectionAnchor-1;
        self.selectedRange = inputState.get_anchored_range();
    }
    else
    {
        if(inputState.has_valid_selection_anchor())
        {
            inputState.selectionLocator = MAX(inputState.selectionLocator-1, 0);
            self.selectedRange = inputState.get_anchored_range();
        }
        else
        {
            if(inputState.is_selected_range_at_head() == false)
            {
                inputState.set_anchor_to_tail_of_selection();
                inputState.selectionLocator = inputState.selectedRange.location -1;
                self.selectedRange = inputState.get_anchored_range();
            }
        }
    }
    
    [self scrollToReferenceIndex: inputState.selectionLocator];
    [self setNeedsDisplay: YES];
}

- (void)_moveAndModifySelectionToNextCharacter
{
    caretState.clear_height();
    NSTextStorage *storage = pager_.content.textStorage;
    CFIndex limit = [storage length];
    if(inputState.is_selected_range_zero())
    {
        if(inputState.selectedRange.location == limit)
        {
            if(limit != 0)
                [self scrollToReferenceIndex: limit-1];
            return;
        }
        inputState.set_anchor_to_head_of_selection();
        inputState.selectionLocator = inputState.selectionAnchor+1;
        self.selectedRange = inputState.get_anchored_range();
    }
    else
    {
        if(inputState.has_valid_selection_anchor())
        {
            inputState.selectionLocator = MIN(inputState.selectionLocator+1, limit);
            self.selectedRange = inputState.get_anchored_range();
        }
        else
        {
            if(NSMaxRange(inputState.selectedRange) != limit)
            {
                inputState.set_anchor_to_head_of_selection();
                inputState.selectionLocator = NSMaxRange(inputState.selectedRange)+1;
                self.selectedRange = inputState.get_anchored_range();
            }
        }
    }
    
    [self scrollToReferenceIndex: inputState.selectionLocator];
    [self setNeedsDisplay: YES];
}

#pragma mark Moving
- (void)moveUp:(id)sender
{
    [self _moveToPreviousLine];
}

- (void)moveDown:(id)sender
{
    [self _moveToNextLine];
}

- (void)moveRight:(id)sender
{
    [self _moveToNextCharacter];
}

- (void)moveLeft:(id)sender
{
    [self _moveToPreviousCharacter];
}

- (void)moveUpAndModifySelection:(id) sender
{
    [self _moveAndModifySelectionToPreviousLine];
}

- (void)moveDownAndModifySelection:(id) sender
{
    [self _moveAndModifySelectionToNextLine];
}

- (void)moveLeftAndModifySelection:(id)sender
{
    [self _moveAndModifySelectionToPreviousCharacter];
}

- (void)moveRightAndModifySelection:(id)sender
{
    [self _moveAndModifySelectionToNextCharacter];
}

- (void)moveToEndOfLineAndModifySelection:(id)sender
{
    if( caretState.is_extended() )
        return; // なにもしなくていい
    
    CFIndex cIndex;
    if(inputState.is_selected_range_nonzero())
        cIndex = inputState.selectionLocator;
    else
        cIndex = inputState.get_head_of_selection();
    CFIndex endOfLine = [pager_ endOfLineAtCharacterIndex: cIndex];
    
    // 移動しなかった時は何もしないが、表示だけは行う
    if( endOfLine == cIndex )
    {
        [self scrollToReferenceIndex: endOfLine];
        return ;
    }
    
    if(inputState.is_selected_range_zero())
    {
        inputState.set_anchor_to_head_of_selection();
        inputState.selectionLocator = endOfLine;
    }
    else
    {
        if(inputState.has_valid_selection_anchor() == false)
            inputState.set_anchor_to_head_of_selection();
        inputState.selectionLocator = endOfLine;
    }
    
    self.selectedRange = inputState.get_anchored_range();
    
    // caret設定
    caretState.extend();
    caretState.clear_height();
    
    // 表示
    [self scrollToReferenceIndex: endOfLine-1];
    [self setNeedsDisplay: YES];
    [self setNeedsDisplay: YES];    
}

- (void)moveToEndOfLine:(id)sender
{
    if( caretState.is_extended() )
        return; // なにもしなくていい
    
    CFIndex cIndex;
    if(inputState.is_selected_range_nonzero())
        cIndex = inputState.selectionLocator;
    else
        cIndex = inputState.get_head_of_selection();
    CFIndex endOfLine = [pager_ endOfLineAtCharacterIndex: cIndex];
    
    // 移動しなかった時は何もしないが、表示だけは行う
    if( endOfLine == cIndex )
    {
        [self scrollToReferenceIndex: endOfLine];
        return ;
    }
    
    inputState.reset_selection_anchor();
    self.selectedRange = NSMakeRange(endOfLine,0);
    
    // caret設定
    caretState.set_rect([pager_ caretRectForCharacterIndexWithExtend: endOfLine]);
    caretState.extend();
    caretState.clear_height();
    
    // 表示
    [self scrollToReferenceIndex: endOfLine-1];
    [self setNeedsDisplay: YES];
}

- (void)moveToBeginningOfLine:(id)sender
{
    CFIndex cIndex;
    if(inputState.is_selected_range_nonzero())
        cIndex = inputState.selectionLocator;
    else
        cIndex = inputState.get_head_of_selection();
    
    CFIndex headOfLine;
    if( caretState.is_extended() )
    {   // バック可能な時しかextendはたたない
        cIndex--;
    }
    
    headOfLine = [pager_ beginningOfLineAtCharacterIndex: cIndex];
    if( cIndex == headOfLine )
    {
        [self scrollToReferenceIndex: headOfLine];
        return;
    }
    
    inputState.reset_selection_anchor();
    self.selectedRange = NSMakeRange(headOfLine,0);
    
    // caret設定
    caretState.cancel_extend();
    caretState.set_rect([pager_ caretRectForCharacterIndex: headOfLine]);
    caretState.clear_height();
    
    // 表示
    [self scrollToReferenceIndex: headOfLine];
    [self setNeedsDisplay: YES];
}

- (void)moveToBeginningOfLineAndModifySelection:(id)sender
{
    CFIndex cIndex;
    if(inputState.is_selected_range_nonzero())
        cIndex = inputState.selectionLocator;
    else
        cIndex = inputState.get_head_of_selection();
    
    CFIndex headOfLine;
    if( caretState.is_extended() )
    {   // バック可能な時しかextendはたたない
        cIndex--;
    }
    
    headOfLine = [pager_ beginningOfLineAtCharacterIndex: cIndex];
    if( cIndex == headOfLine )
    {
        [self scrollToReferenceIndex: headOfLine];
        return;
    }
    
    // selection
    if(inputState.is_selected_range_zero())
    {
        inputState.set_anchor_to_head_of_selection();
        inputState.selectionLocator = headOfLine;
    }
    else
    {
        if(inputState.has_valid_selection_anchor() == false)
            inputState.set_anchor_to_tail_of_selection();
        inputState.selectionLocator = headOfLine;
    }
    self.selectedRange = inputState.get_anchored_range();
    
    // caret設定
    caretState.cancel_extend();
    caretState.clear_height();
    
    // 表示
    [self scrollToReferenceIndex: headOfLine];
    [self setNeedsDisplay: YES];
}

- (void)moveWordForward:(id)sender
{
    NSTextStorage *storage = pager_.content.textStorage;
    NSUInteger index = [storage nextWordFromIndex: NSMaxRange(inputState.selectedRange) 
                                          forward: YES];
    inputState.reset_selection_anchor();
    self.selectedRange = NSMakeRange(index, 0);
    [self scrollToReferenceIndex: index];
    [self setNeedsDisplay: YES];
}

- (void)moveWordForwardAndModifySelection:(id)sender
{
    NSTextStorage *storage = pager_.content.textStorage;
    if(inputState.is_selected_range_zero())
    {
        inputState.set_anchor_to_head_of_selection();
        inputState.selectionLocator = [storage nextWordFromIndex: inputState.selectionAnchor
                                                         forward: YES];
    }
    else
    {
        if(inputState.has_valid_selection_anchor())
        {
            inputState.set_anchor_to_head_of_selection();
            inputState.selectionLocator = [storage nextWordFromIndex: inputState.get_tail_of_selection()
                                                             forward: YES];
        }
        else
        {
            inputState.selectionLocator = [storage nextWordFromIndex: inputState.selectionLocator
                                                             forward: YES];
        }
    }
    self.selectedRange = inputState.get_anchored_range();
    [self scrollToReferenceIndex: inputState.selectionLocator];
    [self setNeedsDisplay: YES];
}

- (void)moveWordBackward:(id)sender
{
    NSTextStorage *storage = pager_.content.textStorage;
    NSUInteger index = [storage nextWordFromIndex: NSMaxRange(inputState.selectedRange) forward: NO];
    inputState.reset_selection_anchor();
    self.selectedRange = NSMakeRange(index, 0);
    [self scrollToReferenceIndex: index];
    [self setNeedsDisplay: YES];
}

- (void)moveWordBackwardAndModifySelection:(id)sender
{
    NSTextStorage *storage = pager_.content.textStorage;
    if(inputState.is_selected_range_zero())
    {
        inputState.set_anchor_to_head_of_selection();
        inputState.selectionLocator = [storage nextWordFromIndex: inputState.selectionAnchor
                                                         forward: NO];
    }
    else
    {
        if(inputState.has_valid_selection_anchor() == false)
        {
            inputState.set_anchor_to_tail_of_selection();
            inputState.selectionLocator = [storage nextWordFromIndex: inputState.get_head_of_selection()
                                                             forward: NO];
        }
        else
        {
            inputState.selectionLocator = [storage nextWordFromIndex: inputState.selectionLocator
                                                             forward: NO];
        }
    }
    self.selectedRange = inputState.get_anchored_range();
    [self scrollToReferenceIndex: inputState.selectionLocator];
    [self setNeedsDisplay: YES];
}

- (void)moveToBeginningOfParagraph:(id)sender
{
    id str = [pager_.content.textStorage mutableString];
    NSUInteger start;
    [str getParagraphStart: &start
                       end: NULL
               contentsEnd: NULL
                  forRange: inputState.selectedRange];
    inputState.reset_selection_anchor();
    self.selectedRange = NSMakeRange(start, 0);
    [self scrollToReferenceIndex: start];
    [self setNeedsDisplay: YES];
}

- (void)moveToBeginningOfParagraphAndModifySelection:(id)sender
{
    id str = [pager_.content.textStorage mutableString];
    NSUInteger start;
    [str getParagraphStart: &start
                       end: NULL
               contentsEnd: NULL
                  forRange: inputState.selectedRange];
    
    if(inputState.is_selected_range_zero())
    {
        inputState.set_anchor_to_head_of_selection();
        inputState.selectionLocator = start;
    }
    else
    {
        if(inputState.has_valid_selection_anchor() == false)
            inputState.set_anchor_to_tail_of_selection();
        inputState.selectionLocator = start;
    }
    self.selectedRange = inputState.get_anchored_range();
    [self scrollToReferenceIndex: start];
    [self setNeedsDisplay: YES];
}

- (void)moveToEndOfParagraph:(id)sender
{
    id str = [pager_.content.textStorage mutableString];
    NSUInteger end;
    [str getParagraphStart: NULL
                       end: NULL
               contentsEnd: &end
                  forRange: inputState.selectedRange];
    inputState.reset_selection_anchor();
    self.selectedRange = NSMakeRange(end, 0);
    [self scrollToReferenceIndex: end];
    [self setNeedsDisplay: YES];
}

- (void)moveToEndOfParagraphAndModifySelection:(id)sender
{
    id str = [pager_.content.textStorage mutableString];
    NSUInteger end;
    [str getParagraphStart: NULL
                       end: NULL
               contentsEnd: &end
                  forRange: inputState.selectedRange];
    
    if(inputState.is_selected_range_zero())
    {
        inputState.set_anchor_to_head_of_selection();
        inputState.selectionLocator = end;
    }
    else
    {
        if(inputState.has_valid_selection_anchor() == false)
            inputState.set_anchor_to_head_of_selection();
        inputState.selectionLocator = end;
    }
    self.selectedRange = inputState.get_anchored_range();
    [self scrollToReferenceIndex: end];
    [self setNeedsDisplay: YES];
}

- (void)moveToBeginningOfDocument:(id)sender
{
    self.selectedRange = NSMakeRange(0, 0);
    inputState.reset_selection_anchor();
    [self scrollToReferenceIndex: 0];
    [self setNeedsDisplay: YES];
}

- (void)moveToBeginningOfDocumentAndModifySelection:(id)sender
{
    if(inputState.is_selected_range_zero())
    {
        inputState.set_anchor_to_head_of_selection();
        inputState.selectionLocator = 0;
    }
    else
    {
        if(inputState.has_valid_selection_anchor() == false)
            inputState.set_anchor_to_tail_of_selection();
        inputState.selectionLocator = 0;
    }
    self.selectedRange = inputState.get_anchored_range();
    [self scrollToReferenceIndex: 0];
    [self setNeedsDisplay: YES];
}

- (void)moveToEndOfDocumentAndModifySelection:(id)sender
{
    NSUInteger tail = pager_.content.textStorage.length;
    if(inputState.is_selected_range_zero())
    {
        inputState.set_anchor_to_head_of_selection();
        inputState.selectionLocator = tail;
    }
    else
    {
        if(inputState.has_valid_selection_anchor() == false)
            inputState.set_anchor_to_head_of_selection();
        inputState.selectionLocator = tail;
    }
    self.selectedRange = inputState.get_anchored_range();
    [self scrollToReferenceIndex: tail];
    [self setNeedsDisplay: YES];    
    
}

- (void)moveToEndOfDocument:(id)sender
{
    NSUInteger tail = pager_.content.textStorage.length;
    self.selectedRange = NSMakeRange(tail,0);
    inputState.reset_selection_anchor();
    [self scrollToReferenceIndex: tail];
    [self setNeedsDisplay: YES];    
}

#pragma mark Insert
- (void)insertNewline:(id)sender
{
    id attr = [NSDictionary dictionaryWithObject: pager_.format.font
                                          forKey: NSFontAttributeName];
    id aStr = [[NSAttributedString alloc] initWithString: @"\n"
                                              attributes: attr];
    
    [self insertText: aStr
    replacementRange: inputState.selectedRange];
    [self scrollToReferenceIndex: NSMaxRange(inputState.selectedRange)];
}

#pragma mark Select
- (void)selectAll:(id)sender
{
    inputState.reset_selection_anchor();
    self.selectedRange = NSMakeRange(0, pager_.content.textStorage.length);
    [self setNeedsDisplay: YES];
}

- (void)deselect:(id)sender
{
    NSUInteger index;
    if(inputState.has_valid_selection_anchor())
        index = inputState.selectedRange.location;
    else
        index = inputState.selectionLocator;
    self.selectedRange = NSMakeRange(index, 0);
    [self scrollToReferenceIndex: index];
    inputState.reset_selection_anchor();
    [self setNeedsDisplay: YES];    
}

#pragma mark Cut/Copy/Paste/Delete
- (void)cut:(id)sender
{
    if(inputState.is_selected_range_zero())
        return ;
    
    NSPasteboard *pboard = [NSPasteboard generalPasteboard];
    [self _copySelectionTo: pboard];
    
    [self deleteBackward: sender];
}

- (void)copy:(id)sender
{
    if(inputState.is_selected_range_zero())
        return ;
    
    NSPasteboard *pboard = [NSPasteboard generalPasteboard];
    [self _copySelectionTo: pboard];
}

- (void)paste:(id)sender
{
    id pboard = [NSPasteboard generalPasteboard];
    id obj = readObjectFromPasteboard(pboard, [NSAttributedString class]);
    if(obj)
    {
        [self pasteAttributedString: obj];
        return ;
    }
    
    obj = readObjectFromPasteboard(pboard, [NSString class]);
    if(obj)
    {
        [self pasteAttributedString: obj];
        return ;
    }
}

- (void)delete:(id)sender
{
    [self deleteCharactersInRange: inputState.selectedRange];
}

- (void)deleteBackward:(id)sender {
    // Find the range to delete, handling an empty selection and the input point being at 0
    NSRange deleteRange = inputState.selectedRange;
    if (deleteRange.length == 0) {
        if (deleteRange.location == 0) {
            return;
        } else {
            deleteRange.location -= 1;
            deleteRange.length = 1;
            
            // Make sure we handle composed characters correctly
            deleteRange = [[pager_.content.textStorage string] rangeOfComposedCharacterSequencesForRange:deleteRange];
        }
    }
    
    [self deleteCharactersInRange:deleteRange];
}

- (void)deleteForward:(id)sender {
    // Find the range to delete, handling an empty selection and the input point being at the end
    NSTextStorage *storage = pager_.content.textStorage;
    NSRange deleteRange = inputState.selectedRange;
    if (deleteRange.length == 0) {
        if (deleteRange.location == [storage length]) {
            return;
        } else {
            deleteRange.length = 1;
            
            // Make sure we handle composed characters correctly
            deleteRange = [[storage string] rangeOfComposedCharacterSequencesForRange:deleteRange];
        }
    }
    
    [self deleteCharactersInRange:deleteRange];
}

#pragma mark -
- (BOOL)validateUserInterfaceItem:(id < NSValidatedUserInterfaceItem >)anItem
{
    if(self.lock)
        return NO;
    
    if([self hasMarkedText])
        return NO;

    SEL action = [anItem action];
    if(action == @selector(copy:) ||
       action == @selector(delete:) ||
       action == @selector(cut:) ||
       action == @selector(deselect:))
        return (inputState.is_selected_range_nonzero());
    
    if(action == @selector(selectAll:))
        return YES;
    
    if(action == @selector(paste:))
    {
        id pboard = [NSPasteboard generalPasteboard];
        return [pboard canReadObjectForClasses: [self availablePboardClasses]
                                       options: nil];
    }
    
    if(action == @selector(undo:))
        return [[delegate_ undoManager] canUndo];
    
    if(action == @selector(redo:))
        return [[delegate_ undoManager] canRedo];
    
    return YES;
}

@end


@implementation MSTextView (Dragging)
- (BOOL)_isCharacterOnSelection:(NSUInteger)index
{
    // 境界も含む
    return (NSLocationInRange(index, inputState.selectedRange) || 
            index == NSMaxRange(inputState.selectedRange));
}

- (BOOL)_acceptDropCopy:(id <NSDraggingInfo>)sender
{
    id pboard = [sender draggingPasteboard];
    // caretLocatorに直接代入
    id obj = readObjectFromPasteboard(pboard, [NSAttributedString class]);
    if(obj)
    {
        
        id aStr = delegate_ == nil ? obj : [delegate_ normalizeAttributedString: obj];
        [self insertText: aStr
        replacementRange: NSMakeRange(caretState.locator, 0)];
        return YES;
    }
    
    obj = readObjectFromPasteboard(pboard, [NSString class]);
    if(obj)
    {
        [self insertText: obj
        replacementRange: NSMakeRange(caretState.locator, 0)];
    }
    
    return YES;
}

- (NSDragOperation)draggingSourceOperationMaskForLocal:(BOOL) isLocal
{
    return NSDragOperationCopy | NSDragOperationMove | NSDragOperationGeneric;
}

- (BOOL)ignoreModifierKeysForDraggingSession:(NSDraggingSession *)session
{
    return NO;
}

- (BOOL)wantsPeriodicDraggingUpdates
{
    return NO;
}

- (NSDragOperation)draggingEntered:(id < NSDraggingInfo >) sender
{
    if(self.lock)
        return NSDragOperationNone;
    
    _dropState.dragging = YES;

    BOOL isTail;
//    NSPoint point = [self convertPointFromBase: [sender draggingLocation]];
    NSPoint point = [self convertPoint: [sender draggingLocation]
                              fromView: nil];
    caretState.locator = [pager_ indexOfCharacterOfNearestBorderAtPoint: point
                                                                 isTail: &isTail];

    if(caretState.locator != NSNotFound)
    {
        if([sender draggingSource] == self && [self _isCharacterOnSelection:caretState.locator])
        {
            caretState.invalide_rect();
        }
        else
        {
            // キャレットを設定
            if(isTail)
                caretState.set_rect([pager_ caretRectForCharacterIndexWithExtend: caretState.locator]);
            else
                caretState.set_rect([pager_ caretRectForCharacterIndex: caretState.locator]);
        }
    }
    [self setNeedsDisplay: YES];
    
    if(caretState.locator == NSNotFound)
        return NSDragOperationNone;
    
    NSDragOperation mask = [sender draggingSourceOperationMask];
    if(mask & NSDragOperationMove)
        return NSDragOperationMove;
    else
        return NSDragOperationCopy;
}

- (NSDragOperation)draggingUpdated:(id < NSDraggingInfo >) sender
{
    if(self.lock)
        return NSDragOperationNone;

    BOOL isTail;
    NSPoint point = [self convertPoint: [sender draggingLocation]
                              fromView: nil];
    caretState.locator = [pager_ indexOfCharacterOfNearestBorderAtPoint: point
                                                                 isTail: &isTail];
    if(caretState.locator != NSNotFound)
    {
        if([sender draggingSource] == self && [self _isCharacterOnSelection:caretState.locator])
        {
            caretState.invalide_rect();
        }
        else
        {
            // キャレットを設定
            if(isTail)
                caretState.set_rect([pager_ caretRectForCharacterIndexWithExtend: caretState.locator]);
            else
                caretState.set_rect([pager_ caretRectForCharacterIndex: caretState.locator]);
        }
    }

    [self setNeedsDisplay: YES];
    
    if(caretState.locator == NSNotFound)
        return NSDragOperationNone;
    
    NSDragOperation mask = [sender draggingSourceOperationMask];
    if(mask & NSDragOperationMove)
        return NSDragOperationMove;
    else
        return NSDragOperationCopy;
}

- (void)draggingExited:(id < NSDraggingInfo >) sender
{
    _dropState.dragging = NO;
}

- (void)draggingEnded:(id <NSDraggingInfo>) sender
{
    caretState.locator = inputState.selectedRange.location;
    [self updateCaretRect];
    [self setNeedsDisplay: YES];
}

- (BOOL)prepareForDragOperation:(id < NSDraggingInfo >) sender
{
    _dropState.dragging = NO;
    if(caretState.locator == NSNotFound)
        return NO;
    if([sender draggingSource] == self)
    {
        // 内部ドラッグの場合、selectionに移動することはできない
        if([self _isCharacterOnSelection:caretState.locator])
            return NO;
    }
    return YES;
}

- (BOOL)performDragOperation:(id < NSDraggingInfo >) sender
{
    if([sender draggingSource] == self && 
       [sender draggingSourceOperationMask] & NSDragOperationMove)
    {
        [self _moveTextInRange: inputState.selectedRange 
                            to: caretState.locator];
        return YES;
    }
    if([sender draggingSourceOperationMask] & NSDragOperationCopy)
        return [self _acceptDropCopy: sender];
    return NO;
}

- (void)concludeDragOperation:(id < NSDraggingInfo >) sender
{
}
@end
