/*
 * The MIT License
 *
 * Copyright 2013 Masahiko, SAWAI <masahiko.sawai@gmail.com>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package com.example.hello.android.textview_find_word_in_actionmode;

import android.app.Activity;
import android.content.Context;
import android.graphics.Rect;
import android.os.Bundle;
import android.text.Editable;
import android.text.TextWatcher;
import android.util.Log;
import android.view.ActionMode;
import android.view.KeyEvent;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.inputmethod.EditorInfo;
import android.view.inputmethod.InputMethodManager;
import android.widget.EditText;
import android.widget.ScrollView;
import android.widget.TextView;

public class MainActivity extends Activity
	implements View.OnLongClickListener, TextWatcher, TextView.OnEditorActionListener
{

	private static final String LOG_TAG = "XXX";
	private final SearchActionModeCallback searchActionModeCallback = new SearchActionModeCallback();
	private ActionMode actionMode = null;
	private final TextViewSearchContext searchContext = new TextViewSearchContext();
	private TextView helloTextView;
	private ScrollView helloScrollView;

	@Override
	public void onCreate(Bundle savedInstanceState)
	{
		Log.v(LOG_TAG, "onCreate() : Hello");

		super.onCreate(savedInstanceState);
		setContentView(R.layout.main_activity);

		helloTextView = (TextView) findViewById(R.id.hello_textview);
		helloTextView.setOnLongClickListener(this);

		helloScrollView = (ScrollView) findViewById(R.id.hello_scrollview);

		setText(helloTextView.getText());

		Log.v(LOG_TAG, "onCreate() : Bye");
	}

	@Override
	public boolean onCreateOptionsMenu(Menu menu)
	{
		MenuInflater menuInflater = getMenuInflater();
		menuInflater.inflate(R.menu.quit_menu, menu);

		return true;
	}

	@Override
	public boolean onOptionsItemSelected(MenuItem item)
	{
		boolean result;
		int itemId = item.getItemId();
		switch (itemId)
		{
			case R.id.quit_menuitem:
				finish();
				result = true;
				break;
			default:
				result = super.onOptionsItemSelected(item);
		}
		return result;
	}

	@Override
	public boolean onLongClick(View view)
	{
		boolean result = false;
		Log.v(LOG_TAG, "onLongClick() : Hello");

		if (actionMode == null)
		{
			actionMode = startActionMode(searchActionModeCallback);
			result = true;
		}

		Log.v(LOG_TAG, "onLongClick() : Bye");
		return result;
	}

	@Override
	public void beforeTextChanged(CharSequence cs, int i, int i1, int i2)
	{
		Log.v(LOG_TAG, "beforeTextChanged() : Hello");
		Log.v(LOG_TAG, "beforeTextChanged() : Bye");
	}

	@Override
	public void onTextChanged(CharSequence cs, int i, int i1, int i2)
	{
		Log.v(LOG_TAG, "onTextChanged() : Hello");
		Log.v(LOG_TAG, "onTextChanged() : i => " + i + ", i1 => " + i1 + ", i2 => " + i2);
		setSearchWord(cs);
		Log.v(LOG_TAG, "onTextChanged() : Bye");
	}

	@Override
	public void afterTextChanged(Editable edtbl)
	{
		Log.v(LOG_TAG, "afterTextChanged() : Hello");
		Log.v(LOG_TAG, "afterTextChanged() : Bye");
	}

	private void setText(CharSequence text)
	{
		this.searchContext.setContentText(text);
		updateView();
	}

	private void setSearchWord(CharSequence searchWord)
	{
		this.searchContext.setSearchWord(searchWord);
		updateView();
	}

	private void findPrevWord()
	{
		searchContext.prevWord();
		updateView();
	}

	private void findNextWord()
	{
		searchContext.nextWord();
		updateView();
		this.helloTextView.setText(this.searchContext.getSpannable());
	}

	private void updateView()
	{
		this.helloTextView.setText(this.searchContext.getSpannable());

		int selectedWordLineNumber = this.searchContext.getSelectedWordLineNumber();
		Log.d(LOG_TAG, "selectedWordLineNumber => " + selectedWordLineNumber);
		if (selectedWordLineNumber >= 0)
		{
			Rect rect = new Rect();
			helloTextView.getLineBounds(selectedWordLineNumber, rect);
			helloScrollView.scrollTo(rect.left, rect.top - helloTextView.getLineHeight());
		}

	}

	public boolean onEditorAction(TextView textView, int actionId, KeyEvent keyEvent)
	{
		if (actionId == EditorInfo.IME_ACTION_SEARCH)
		{
			setSearchWord(textView.getText());
			closeIMEWindow(textView);
		}
		return true;
	}

	private void openIMEWindow(View view)
	{
		InputMethodManager inputMethodManager = (InputMethodManager) getSystemService(Context.INPUT_METHOD_SERVICE);
		inputMethodManager.showSoftInput(view, InputMethodManager.SHOW_IMPLICIT);
	}

	private void closeIMEWindow(View view)
	{
		InputMethodManager inputMethodManager = (InputMethodManager) getSystemService(Context.INPUT_METHOD_SERVICE);
		inputMethodManager.hideSoftInputFromWindow(view.getWindowToken(), 0);
	}

	private class SearchActionModeCallback implements ActionMode.Callback
	{

		private EditText findWordEditText;

		/**
		 * このアクションモードに入った際に呼ばれるコールバック。
		 *
		 * ここでアクションアイテムの作成やタイトルの設定などを行う。 ここで作成したメニューアイテムはこのアクションモード専用のものとなる。
		 *
		 * @param mode
		 * @param menu
		 * @return
		 */
		public boolean onCreateActionMode(ActionMode mode, Menu menu)
		{
			Log.v(LOG_TAG, "onCreateActionMode() : Hello");

			MenuInflater menuInflater = getMenuInflater();
			menuInflater.inflate(R.menu.find_word_option_menu, menu);

			mode.setTitle(R.string.action_mode_title);
//			mode.setSubtitle(R.string.action_mode_sub_title);

			Log.v(LOG_TAG, "onCreateActionMode() : Bye");
			return true;
		}

		/**
		 * このアクションモードになる直前に呼ばれるコールバック。
		 *
		 * onCreateActionMode() より後でり、アクションアイテムなどの 各種ビューが初期化完了している事が期待できる。
		 *
		 * @param mode
		 * @param menu
		 * @return mode や menu が更新されたら true を返す
		 */
		public boolean onPrepareActionMode(ActionMode mode, Menu menu)
		{
			Log.v(LOG_TAG, "onPrepareActionMode() : Hello");

			MenuItem findItem = menu.findItem(R.id.find_word_menuitem);
			View actionView = findItem.getActionView();

			findWordEditText = (EditText) actionView.findViewById(R.id.find_word_edittext);
			Log.v(LOG_TAG, "onPrepareActionMode() : findWordEditText = " + findWordEditText);
			findWordEditText.setOnEditorActionListener(MainActivity.this);
//			findWordEditText.addTextChangedListener(MainActivity.this);

			Log.v(LOG_TAG, "onPrepareActionMode() : Bye");
			return false;
		}

		/**
		 * アクションアイテムがクリックされた際のコールバック。
		 *
		 * @param mode
		 * @param item
		 * @return
		 */
		public boolean onActionItemClicked(ActionMode mode, MenuItem item)
		{
			Log.v(LOG_TAG, "onActionItemClicked() : Hello");
			int itemId = item.getItemId();
			switch (itemId)
			{
				case R.id.prev_word_menuitem:
					Log.v(LOG_TAG, "onActionItemClicked() : prev_word_menuitem is clicked.");
					findPrevWord();
					break;
				case R.id.next_word_menuitem:
					Log.v(LOG_TAG, "onActionItemClicked() : next_word_menuitem is clicked.");
					findNextWord();
					break;
			}
			Log.v(LOG_TAG, "onActionItemClicked() : Bye");
			return true;
		}

		/**
		 * アクションモード終了時のコールバック。
		 *
		 * ActionMode#finish() を呼び出したり、左上の「×」をクリックして アクションモードが終了したときに呼ばれる。
		 *
		 * @param mode
		 */
		public void onDestroyActionMode(ActionMode mode)
		{
			Log.v(LOG_TAG, "onDestroyActionMode() : Hello");
			setSearchWord("");
			closeIMEWindow(findWordEditText);
			actionMode = null;
			Log.v(LOG_TAG, "onDestroyActionMode() : Bye");
		}

	}

}
