package kisscelltopng.kiss.io;

import java.io.BufferedInputStream;
import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;

import kisscelltopng.kiss.types.KiSSPalette;

/**
 * KiSSパレットファイル(*.kcf)を読み込むためのローダー.<br>
 * 仕様は、 http://www2s.biglobe.ne.jp/~yav/kiss/kissgsj.html を参照した。<br>
 * @author seraphy
 */
public class KiSSPaletteLoader {

	/**
	 * 入力ストリームを指定してパレットファイルを読み込みます.<br>
	 * @param is 入力ストリーム
	 * @return KiSSパレット情報
	 * @throws IOException 読み込みに失敗した場合
	 */
	public KiSSPalette load(InputStream is) throws IOException {
		if (is == null) {
			throw new IllegalArgumentException();
		}
		
		DataInputStream dis = new DataInputStream(new BufferedInputStream(is));

		// シグネチャのリード
		byte[] lead = new byte[4];
		dis.read(lead);
		
		int bits;
		int numOfColor;
		int numOfPalette;
		
		if ( !Arrays.equals(lead, "KiSS".getBytes("UTF-8"))) {
			// KiSSでない場合は旧形式
			bits = 12;
			numOfColor = 16;
			numOfPalette = 10;
			
		} else {
			byte mark = dis.readByte();
			if (mark != 0x10) {
				System.err.println("not a palette file.");
			}
			bits = dis.readByte();
			dis.skipBytes(2);
			
			byte[] buf = new byte[16];
			ByteBuffer bbuf = ByteBuffer.wrap(buf);
			bbuf.putShort(dis.readShort());
			bbuf.putShort(dis.readShort());
			bbuf.rewind();
			bbuf.order(ByteOrder.LITTLE_ENDIAN);

			numOfColor = bbuf.getShort();
			numOfPalette = bbuf.getShort();
			
			dis.skipBytes(2);
			dis.skipBytes(2);
			dis.skipBytes(16);
			
			lead = null;
		}
		
		KiSSPalette palette = new KiSSPalette(bits, numOfColor, numOfPalette);
		
		int siz;
		if (bits == 12) {
			siz = 2 * numOfColor;

		} else if (bits == 24) {
			siz = 3 * numOfColor;
		
		} else {
			throw new KiSSDataException("unsupported bits: " + bits);
		}
		
		byte[] buf = new byte[siz];
		for (int page = 0; page < numOfPalette; page++) {
			int len;
			if (lead != null) {
				System.arraycopy(lead, 0, buf, 0, lead.length);
				len = dis.read(buf, lead.length, siz - lead.length);
				len += lead.length;
				lead = null;

			} else {
				len = dis.read(buf);
			}
			if (len != siz) {
				throw new KiSSDataException("invalid palette size");
			}
			
			int[] pl = palette.getPaletteGroup(page);
			
			ByteBuffer bbuf = ByteBuffer.wrap(buf);
			if (bits == 24) {
				for (int color = 0; color < numOfColor; color++) {
					int r = bbuf.get();
					int g = bbuf.get();
					int b = bbuf.get();
					
					int rgb = (r & 0xff) << 16 | (g & 0xff) << 8 | (b & 0xff);
					pl[color] = rgb;
				}

			} else {
				for (int color = 0; color < numOfColor; color++) {
					byte l = bbuf.get();
					byte h = bbuf.get();
					
					int r = l & 0xf0;
					int b = (l & 0x0f) << 4;
					int g = (h & 0x0f) << 4;
					
					int rgb = r << 16 | g << 8 | b;
					pl[color] = rgb;
				}
			}
		}

		return palette;
	}
	
}
