/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package textkeymatcher.ui.model;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.IOException;
import java.util.Map;
import org.apache.commons.lang3.StringUtils;
import textkeymatcher.entity.DisplayMode;
import textkeymatcher.entity.FilteredRowMap;
import textkeymatcher.entity.KeyMatchedRowMap;
import textkeymatcher.entity.KeyMatchedRowView;
import textkeymatcher.entity.LineDataList;
import textkeymatcher.entity.RowKey;
import textkeymatcher.entity.RowValues;
import textkeymatcher.io.DocArchive;
import textkeymatcher.io.TextKeyMatcherDoc;
import textkeymatcher.entity.KeyMatcher;

/**
 * ビューに表示されるテーブルのモデル.<br>
 * @author seraphy
 */
public class DataViewTableModel extends KeyMatchedRowView implements TextKeyMatcherDoc {
    
    /**
     * プロパティ変更サポート
     */
    private PropertyChangeSupport propCng = new PropertyChangeSupport(this);
    
    /**
     * キーカラムのカラム名.
     */
    public static final String COLUMN_NAME_KEY = "key";
    
    /**
     * キーマッチング方法のプロパティ名
     */
    public static final String PROPERTY_KEY_MATCHER = "keyMatcher";
    
    /**
     * 表示モード(行フィルタ方法)のプロパティ名
     */
    public static final String PROPERTY_DISPLAY_MODE = "displayMode";

    /**
     * データ変更フラグ.<br>
     */
    public static final String PROPERTY_DIRTY = "dirty";

    /**
     * 取り込んだデータソースや、キーマッチングなどを決定している
     * データマップ
     */
    private KeyMatchedRowMap rowMap = new KeyMatchedRowMap();
    
    /**
     * データマップのカラムの関係により行の表示有無を切り替える.
     */
    private DisplayMode displayMode = DisplayMode.ALL;

    /**
     * ダーティフラグ.<br>
     * データソースが追加・クリアされるか、キーマッチャが変更されると設定される.<br>
     * 一度設定されると、明示にクリアされるまで維持される.<br>
     */
    private boolean dirty;
    

    
    public void addPropertyChangeListener(PropertyChangeListener pl) {
        propCng.addPropertyChangeListener(pl);
    }

    public void removePropertyChangeListener(PropertyChangeListener pl) {
        propCng.removePropertyChangeListener(pl);
    }

    public PropertyChangeListener[] getPropertyChangeListeners() {
        return propCng.getPropertyChangeListeners();
    }

    public void addPropertyChangeListener(String string, PropertyChangeListener pl) {
        propCng.addPropertyChangeListener(pl);
    }

    public void removePropertyChangeListener(String string, PropertyChangeListener pl) {
        propCng.removePropertyChangeListener(pl);
    }

    public PropertyChangeListener[] getPropertyChangeListeners(String string) {
        return propCng.getPropertyChangeListeners(string);
    }
    

    /**
     * 変更フラグを取得する
     * @return 変更フラグ
     */
    public boolean isDirty() {
        return dirty;
    }
    
    /**
     * 変更フラグを設定する
     * @param dirty 変更フラグ
     */
    public void setDirty(boolean dirty) {
        boolean oldValue = this.dirty;
        this.dirty = dirty;
        propCng.firePropertyChange(PROPERTY_DIRTY, oldValue, dirty);
    }
    

    /**
     * すべてのデータソースをクリアする.
     */
    public void clear() {
        rowMap.clear();
        renumbering(rowMap);
        setDirty(true);
    }
    
    /**
     * データソースを追加する.
     * @param lineDataList 
     */
    public void addLineDataList(LineDataList lineDataList) {
        if (lineDataList == null) {
            throw new IllegalArgumentException();
        }
        rowMap.addLineDataList(lineDataList);
        rowMap.remap();

        renumbering(rowMap);
        setDirty(true);
    }
    
    /**
     * 保持しているデータソースの数を返す
     * @return データソースの数
     */
    public int getDataListCount() {
        return rowMap.getNumOfLineDataLists();
    }
    
    /**
     * 現在適用されているキーマッチング方法を取得する.
     * @return キーマッヂング方法
     */
    public KeyMatcher getKeyMatcher() {
        return rowMap.getKeyMatcher();
    }
    
    /**
     * キーマッチング方法を設定する.<br>
     * 必要に応じて表は再構成される.<br>
     * @param keyMatcher キーマッチング方法
     */
    public void setKeyMatcher(KeyMatcher keyMatcher) {
        if (keyMatcher == null) {
            throw new IllegalArgumentException();
        }
        
        KeyMatcher oldValue = getKeyMatcher();
        if (oldValue != keyMatcher) {
            // キータイプが変更される場合のみ
            // キーを変更してマッチチングを再適用し、表を構成し直す.
            rowMap.setKeyMatcher(keyMatcher);
            rowMap.remap();

            renumbering(rowMap);

            propCng.firePropertyChange(PROPERTY_KEY_MATCHER, oldValue, keyMatcher);

            setDirty(true);
        }
    }
    
    /**
     * キーまたはデータソースのタイトルを返す.<br>
     * 0列がキーで、それ以降はデータソースごとのタイトルが返される.<br>
     * @param i
     * @return 
     */
    @Override
    public String getColumnName(int i) {
        if (i == 0) {
            return COLUMN_NAME_KEY;
        }
        int column = i - 1; // 左端はキーカラム固定
        String title = rowMap.getTitle(column);

        if (StringUtils.isBlank(title)) {
            // 空の場合は列番号をつけて返す.
            title = "(" + column + ")";
        }
        
        return title;
    }

    @Override
    public KeyMatchedRowMap getRowMap() {
        return rowMap;
    }

    @Override
    public KeyMatchedRowView getRowView() {
        return this;
    }

    /**
     * このデータをファイルに保存する.
     * @param file 保存先ファイル
     * @throws IOException 失敗
     */
    public void save(DocArchive archive) throws IOException {
        if (archive == null) {
            throw new IllegalArgumentException();
        }
        
        archive.save(this);
    }
    
    /**
     * ファイルからドキュメントをロードする.
     * @param archive 保存元ファイル
     * @throws IOException 失敗
     */
    public void load(DocArchive archive) throws IOException {
        if (archive == null) {
            throw new IllegalArgumentException();
        }

        TextKeyMatcherDoc doc = archive.load();
        
        clear();
        
        KeyMatchedRowMap newRowMap = doc.getRowMap();

        int mx = newRowMap.getNumOfLineDataLists();
        for (int idx = 0; idx < mx; idx++) {
            LineDataList lineDataList = newRowMap.getLineDataList(idx);
            rowMap.addLineDataList(lineDataList);
        }
        
        rowMap.setKeyMatcher(newRowMap.getKeyMatcher());
        rowMap.remap();
        renumbering(rowMap);
    }
    
    /**
     * ディスプレイモードによる行フィルタを適用する.<br>
     * @param displayMode ディスプレイモード
     */
    public void setDisplayMode(DisplayMode displayMode) {
        if (displayMode == null) {
            throw new IllegalArgumentException();
        }
        DisplayMode oldValue = this.displayMode;
        this.displayMode = displayMode;
        
        if (displayMode != oldValue) {
            propCng.firePropertyChange(PROPERTY_DISPLAY_MODE, oldValue, displayMode);

            // 行フィルタを適用しなおす
            renumbering(rowMap);
        }
    }
    
    /**
     * 現在適用されているディスプレイモード(行フィルタ)を取得する.
     * @return ディスプレイモード
     */
    public DisplayMode getDisplayMode() {
        return displayMode;
    }

    @Override
    public void renumbering(Map<RowKey, RowValues> dataMap) {
        super.renumbering(new FilteredRowMap(dataMap, displayMode));
    }
}
