/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package textkeymatcher.entity;

/**
 * カラムの関係で行の表示・非表示を決定する.<br>
 * 
 * @author seraphy
 */
public enum DisplayMode implements RowFilter {
    
    /**
     * すべて表示.<br>
     */
    ALL() {
        @Override
        public boolean isAcceptable(RowValues values) {
            // フィルタなし
            return true;
        }
    },
    /**
     * すべてのカラムに値があるもののみ表示.<br>
     */
    MATCHED() {
        @Override
        public boolean isAcceptable(RowValues values) {
            if (values == null) {
                throw new IllegalArgumentException();
            }
            int mx = values.getDataWidth();
            int validColumns = 0;
            for (int idx = 0; idx < mx; idx++) {
                int cnt = values.getRowCountAt(idx);
                if (cnt > 0) {
                    // 1件以上のデータがあるカラムをカウントする.
                    validColumns++;
                }
            }
            // すべてのカラムにデータがあればアクセプト
            return mx == validColumns;
        }
    },
    /**
     * いずれかのカラムに値がないもののみを表示.<br>
     */
    UNMATCHED() {
        @Override
        public boolean isAcceptable(RowValues values) {
            if (values == null) {
                throw new IllegalArgumentException();
            }
            int mx = values.getDataWidth();
            int validColumns = 0;
            for (int idx = 0; idx < mx; idx++) {
                int cnt = values.getRowCountAt(idx);
                if (cnt > 0) {
                    // 1件以上のデータがあるカラムをカウントする.
                    validColumns++;
                }
            }
            // いずれかのカラムにデータがなければアクセプト
            return mx != validColumns;
        }
    },
    /**
     * 最初のカラムに値があるもののみ表示.<br>
     */
    EXISTS_FIRST_COLUMN() {
        @Override
        public boolean isAcceptable(RowValues values) {
            if (values == null) {
                throw new IllegalArgumentException();
            }
            int mx = values.getDataWidth();
            if (mx > 0) {
                int cnt = values.getRowCountAt(0);
                if (cnt > 0) {
                    return true;
                }
            }
            return false;
        }
    },
    /**
     * 最初のカラムに値がないもののみ表示.<br>
     */
    MISSING_FIRST_COLUMN() {
        @Override
        public boolean isAcceptable(RowValues values) {
            if (values == null) {
                throw new IllegalArgumentException();
            }
            int mx = values.getDataWidth();
            if (mx > 0) {
                int cnt = values.getRowCountAt(0);
                if (cnt == 0) {
                    return true;
                }
            }
            return false;
        }
    },
    UNIQUE() {
        @Override
        public boolean isAcceptable(RowValues values) {
            if (values == null) {
                throw new IllegalArgumentException();
            }
            int mx = values.getDataWidth();
            int validColumns = 0;
            for (int idx = 0; idx < mx; idx++) {
                int cnt = values.getRowCountAt(idx);
                if (cnt > 0) {
                    // 1件以上のデータがあるカラムをカウントする.
                    validColumns++;
                }
            }
            // データ列の個数が1であればユニーク
            return validColumns == 1;
        }
    },
    UNUNIQUE() {
        @Override
        public boolean isAcceptable(RowValues values) {
            if (values == null) {
                throw new IllegalArgumentException();
            }
            int mx = values.getDataWidth();
            int validColumns = 0;
            for (int idx = 0; idx < mx; idx++) {
                int cnt = values.getRowCountAt(idx);
                if (cnt > 0) {
                    // 1件以上のデータがあるカラムをカウントする.
                    validColumns++;
                }
            }
            // データ列の個数が1でなければアンユニーク
            return validColumns != 1;
        }
    },
    UNIQUE_ALL() {
        @Override
        public boolean isAcceptable(RowValues values) {
            if (values == null) {
                throw new IllegalArgumentException();
            }
            int mx = values.getDataWidth();
            int numOfData = 0;
            for (int idx = 0; idx < mx; idx++) {
                int cnt = values.getRowCountAt(idx);
                numOfData += cnt;
            }
            // データ数が1であればユニーク
            return numOfData == 1;
        }
    },
    UNUNIQUE_ALL() {
        @Override
        public boolean isAcceptable(RowValues values) {
            if (values == null) {
                throw new IllegalArgumentException();
            }
            int mx = values.getDataWidth();
            int numOfData = 0;
            for (int idx = 0; idx < mx; idx++) {
                int cnt = values.getRowCountAt(idx);
                numOfData += cnt;
            }
            // データ数が1でなければアンユニーク
            return numOfData != 1;
        }
    },
    ;
    
    /**
     * 行ごとのカラムデータを指定し、カラムの関係から、その行を表示するか否かを判定する.<br>
     * @param values 行ごとのカラムデータ
     * @return 行が受け入れ可能であればtrue
     */
    @Override
    public abstract boolean isAcceptable(RowValues values);
    
}
