/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package textkeymatcher.entity;

import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeMap;

/**
 * 各カラムデータと、キーによるマッチングを行い、キーと値リストの組にする.<br>
 * @author seraphy
 */
public class KeyMatchedRowMap extends AbstractMap<RowKey, RowValues> {

    /**
     * データソースのコレクション.<br>
     */
    private ArrayList<LineDataList> lineDataLists = new ArrayList<LineDataList>();
    
    
    /**
     * キーの判定方法を示すキーマッチャ
     */
    private KeyMatcher keyMatcher = KeyMatcher.TEXT;
    
    /**
     * キーの判定方法を設定する.<br>
     * 設定後は{@link #remap() }を呼び出して再構築しないかぎり反映されない.<br>
     * @param keyMatcher 
     */
    public void setKeyMatcher(KeyMatcher keyMatcher) {
        if (keyMatcher == null) {
            throw new IllegalArgumentException();
        }
        KeyMatcher oldValue = this.keyMatcher;
        this.keyMatcher = keyMatcher;
    }
    
    /**
     * キーの判定方法を取得する.
     * @return 
     */
    public KeyMatcher getKeyMatcher() {
        return keyMatcher;
    }

    /**
     * データソースをクリアし、マップを初期化する.<br>
     */
    @Override
    public void clear() {
        lineDataLists.clear();
        remap();
    }

    /**
     * データソース数を返す.<br>
     * @return データソース数
     */
    public int getNumOfLineDataLists() {
        return lineDataLists.size();
    }

    /**
     * データソースを追加する.
     * @param lineDataList データソース 
     */
    public void addLineDataList(LineDataList lineDataList) {
        if (lineDataList == null) {
            throw new IllegalArgumentException();
        }
        lineDataLists.add(lineDataList);
    }
    
    /**
     * データソースを削除する.
     * @param idx データソース
     */
    public void removeLineDataList(int idx) {
        lineDataLists.remove(idx);
    }

    /**
     * 指定されたインデックスのデータソースを取得する
     * @param idx インデックス
     * @return データソース
     */
    public LineDataList getLineDataList(int idx) {
        return lineDataLists.get(idx);
    }

    /**
     * データソースのタイトルを取得する.<br>
     * タイトル自身はデータソースが保持しているため、これはアクセスヘルパである.<br>
     * @param columnIndex データソースのインデックス
     * @return タイトル
     */
    public String getTitle(int columnIndex) {
        return getLineDataList(columnIndex).getTitle();
    }
    
    

    /**
     * キーと値のリストに選別されたデータ表現
     */
    private Map<RowKey, RowValues> dataMap = Collections.emptyMap();
    
    
    /**
     * データソースをキーに従って分類しマッピングする.<br>
     */
    public void remap() {
        TreeMap<RowKey, RowValues> map = new TreeMap<RowKey, RowValues>();
        
        // データ幅(データ数)
        int dataWidth = lineDataLists.size();
        
        // 全データからキー順のマップを構築する.
        for (int dataIndex = 0; dataIndex < dataWidth; dataIndex++) {
            LineDataList dataList = lineDataLists.get(dataIndex);
            
            for (LineData lineData : dataList) {
                // キーの構築
                String key = lineData.getKey();
                RowKey rowKey = new RowKey(keyMatcher, key);

                // nullは空文字に置換されているので非null
                String value = lineData.getValue();
                
                // キーのエントリを取得
                // なければエントリを作成する.
                RowValues rowValue = map.get(rowKey);
                if (rowValue == null) {
                    rowValue = new RowValues(dataWidth);
                    map.put(rowKey, rowValue);
                }
                rowValue.add(dataIndex, value);
            }
        }
        
        // 確定
        this.dataMap = map;
    }

    
    @Override
    public Set<Entry<RowKey, RowValues>> entrySet() {
        return Collections.unmodifiableSet(dataMap.entrySet());
    }

    @Override
    public int size() {
        return dataMap.size();
    }
}
