/*
 * CAN/LIN Checker for RL78/F13,F14
 *
 * Target: QB-R5F10PPJ-TB (RL78/F14, 100pin, 256KB ROM, 20KB RAM)
 * Author: Yasushi Tanaka
 *
 * [ DTChCo ]
 */

#include "common.h"
#include "iodefine.h"
#include "cpu.h"
#include "log.h"
#include "dtc.h"

/*
 * DTC
 * n[hEFA萔
 */
#define DTC_BUFFER_SIZE			((u2)(0x0050))
									/* DTCRg[f[^TCY(xN^܂) */
#define DTC_UART0_RX			((u1)(10))
									/* DTCNvԍ(UART0M) */
#define DTC_UART0_TX			((u1)(11))
									/* DTCNvԍ(UART0M) */
#define DTC_UART1_RX			((u1)(12))
									/* DTCNvԍ(UART1M) */
#define DTC_UART1_TX			((u1)(13))
									/* DTCNvԍ(UART1M) */

/*
 * DTCRg[Ef[^
 * ItZbg
 */
#define DTC_CTRL0_OFFSET		((u1)(0x40))
									/* DTCxN^Rg[Ef[^0ւ̃ItZbg */
#define DTC_CTRL1_OFFSET		((u1)(0x48))
									/* DTCxN^Rg[Ef[^1ւ̃ItZbg */
#define DTC_DTCCR_OFFSET		((u1)(0))
									/* Rg[Ef[^擪DTCCRjւ̃ItZbg */
#define DTC_DTBLS_OFFSET		((u1)(1))
									/* Rg[Ef[^擪DTBLSjւ̃ItZbg */
#define DTC_DTCCT_OFFSET		((u1)(2))
									/* Rg[Ef[^擪DTCCTjւ̃ItZbg */
#define DTC_DTRLD_OFFSET		((u1)(3))
									/* Rg[Ef[^擪DTRLDւ̃ItZbg */
#define DTC_DTSARL_OFFSET		((u1)(4))
									/* Rg[Ef[^擪DTSAR(LO)ւ̃ItZbg */
#define DTC_DTSARH_OFFSET		((u1)(5))
									/* Rg[Ef[^擪DTSAR(HI)ւ̃ItZbg */
#define DTC_DTDARL_OFFSET		((u1)(6))
									/* Rg[Ef[^擪DTDAR(LO)ւ̃ItZbg */
#define DTC_DTDARH_OFFSET		((u1)(7))
									/* Rg[Ef[^擪DTDAR(HI)ւ̃ItZbg */

/*
 * SFRݒl(PER1WX^)
 */
#define RL78F_PER1_DTCEN		((u1)(0x08))
									/* DTCL */

/*
 * SFRݒl(DTCENiWX^)
 */
#define RL78F_DTCEN_DISABLE		((u1)(0x00))
									/* DTCN֎~(Srbg) */

/*
 * SFRݒl(DTCEN1WX^)
 */
#define RL78F_DTCEN1_UART1_TX	((u1)(0x04))
									/* DTCN(UART1M) */
#define RL78F_DTCEN1_UART1_RX	((u1)(0x08))
									/* DTCN(UART1M) */
#define RL78F_DTCEN1_UART0_TX	((u1)(0x10))
									/* DTCN(UART0M) */
#define RL78F_DTCEN1_UART0_RX	((u1)(0x20))
									/* DTCN(UART0M) */

/*
 * SFRݒl(DTCCRjWX^)
 */
#define RL78F_DTCCR_SAMOD		((u1)(0x04))
									/* ]AhXZ */
#define RL78F_DTCCR_DAMOD		((u1)(0x08))
									/* ]AhXZ */

/*
 * SFRݒl(DTBLSjWX^)
 */
#define RL78F_DTBLS_ONE			((u1)(0x01))
									/* 1̓]8rbg܂16rbg] */

/*
 * SFRݒl(DTRLDjWX^)
 */
#define RL78F_DTRLD_ONE			((u1)(0x01))
									/* s[gE[hŃ[h] */

/*
 * SFRݒl(DTDARjWX^)
 */
#define RL78F_DTDAR_SDR00L		((u2)(&(SDR00L)))
									/* UART0Mobt@WX^ */
#define RL78F_DTDAR_SDR01L		((u2)(&(SDR01L)))
									/* UART0Mobt@WX^ */
#define RL78F_DTDAR_SDR10L		((u2)(&(SDR10L)))
									/* UART1Mobt@WX^ */
#define RL78F_DTDAR_SDR11L		((u2)(&(SDR11L)))
									/* UART1Mobt@WX^ */

/*
 * DTC
 * xN^Ee[uRg[Ef[^̈
 */
static u1 dtc_buffer[DTC_BUFFER_SIZE + LOG_BUFFER_SIZE];
									/* DTC_BUFFER_SIZE + LOG_BUFFER_SIZE >= 0x0200OƂ */
static SADDR volatile u1* dtc_vector;
									/* DTCxN^Ee[u(0x100oCgŃACgς) */

/*
 * DTC
 * 
 */
void dtc_init(void)
{
	u2  dtc_buffer_near;
	u2  dtc_buffer_add;
	u2  dtc_offset;
	u1* log_buf;
	u2  log_bytes;

	/* ӃCl[uEWX^1DTCLɂ */
	PER1 |= RL78F_PER1_DTCEN;

	/* DTCxN^̐擪AhX𓾂(0x100oCgPʂɃACg) */
	dtc_buffer_near = (u2)&dtc_buffer[0];
	dtc_buffer_add = (u2)((dtc_buffer_near + 0x00ff) & 0xff00);

	/* ZoDTCxN^ƃobt@Ƃ̃ItZbgZo */
	dtc_offset = (u2)(dtc_buffer_add - dtc_buffer_near);

	/* DTCxN^ݒ */
	dtc_vector = &dtc_buffer[dtc_offset];

	/* Oobt@ݒ */
	log_buf = &dtc_buffer[dtc_offset + DTC_BUFFER_SIZE];

	/* Oobt@Ŏgp\ȃoCg */
	log_bytes = sizeof(dtc_buffer) - (dtc_offset + DTC_BUFFER_SIZE);

	/* O֒ʒm */
	log_set_buf(log_buf, log_bytes);

	/* DTCNvׂċ֎~ */
	DTCEN0 = RL78F_DTCEN_DISABLE;
	DTCEN1 = RL78F_DTCEN_DISABLE;
	DTCEN2 = RL78F_DTCEN_DISABLE;
	DTCEN3 = RL78F_DTCEN_DISABLE;
	DTCEN4 = RL78F_DTCEN_DISABLE;
#ifdef R5F10PPJ
	DTCEN5 = RL78F_DTCEN_DISABLE;
#endif /* R5F10PPJ */

	/* DTCx[XWX^ݒ */
	DTCBAR = (u1)(dtc_buffer_add >> 8);

	/* Rg[Ef[^0̋NvUART0MɊ蓖Ă */
	dtc_vector[DTC_UART0_TX] = DTC_CTRL0_OFFSET;

	/* Rg[Ef[^0̐ݒ(Mɐݒ肷镔̓XLbv) */
	dtc_vector[DTC_CTRL0_OFFSET + DTC_DTCCR_OFFSET] = RL78F_DTCCR_SAMOD;
	dtc_vector[DTC_CTRL0_OFFSET + DTC_DTBLS_OFFSET] = RL78F_DTBLS_ONE;
	dtc_vector[DTC_CTRL0_OFFSET + DTC_DTRLD_OFFSET] = RL78F_DTRLD_ONE;
	dtc_vector[DTC_CTRL0_OFFSET + DTC_DTDARL_OFFSET] = (u1)(RL78F_DTDAR_SDR00L & 0x00ff);
	dtc_vector[DTC_CTRL0_OFFSET + DTC_DTDARH_OFFSET] = (u1)(RL78F_DTDAR_SDR00L >> 8);

	/* Rg[Ef[^1̋NvUART1MɊ蓖Ă */
	dtc_vector[DTC_UART1_TX] = DTC_CTRL1_OFFSET;

	/* Rg[Ef[^1̐ݒ(Mɐݒ肷镔̓XLbv) */
	dtc_vector[DTC_CTRL1_OFFSET + DTC_DTCCR_OFFSET] = RL78F_DTCCR_SAMOD;
	dtc_vector[DTC_CTRL1_OFFSET + DTC_DTBLS_OFFSET] = RL78F_DTBLS_ONE;
	dtc_vector[DTC_CTRL1_OFFSET + DTC_DTRLD_OFFSET] = RL78F_DTRLD_ONE;
	dtc_vector[DTC_CTRL1_OFFSET + DTC_DTDARL_OFFSET] = (u1)(RL78F_DTDAR_SDR10L & 0x00ff);
	dtc_vector[DTC_CTRL1_OFFSET + DTC_DTDARH_OFFSET] = (u1)(RL78F_DTDAR_SDR10L >> 8);

	/* DTC`l0蓖ĂxN^ցA_~[f[^ݒ肷 */
	dtc_vector[DTC_UART0_RX] = DTC_CTRL0_OFFSET;

	/* DTC`l0̐ݒ(Mɐݒ肷镔̓XLbv) */
	SELHS0 = DTC_UART0_RX;
	HDTCCR0 = RL78F_DTCCR_DAMOD;
	HDTSAR0 = (u2)(RL78F_DTDAR_SDR01L & 0x0fff);
	HDTRLD0 = RL78F_DTRLD_ONE;

	/* DTC`l1蓖ĂxN^ցA_~[f[^ݒ肷 */
	dtc_vector[DTC_UART1_RX] = DTC_CTRL1_OFFSET;

	/* DTC`l1̐ݒ(Mɐݒ肷镔̓XLbv) */
	SELHS1 = DTC_UART1_RX;
	HDTCCR1 = RL78F_DTCCR_DAMOD;
	HDTSAR1 = (u2)((u2)(&SDR11L) & 0x0fff);
	HDTRLD1 = RL78F_DTRLD_ONE;
}

/*
 * DTC
 * UARTMREADY`FbN
 *
 * READYłU1_TRUEԂ
 */
u1 dtc_tx_is_ready(u1 ch)
{
	u1 ret;

	/* ߂lREADYŏ */
	ret = U1_FALSE;

	/* `l */
	if (0 == ch)
	{
		if (0 == (u1)(DTCEN1 & RL78F_DTCEN1_UART0_TX))
		{
			/* UART0fB */
			ret = U1_TRUE;
		}
	}
	else
	{
		if (0 == (u1)(DTCEN1 & RL78F_DTCEN1_UART1_TX))
		{
			/* UART1fB */
			ret = U1_TRUE;
		}
	}

	return ret;
}

/*
 * DTC
 * ]Jn
 *
 * http://documentation.renesas.com/doc/products/mpumcu/tu/tnr8ca047aj.pdf
 * 3-2߂QƁBDTCEN1ORlƂ͕ʂ̗vċN邱Ƃh
 *
 * 荞݋֎~ԂŌĂ΂
 */
static void dtc_enable(u1 en)
{
	u1 dtcen;
	u1 dtcct;
	u1 retry;

	/* dtcen */
	dtcen = (u1)(DTCEN1 | en);

	/* gC */
	retry = U1_TRUE;

	/* [v */
	while (U1_TRUE == retry)
	{
		/* DTCEN1Zbg */
		DTCEN1 = dtcen;

		/* gCȂ */
		retry = U1_FALSE;
	
		/* UART0M */
		if (0 != (u1)(DTCEN1 & RL78F_DTCEN1_UART0_RX))
		{
			/* DTCCT擾 */
			dtcct = HDTCCT0;

			/* 0x00,0x80-0xff͈̔͂ł΃^C~OĂ */
			if ((dtcct < 0x01) || (dtcct > 0x7f))
			{
				dtcen &= (u1)(~RL78F_DTCEN1_UART0_RX);
				retry = U1_TRUE;
			}
		}

		/* UART0M */
		if (0 != (u1)(DTCEN1 & RL78F_DTCEN1_UART0_TX))
		{
			/* DTCCT擾 */
			dtcct = dtc_vector[DTC_CTRL0_OFFSET + DTC_DTCCT_OFFSET];

			/* 0x00,0x80-0xff͈̔͂ł΃^C~OĂ */
			if ((dtcct < 0x01) || (dtcct > 0x7f))
			{
				dtcen &= (u1)(~RL78F_DTCEN1_UART0_TX);
				retry = U1_TRUE;
			}
		}

		/* UART1M */
		if (0 != (u1)(DTCEN1 & RL78F_DTCEN1_UART1_RX))
		{
			/* DTCCT擾 */
			dtcct = HDTCCT1;

			/* 0x00,0x80-0xff͈̔͂ł΃^C~OĂ */
			if ((dtcct < 0x01) || (dtcct > 0x7f))
			{
				dtcen &= (u1)(~RL78F_DTCEN1_UART1_RX);
				retry = U1_TRUE;
			}
		}

		/* UART1M */
		if (0 != (u1)(DTCEN1 & RL78F_DTCEN1_UART1_TX))
		{
			/* DTCCT擾 */
			dtcct = dtc_vector[DTC_CTRL1_OFFSET + DTC_DTCCT_OFFSET];

			/* 0x00,0x80-0xff͈̔͂ł΃^C~OĂ */
			if ((dtcct < 0x01) || (dtcct > 0x7f))
			{
				dtcen &= (u1)(~RL78F_DTCEN1_UART1_TX);
				retry = U1_TRUE;
			}
		}
	}
}

/*
 * DTC
 * UARTMJn
 *
 * 荞݋֎~ԂŌĂ΂
 */
void dtc_tx_start(u1 ch, u1* buf, u1 len)
{
	u2 buf_near;

	/* obt@AhX̉16bit̂ݎ擾(RAM܂ROM~[ւ̔zuO) */
	buf_near = (u2)&buf[0];

	/* `l */
	if (0 == ch)
	{
		/* ]AhX */
		dtc_vector[DTC_CTRL0_OFFSET + DTC_DTSARL_OFFSET] = (u1)(buf_near);
		dtc_vector[DTC_CTRL0_OFFSET + DTC_DTSARH_OFFSET] = (u1)(buf_near >> 8);

		/* ] */
		dtc_vector[DTC_CTRL0_OFFSET + DTC_DTCCT_OFFSET] = len;

		/* ]Jn */
		dtc_enable(RL78F_DTCEN1_UART0_TX);
	}
	else
	{
		/* ]AhX */
		dtc_vector[DTC_CTRL1_OFFSET + DTC_DTSARL_OFFSET] = (u1)(buf_near);
		dtc_vector[DTC_CTRL1_OFFSET + DTC_DTSARH_OFFSET] = (u1)(buf_near >> 8);

		/* ] */
		dtc_vector[DTC_CTRL1_OFFSET + DTC_DTCCT_OFFSET] = len;

		/* ]Jn */
		dtc_enable(RL78F_DTCEN1_UART1_TX);
	}
}

/*
 * DTC
 * UARTMJn
 *
 * 荞݋֎~ԂŌĂ΂
 */
void dtc_rx_start(u1 ch, u1* buf, u1 len)
{
	u2 buf_near;

	/* obt@AhX̉16bit̂ݎ擾(RAM܂ROM~[ւ̔zuO) */
	buf_near = (u2)&buf[0];

	/* `l */
	if (0 == ch)
	{
		/* ]AhX */
		HDTDAR0 = buf_near;

		/* ] */
		HDTCCT0 = len;

		/* ]Jn */
		dtc_enable(RL78F_DTCEN1_UART0_RX);
	}
	else
	{
		/* ]AhX */
		HDTDAR1 = buf_near;

		/* ] */
		HDTCCT1 = len;

		/* ]Jn */
		dtc_enable(RL78F_DTCEN1_UART1_RX);
	}
}

/*
 * DTC
 * UART]񐔎擾
 */
 void dtc_rx_get_dtcct(u1 ch, u1* dtcct)
{
	/* `l */
	if (0 == ch)
	{
		*dtcct = HDTCCT0;
	}
	else
	{
		*dtcct = HDTCCT1;
	}
}
