
#ifndef CM_CM_SOCKET_SETTER_H_
#define CM_CM_SOCKET_SETTER_H_

#include <arpa/inet.h>
#include <utility>

#include "cm_socket_type.h"
#include "cm_socket_address.h"
#include "mt_mpl.h"
#include "mt_unuse.h"
#include "mt_typelist.h"
#include "mt_shim_clear_memory.h"

namespace cm {

template <SocketType SOCK_TYPE, int PROT_FAMILY, int SOCKET>
struct SockSetter
{
    void operator()(std::pair<int, int>& pair)
    {
        pair.first = PROT_FAMILY;
        pair.second = SOCKET;
    }

    static SocketType getKey()
    {
        return SOCK_TYPE;
    }
};

struct SocketAddressSetterUnix
{
    typedef mt::StaticValue<int, 2> CtorArgsType;

    SocketAddressSetterUnix(const char*& name, unsigned short& port)
        : name_(name), port_(port)
    {}

    void operator()(std::pair<SocketAddress, socklen_t>& pair)
    {
        mt::unuse(pair);
    }

    static int getKey()
    {
        return PF_UNIX;
    }

    const char*& name_;
    unsigned short& port_;
};

struct SocketAddressSetterInet
{
    typedef mt::StaticValue<int, 2> CtorArgsType;

    SocketAddressSetterInet(const char*& name, unsigned short& port)
        : name_(name), port_(port)
    {}

    void operator()(std::pair<SocketAddress, socklen_t>& pair)
    {
        mt::clearMemory(pair.first);
        struct sockaddr_in& in4 = pair.first.in4;

        in4.sin_family = PF_INET;
        int ret = inet_pton(PF_INET, name_, &in4.sin_addr);
        assert(ret == 1);
        in4.sin_port = htons(port_);

        pair.second = sizeof(in4);
    }

    static int getKey()
    {
        return PF_INET;
    }

    const char*& name_;
    unsigned short& port_;
};

template <SocketType SOCK_TYPE, int PROT_FAMILY, int SOCKET>
inline SocketType (*getFindIfKey(SockSetter<SOCK_TYPE, PROT_FAMILY, SOCKET>&))()
{
    return &SockSetter<SOCK_TYPE, PROT_FAMILY, SOCKET>::getKey;
}

template <typename T>
inline int (*getFindIfKey(T&))()
{
    return &T::getKey;
}

typedef mt::MakeTypelist<SockSetter<SOCKET_TYPE_UNIX_DGRAM,     PF_LOCAL, SOCK_DGRAM>,
                         SockSetter<SOCKET_TYPE_UNIX_STREAM,    PF_LOCAL, SOCK_STREAM>,
                         SockSetter<SOCKET_TYPE_UNIX_SEQPACKET, PF_LOCAL, SOCK_SEQPACKET>,
                         SockSetter<SOCKET_TYPE_INET_DGRAM,     PF_INET,  SOCK_DGRAM>,
                         SockSetter<SOCKET_TYPE_INET_STREAM,    PF_INET,  SOCK_STREAM>
                        >::Result SocketSetterTypes;

typedef mt::MakeTypelist<SocketAddressSetterUnix,
                         SocketAddressSetterInet
                        >::Result SocketAddressSetterTypes;

} // namespace cm

#endif // CM_CM_SOCKET_SETTER_H_
