
#include "gmock/gmock.h"
#include "mt_auto_ptr.h"

namespace {

class MockClass
{
public:
    MOCK_METHOD0(getNonConst, unsigned int());

    MOCK_CONST_METHOD0(getConst, unsigned int());

    MOCK_CONST_METHOD0(get, unsigned int());
    MOCK_METHOD0(get, unsigned int());

    virtual ~MockClass()
    {}
};

#if 0 // Should not be compiled.
class MockClassUsingIncompleteType
{
public:
    MOCK_METHOD0(getNonConst, unsigned int());

    ~MockClassUsingIncompleteType()
    {}

private:
    struct Impl;    // This indicates incomplete type;

    mt::AutoPtr<Impl> pimpl_;
};
#endif

} // namespace

#if 0 // Should not be compiled
TEST(AutoPtrTest, non_const_auto_ptr_for_incomplete_type)
{
    mt::AutoPtr<MockClassUsingIncompleteType> auto_ptr;
}
#endif

TEST(AutoPtrTest, non_const_auto_ptr)
{
    MockClass* mock = new MockClass;

    mt::AutoPtr<MockClass> auto_ptr(mock);

    EXPECT_CALL(*mock, getNonConst())
        .WillOnce(::testing::Return(1234u));

    EXPECT_EQ(auto_ptr->getNonConst(), 1234u);

    EXPECT_EQ(auto_ptr.get(), mock);

    // Since this test uses Mock object in its managed object, we will be able to
    // confirm whether the dynamically allocated object is properly released after
    // the use.
}

TEST(AutoPtrTest, non_const_auto_ptr_ownership_move)
{
    MockClass* mock = new MockClass;

    mt::AutoPtr<MockClass> auto_ptr(mock), auto_ptr2;

    EXPECT_EQ(auto_ptr.get(), mock);

    auto_ptr2 = auto_ptr;

    EXPECT_EQ(auto_ptr.get(), static_cast<MockClass*>(0));
    EXPECT_EQ(auto_ptr2.get(), mock);

    EXPECT_CALL(*mock, getNonConst())
        .WillOnce(::testing::Return(1235u))
        .WillOnce(::testing::Return(1236u));

    EXPECT_EQ(auto_ptr2->getNonConst(), 1235u);

    mt::AutoPtr<MockClass> auto_ptr3(auto_ptr2);

    EXPECT_EQ(auto_ptr2.get(), static_cast<MockClass*>(0));
    EXPECT_EQ(auto_ptr3.get(), mock);

    EXPECT_EQ(auto_ptr3->getNonConst(), 1236u);
}

TEST(AutoPtrTest, const_auto_ptr_test)
{
    MockClass* mock = new MockClass;

    const mt::AutoPtr<MockClass> auto_ptr(mock);

    EXPECT_EQ(auto_ptr.get(), mock);

// The following should not be compiled since the below methods are non-const methods.
#if 0
    auto_ptr.release();
    auto_ptr.reset();
    MockClass* mock_new = new MockClass;
    auto_ptr.set(mock_new);
    delete mock_new;
#endif

    {   
        ::testing::InSequence dummy;
        EXPECT_CALL(*mock, getConst())
            .WillOnce(::testing::Return(1234u));

        EXPECT_CALL(static_cast<const MockClass&>(*mock), get())
            .WillOnce(::testing::Return(1235u));
    
        EXPECT_EQ(auto_ptr->getConst(), 1234u);

// The following should not be compiled since getNonConst() member function
// is non-const member function. Thus it should not be invoked via const AutoPtr.
#if 0 
        auto_ptr->getNonConst();
#endif

        EXPECT_EQ(auto_ptr->get(), 1235u);
    }

// The following should not be compiled as const AutoPtr will not allow ownership movement.
#if 0
    const mt::AutoPtr<MockClass> auto_ptr2(auto_ptr);
    const mt::AutoPtr<MockClass> auto_ptr2 = auto_ptr;
    const mt::AutoPtr<MockClass> auto_ptr3;
    auto_ptr3 = auto_ptr;
#endif
}

TEST(AutoPtrTest, auto_ptr_dereference_operation_test)
{
    MockClass* mock = new MockClass;

    {
        mt::AutoPtr<MockClass> auto_ptr(mock);
        MockClass& mock_ref = *auto_ptr;
        EXPECT_EQ(&mock_ref, mock);
    }

    mock = new MockClass;

    {
        const mt::AutoPtr<MockClass> auto_ptr(mock);
        const MockClass& cmock_ref = *auto_ptr;
        EXPECT_EQ(&cmock_ref, mock);
    }
}

