
#include <vector>
#include <string>
#include <list>

#include "gtest/gtest.h"
#include "mt_range.h"
#include "mt_type_identity.h"

TEST(MtRangeTest, begin_end_StdVector)
{
    std::vector<char> vec;
    vec.push_back('h');
    vec.push_back('e');
    vec.push_back('l');
    vec.push_back('l');
    vec.push_back('o');
    vec.push_back('!');

    EXPECT_EQ(mt::begin(vec), vec.begin());
    EXPECT_EQ(mt::end(vec), vec.end());

    EXPECT_EQ(*mt::begin(vec), 'h');

    *mt::begin(vec) = 'H';

    EXPECT_EQ(*mt::begin(vec), 'H');
}

TEST(MtRagneTest, cbegin_cend_StdVector)
{
    std::vector<char> vec;
    vec.push_back('h');
    vec.push_back('e');
    vec.push_back('l');
    vec.push_back('l');
    vec.push_back('o');
    vec.push_back('!');

    EXPECT_EQ(mt::cbegin(vec), vec.begin());
    EXPECT_EQ(mt::cend(vec), vec.end());

    EXPECT_EQ(*mt::cbegin(vec), 'h');

#if 0 // The following should not be compiled successfully
    *mt::cbegin(vec) = 'H';
#endif
    EXPECT_EQ(*mt::cbegin(vec), 'h');
}

TEST(MtRangeTest, begin_end_Array)
{
    char array[] = { 'h', 'e', 'l', 'l', 'o', '!' };

    EXPECT_EQ(mt::begin(array), &array[0]);
    EXPECT_EQ(mt::end(array), &array[sizeof(array)/sizeof(array[0])]);

    EXPECT_EQ(*mt::begin(array), 'h');

    *mt::begin(array) = 'H';

    EXPECT_EQ(*mt::begin(array), 'H');
}

TEST(MtRangeTest, cbegin_cend_Array)
{
    char array[] = { 'h', 'e', 'l', 'l', 'o', '!' };

    EXPECT_EQ(mt::cbegin(array), &array[0]);
    EXPECT_EQ(mt::cend(array), &array[sizeof(array)/sizeof(array[0])]);

    EXPECT_EQ(*mt::cbegin(array), 'h');

#if 0 // The following should not be compiled succressfully
    *mt::cbegin(array) = 'H';
#endif
    EXPECT_EQ(*mt::cbegin(array), 'h');
}

TEST(MtRangeTest, GetIteratorType_meta_function)
{
    typedef mt::GetIteratorType< std::vector<std::string> >::Result NonConstStdStringVectorIteratorType;
    typedef std::vector<std::string>::iterator ResultType1;
    bool is_identical_type = mt::IsIdentical<ResultType1, NonConstStdStringVectorIteratorType>::value;
    EXPECT_EQ(is_identical_type, true);

    typedef mt::GetIteratorType< std::list<std::string> >::Result NonConstStdStringListIteratorType;
    typedef std::list<std::string>::iterator ResultType2;
    is_identical_type = mt::IsIdentical<ResultType2, NonConstStdStringListIteratorType>::value;
    EXPECT_EQ(is_identical_type, true);

    typedef mt::GetIteratorType< std::string[10] >::Result NonConstStdString10ArrayIteratorType;
    typedef std::string* ResultType3;
    is_identical_type = mt::IsIdentical<ResultType3, NonConstStdString10ArrayIteratorType>::value;
    EXPECT_EQ(is_identical_type, true);
}

TEST(MtRangeTest, GetConstIteratorType_meta_function)
{
    typedef mt::GetConstIteratorType< std::vector<std::string> >::Result NonConstStdStringVectorIteratorType;
    typedef std::vector<std::string>::const_iterator ResultType1;
    bool is_identical_type = mt::IsIdentical<ResultType1, NonConstStdStringVectorIteratorType>::value;
    EXPECT_EQ(is_identical_type, true);

    typedef mt::GetConstIteratorType< std::list<std::string> >::Result NonConstStdStringListIteratorType;
    typedef std::list<std::string>::const_iterator ResultType2;
    is_identical_type = mt::IsIdentical<ResultType2, NonConstStdStringListIteratorType>::value;
    EXPECT_EQ(is_identical_type, true);

    typedef mt::GetConstIteratorType< std::string[10] >::Result NonConstStdString10ArrayIteratorType;
    typedef const std::string* ResultType3;
    is_identical_type = mt::IsIdentical<ResultType3, NonConstStdString10ArrayIteratorType>::value;
    EXPECT_EQ(is_identical_type, true);
}
