#define poker_create
/// poker_create(player, joker);
// randomize
var player = argument0;
var joker = argument1;

if(player < 1)
{
  // error.
  show_debug_message("Error: poker_create() player invalid = " + string(player));
  player = 1;
}
if(joker < 0)
{
  // error.
  show_debug_message("Error: poker_create() joker invalid = " + string(joker));
  joker = 1;
}

random_set_seed(date_current_datetime());
randomize();

var pk = ds_map_create();

var deck  = ds_list_create();
var talon = ds_list_create();
ds_map_add(pk, "deck", deck);
ds_map_add(pk, "talon", talon);
ds_map_add(pk, "player", player);
ds_map_add(pk, "joker", joker);

// set debug flag.
poker_debug_set(pk, false);

poker_reset(pk);

show_debug_message("poker_create() done.");

return pk;


#define poker_destroy
/// poker_destroy(pk);

var pk = argument0;
var deck = ds_map_find_value(pk, "deck");
var talon = ds_map_find_value(pk, "talon");
var player = ds_map_find_value(pk, "player");
for(var i = 0; i < player; i++)
{
  var key = "hand" + string(i);
  var hand = ds_map_find_value(pk, key);
  ds_list_destroy(hand);
}

ds_list_destroy(deck);
ds_list_destroy(talon);
ds_map_destroy(pk);

show_debug_message("poker_destroy() done.");


#define poker_reset
/// poker_reset(pk);

var pk = argument0;
var deck = ds_map_find_value(pk, "deck");
var talon = ds_map_find_value(pk, "talon");
var player = ds_map_find_value(pk, "player");
var joker = ds_map_find_value(pk, "joker");

ds_list_clear(deck);
ds_list_clear(talon);

for(var i = 0; i < player; i++)
{
  var key = "hand" + string(i);
  var hand = 0;
  if(ds_map_exists(pk, key))
  {
    hand = ds_map_find_value(pk, key);
    for(var j = 0; j < PK_HAND_CARD_NUM; j++)
    {
      ds_list_replace(hand, j, PK_CARD_INVALID);
    }
  }
  else
  {
    hand = ds_list_create();
    for(var j = 0; j < PK_HAND_CARD_NUM; j++)
    {
      ds_list_add(hand, PK_CARD_INVALID);
    }
  }
  ds_map_replace(pk, key, hand);
}

// add card to deck
var nCard = 52 + 1 + joker; // start is 1.
for(var i = 1; i < nCard; i++)
{
  ds_list_add(deck, i);
}

// shuffle.
poker_deck_shuffle(pk);

if(poker_debug_enable(pk))
{
  show_debug_message("poker_reset() done.");
}


#define poker_debug_set
/// poker_debug_set(pk, flag);
var pk = argument0;
var flag = argument1;

ds_map_replace(pk, "debug", flag);

#define poker_debug_enable
/// poker_debug_enable(pk);
var pk = argument0;

var ret = ds_map_find_value(pk, "debug");

return ret;


#define poker_dump
/// poker_dump(pk);
var pk = argument0;

var deck = ds_map_find_value(pk, "deck");
var talon = ds_map_find_value(pk, "talon");
var player = ds_map_find_value(pk, "player");
var joker = ds_map_find_value(pk, "joker");

show_debug_message("=================================");
show_debug_message("poker_dump() start.");
show_debug_message("=================================");

// deck
show_debug_message("[dump deck]");
{
  var str = "";
  for(var i = 0; i < ds_list_size(deck); i++)
  {
    var v = ds_list_find_value(deck, i);
    str += string(v) + ",";
  }
  
  show_debug_message(str);
}

// talon
show_debug_message("[dump talon]");
{
  var str = "";
  for(var i = 0; i < ds_list_size(talon); i++)
  {
    var v = ds_list_find_value(talon, i);
    str += string(v) + ",";
  }
  
  show_debug_message(str);
}

// player
show_debug_message("[player] count = " + string(player));
{
  for(var i = 0; i < player; i++)
  {
    var key = "hand" + string(i);
    var hand = ds_map_find_value(pk, key);
    var str = string(i) + ":";
    for(var j = 0; j < array_length_1d(hand); j++)
    {
      str += string(hand[j]) + ",";
    }
    show_debug_message(str);
  }
}

// joker
show_debug_message("[joker] count = " + string(joker));

show_debug_message("=================================");
show_debug_message("poker_dump() end.");
show_debug_message("=================================");



#define poker_deck_shuffle
/// poker_deck_shuffle(pk);
var pk = argument0;

var deck = ds_map_find_value(pk, "deck");

// shuffle.
ds_list_shuffle(deck);


#define poker_deck_add_talon
/// poker_deck_add_talon(pk);
var pk = argument0;

var deck = ds_map_find_value(pk, "deck");
var talon = ds_map_find_value(pk, "talon");

// talon to deck.
for(var i = 0; i < ds_list_size(talon); i++)
{
  var card = ds_list_find_value(talon, i);
  if(card == PK_CARD_INVALID)
  {
    show_debug_message("Error: poker_deck_add_talon() Invalid Idx = " + string(i));
  }
  else
  {
    ds_list_add(deck, card);
  }
}

// talon clear.
ds_list_clear(talon);


#define poker_deck_size
/// poker_deck_size(pk);
var pk = argument0;

var deck = ds_map_find_value(pk, "deck");

var count = 0;
for(var i = 0; i < ds_list_size(deck); i++)
{
  var card = ds_list_find_value(deck, i);
  if(card != PK_CARD_INVALID)
  {
    count++;
  }
}

return count;


#define poker_deck_get
/// poker_deck_get(pk, idx);
var pk = argument0;
var idx = argument1;

var deck = ds_map_find_value(pk, "deck");

var index = 0;
for(var i = 0; i < ds_list_size(deck); i++)
{
  var card = ds_list_find_value(deck, i);
  if(card != PK_CARD_INVALID)
  {
    if(idx == index)
    {
      // find.
      return card;
    }
    index++;
  }
}

show_debug_message("Error: poker_deck_get() Not found idx = " + string(idx));
return PK_CARD_INVALID;


#define poker_deck_deal_one
/// poker_deck_deal_one(pk);
var pk = argument0;
var deck = ds_map_find_value(pk, "deck");
if(ds_list_empty(deck))
{
  return PK_CARD_INVALID;
}

// deal card.
var ret = ds_list_find_value(deck, 0);
ds_list_delete(deck, 0);

if(poker_debug_enable(pk))
{
  // debug print.
  var str = "";
  for(var i = 0; i < ds_list_size(deck); i++)
  {
    str += string(ds_list_find_value(deck, i)) + ",";
  }
  show_debug_message("deck=" + str);
}

return ret;


#define poker_hand_deal
/// poker_hand_deal(pk, player);
var pk = argument0;
var player = argument1;
if(player < 0 or poker_player_count(pk) <= player)
{
  show_debug_message("Error: invalid player = " + string(player));
  exit;
}

var deck = ds_map_find_value(pk, "deck");

var key = "hand" + string(player);
var hand = ds_map_find_value(pk, key);

var count = poker_hand_get_deal_count(pk, player);

for(var i = 0; i < count; i++)
{
  var card = poker_deck_deal_one(pk);
  if(card == PK_CARD_INVALID)
  {
    if(poker_debug_enable(pk))
    {
      show_debug_message("***shuffle - deck size = " + string(ds_list_size(deck)));
    }
    poker_deck_add_talon(pk);
    poker_deck_shuffle(pk);
    card = poker_deck_deal_one(pk);
  }
  
  for(var j = 0; j < PK_HAND_CARD_NUM; j++)
  {
    var v = ds_list_find_value(hand, j);
    if(v == PK_CARD_INVALID)
    {
      ds_list_replace(hand, j, card);
      break;
    }
    if(j == PK_HAND_CARD_NUM-1)
    {
      show_debug_message("Error: poker_hand_deal() " + key + " is not empyt.");
    }
  }
}


#define poker_hand_get_deal_count
/// poker_hand_get_deal_count(pk, player);
var pk = argument0;
var player = argument1;
if(player < 0 or poker_player_count(pk) <= player)
{
  show_debug_message("Error: invalid player = " + string(player));
  return 0;
}

var deck = ds_map_find_value(pk, "deck");

var key = "hand" + string(player);
var hand = ds_map_find_value(pk, key);

var count = 0;
for(var i = 0; i < PK_HAND_CARD_NUM; i++)
{
  var v = ds_list_find_value(hand, i);
  if(v == PK_CARD_INVALID)
  {
    // find invalid card.
    count++;
  }
}

return count;


#define poker_hand_get_card
/// poker_hand_get_card(pk, player, idx);
var pk = argument0;
var player = argument1;
var idx = argument2;
if(player < 0 or poker_player_count(pk) <= player)
{
  show_debug_message("Error: invalid player = " + string(player));
  return PK_CARD_INVALID;
}

var deck = ds_map_find_value(pk, "deck");

var key = "hand" + string(player);
var hand = ds_map_find_value(pk, key);

if(idx < 0 or PK_HAND_CARD_NUM <= idx)
{
  show_debug_message("Error: poker_hand_get_card() invalid idx = " + string(idx));
  return PK_CARD_INVALID;
}

return ds_list_find_value(hand, idx);


#define poker_hand_discard
/// poker_hand_discard(pk, player, idx);
var pk = argument0;
var player = argument1;
var idx = argument2;

var deck = ds_map_find_value(pk, "deck");
var talon = ds_map_find_value(pk, "talon");

var key = "hand" + string(player);
var hand = ds_map_find_value(pk, key);

if(idx < 0 or PK_HAND_CARD_NUM <= idx)
{
  show_debug_message("Error: poker_hand_discard() invalid idx = " + string(idx));
}
else
{
  var card = ds_list_find_value(hand, idx);
  if(card != PK_CARD_INVALID)
  {
    // add talon.
    ds_list_add(talon, card);
  }
  ds_list_replace(hand, idx, PK_CARD_INVALID);
}


#define poker_hand_discard_all
/// poker_hand_discard_all(pk, player);
var pk = argument0;
var player = argument1;

for(var i = 0; i < PK_HAND_CARD_NUM; i++)
{
  poker_hand_discard(pk, player, i);
}


#define poker_hand_check_rank
/// poker_hand_check_rank(pk, player);
var pk = argument0;
var player = argument1;
if(player < 0 or poker_player_count(pk) <= player)
{
  show_debug_message("Error: invalid player = " + string(player));
  return PK_RANK_HIGH_CARDS;
}

var key = "hand" + string(player);
var hand_list = ds_map_find_value(pk, key);
var hand = 0;
hand[PK_HAND_CARD_NUM - 1] = 0;
for(var i = 0; i < PK_HAND_CARD_NUM; i++)
{
  hand[i] = ds_list_find_value(hand_list, i);
}

var is_flush = false;
var is_straight = false;
var is_royal = false;
var is_five = false;
var is_four = false;
var is_three = false;
var is_one = false;

// check flush.
is_flush = poker_rank_check_flush(hand);

if(poker_debug_enable(pk))
{
  var str = "";
  for(var i = 0; i < array_length_1d(hand); i++)
  {
    str += string(hand[i]) + ",";
  }
  show_debug_message("hand = " + str);
}


var list = ds_list_create();

for(var i = 0; i < array_length_1d(hand); i++)
{
  var v = hand[i];
  if(v >= PK_CARD_JOKER)
  {
    v = PK_CARD_JOKER;
  }
  else
  {
    v = (v - 1)%13;
  }
  ds_list_add(list, v);
}

// sort.
ds_list_sort(list, true);
var arr = 0;
for(var i = ds_list_size(list) - 1; i >= 0; i--)
{
  arr[i] = ds_list_find_value(list, i);
}
// destroy list.
ds_list_destroy(list);
if(poker_debug_enable(pk))
{
  var str = "";
  for(var i = 0; i < array_length_1d(arr); i++)
  {
    str += string(arr[i]) + ",";
  }
  show_debug_message("hand2 = " + str);
}

// check straight.
is_straight = poker_rank_check_straight(arr);
if(is_straight)
{
  // check royal straight.
  is_royal = true;
  for(var i = 0; i < array_length_1d(arr); i++)
  {
    switch(arr[i])
    {
    case 0:
    case 9:
    case 10:
    case 11:
    case 12:
    case PK_CARD_JOKER:
      break;
    default:
      is_royal = false;
      break;
    }
  }
}

if(is_straight)
{
  if(is_flush)
  {
    if(is_royal)
    {
      return PK_RANK_ROYAL_STRAIGHT_FLUSH;
    }
    return PK_RANK_STRAIGHT_FLUSH;
  }
  return PK_RANK_STRAIGHT;
}
if(is_flush)
{
  return PK_RANK_FLUSH;
}

// same count.
var kinds = 0;
for(var i = 12; i >= 0; i--)
{
  kinds[i] = 0;
}
var nJoker = 0;
var nOne = 0;
for(var i = 0; i < array_length_1d(arr); i++)
{
  var v = arr[i];
  if(v >= PK_CARD_JOKER)
  {
    nJoker++;
  }
  else
  {
    kinds[v]++;
    switch(kinds[v])
    {
    case 2:
      is_one = true;
      nOne++;
      break;
      
    case 3:
      is_three = true;
      nOne--;
      if(nOne == 0)
      {
        is_one = false;
      }
      break;
      
    case 4:
      is_three = false;
      is_four = true;
      break;
    }
  }
}

if(is_four)
{
  if(nJoker == 1)
  {
    return PK_RANK_FIVE_OF_KIND;
  }
  return PK_RANK_FOUR_OF_KIND;
}
else if(is_three)
{
  if(nJoker == 2)
  {
    return PK_RANK_FIVE_OF_KIND;
  }
  if(nJoker == 1)
  {
    return PK_RANK_FOUR_OF_KIND;
  }
  if(is_one)
  {
    return PK_RANK_FULL_HOUSE;
  }
  return PK_RANK_THREE_OF_KIND;
}
else if(is_one)
{
  if(nOne == 2)
  {
    if(nJoker == 1)
    {
      return PK_RANK_FULL_HOUSE;
    }
    return PK_RANK_TWO_PAIR;
  }
  
  if(nJoker == 2)
  {
    return PK_RANK_FOUR_OF_KIND;
  }
  if(nJoker == 1)
  {
    return PK_RANK_THREE_OF_KIND;
  }
  return PK_RANK_ONE_PAIR;
}

if(nJoker == 2)
{
  return PK_RANK_THREE_OF_KIND;
}
if(nJoker == 1)
{
  return PK_RANK_ONE_PAIR;
}
return PK_RANK_HIGH_CARDS;


#define poker_talon_size
/// poker_talon_size(pk);
var pk = argument0;

var talon = ds_map_find_value(pk, "talon");

var count = 0;
for(var i = 0; i < ds_list_size(talon); i++)
{
  var card = ds_list_find_value(talon, i);
  if(card != PK_CARD_INVALID)
  {
    count++;
  }
}

return count;


#define poker_talon_get
/// poker_talon_get(pk, idx);
var pk = argument0;
var idx = argument1;

var talon = ds_map_find_value(pk, "talon");

var index = 0;
for(var i = 0; i < ds_list_size(talon); i++)
{
  var card = ds_list_find_value(talon, i);
  if(card != PK_CARD_INVALID)
  {
    if(idx == index)
    {
      // find.
      return card;
    }
    index++;
  }
}

show_debug_message("Error: poker_talon_get() Not found idx = " + string(idx));
return PK_CARD_INVALID;


#define poker_rank_check_flush
/// poker_rank_check_flush(hand);
var hand = argument0;

// check flush.
var now = -1;
for(var i = 0; i < array_length_1d(hand); i++)
{
  var v = hand[i];
  if(v < PK_CARD_JOKER)
  {
    var suit = floor((v - 1)/13);
    if(now == -1)
    {
      now = suit;
    }
    else
    {
      if(now != suit)
      {
        // different.
        return false;
      }
    }
  }
}

// all the same suit.
return true;


#define poker_rank_check_straight
/// poker_rank_check_straight(hand);
var hand = argument0;

var nJoker = 0;
for(var i = 0; i < array_length_1d(hand); i++)
{
  if(hand[i] >= PK_CARD_JOKER)
  {
    nJoker++;
  }
}

for(var i = 0; i < array_length_1d(hand) - 1; i++)
{
  var v1 = hand[i];
  var v2 = hand[i+1];
  if(v2 >= PK_CARD_JOKER)
  {
    // check complete.
    break;
  }
  
  if(v1 + 1 != v2)
  {
    // not next number.
    if(v1 == 0)
    {
      // check royal straight
      var diff = v2 - 9;
      if(diff >= 0 and diff <= nJoker)
      {
        // royal straight.
        //show_debug_message("use joker = " + string(diff));
        nJoker -= diff;
        continue;
      }
      
      // not straight.
      return false;
    }
    
    if(nJoker > 0)
    {
      //show_debug_message("has joker = " + string(nJoker));
      // has joker.
      var diff = v2 - v1 - 1;
      if(diff >= 0 and diff <= nJoker)
      {
        //show_debug_message("use joker = " + string(diff));
        nJoker -= diff;
      }
      else
      {
        // not straight.
        return false;
      }
    }
    else
    {
      // not straight.
      return false;
    }
  }
}

return true;


#define poker_rank_to_string
/// poker_rank_to_string(rank);
var rank = argument0;

switch(rank)
{
case PK_RANK_HIGH_CARDS: return "HIGH CARD";
case PK_RANK_ONE_PAIR: return "ONE PAIR";
case PK_RANK_TWO_PAIR: return "TWO PAIR";
case PK_RANK_THREE_OF_KIND: return "THREE OF A KIND";
case PK_RANK_STRAIGHT: return "STRAIGHT";
case PK_RANK_FLUSH: return "FLUSH";
case PK_RANK_FULL_HOUSE: return "FULL HOUSE";
case PK_RANK_FOUR_OF_KIND: return "FOUR OF A KIND";
case PK_RANK_STRAIGHT_FLUSH: return "STRAIGHT FLUSH";
case PK_RANK_ROYAL_STRAIGHT_FLUSH: return "ROYAL STRAIGHT FLUSH";
case PK_RANK_FIVE_OF_KIND: return "FIVE OF A KIND";
}

return "NONE";


#define poker_player_count
/// poker_player_count(pk);
var pk = argument0;

var cnt = ds_map_find_value(pk, "player");

return cnt;


#define poker_joker_count
/// poker_joker_count(pk);
var pk = argument0;

var cnt = ds_map_find_value(pk, "joker");

return cnt;


