/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.commons.chain2.impl;

import static org.apache.commons.chain2.testutils.HasCommandCount.hasCommandCount;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertThat;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.chain2.Catalog;
import org.apache.commons.chain2.Command;
import org.apache.commons.chain2.Context;
import org.apache.commons.chain2.testutils.AddingCommand;
import org.apache.commons.chain2.testutils.DelegatingCommand;
import org.apache.commons.chain2.testutils.DelegatingFilter;
import org.apache.commons.chain2.testutils.ExceptionCommand;
import org.apache.commons.chain2.testutils.ExceptionFilter;
import org.apache.commons.chain2.testutils.NonDelegatingCommand;
import org.apache.commons.chain2.testutils.NonDelegatingFilter;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;


/**
 * <p>Test case for the <code>CatalogBase</code> class.</p>
 *
 * @version $Id$
 */
public class CatalogBaseTestCase {


    // ---------------------------------------------------- Instance Variables


    /**
     * The {@link Catalog} instance under test.
     */
    private CatalogBase<String, Object, Context<String, Object>> catalog = null;


    // -------------------------------------------------- Overall Test Methods


    /**
     * Set up instance variables required by this test case.
     */
    @Before
    public void setUp() {
        this.catalog = new CatalogBase<>();
    }

    /**
     * Tear down instance variables required by this test case.
     */
    @After
    public void tearDown() {
        this.catalog = null;
    }


    // ------------------------------------------------ Individual Test Methods


    /**
     * Test adding commands
     */
    @Test
    public void testAddCommand() {
        addCommands();
        assertThat(this.catalog, hasCommandCount(8));
    }


    /**
     * Test getting commands
     */
    @Test
    public void testGetCommand() {

        addCommands();

        assertNotNull(this.catalog.getCommand("AddingCommand"));

        assertNotNull(this.catalog.getCommand("DelegatingCommand"));

        assertNotNull(this.catalog.getCommand("DelegatingFilter"));

        assertNotNull(this.catalog.getCommand("ExceptionCommand"));

        assertNotNull(this.catalog.getCommand("ExceptionFilter"));

        assertNotNull(this.catalog.getCommand("NonDelegatingCommand"));

        assertNotNull(this.catalog.getCommand("NonDelegatingFilter"));

        assertNotNull(this.catalog.getCommand("ChainBase"));
    }

    /**
     * Test pristine instance
     */
    @Test
    public void testPristine() {
        assertThat(this.catalog, hasCommandCount(0));
        assertNull(this.catalog.getCommand("AddingCommand"));
        assertNull(this.catalog.getCommand("DelegatingCommand"));
        assertNull(this.catalog.getCommand("DelegatingFilter"));
        assertNull(this.catalog.getCommand("ExceptionCommand"));
        assertNull(this.catalog.getCommand("ExceptionFilter"));
        assertNull(this.catalog.getCommand("NonDelegatingCommand"));
        assertNull(this.catalog.getCommand("NonDelegatingFilter"));
        assertNull(this.catalog.getCommand("ChainBase"));
    }

    /**
     * Test construction with commands collection
     */
    @Test
    public void testInstantiationWithMapOfCommands() {

        Map<String, Command<String, Object, Context<String, Object>>>
            commands = new ConcurrentHashMap<>();
        commands.put("AddingCommand", new AddingCommand("", null));

        CatalogBase<String, Object, Context<String, Object>> ctlg = new CatalogBase<>(commands);

        assertEquals("Correct command count", 1, ctlg.getCommands().size());
    }

    /**
     * Examine construction with null commands collection
     */
    @Test(expected = IllegalArgumentException.class)
    public void testInstantiationWithNullMapOfCommands() {
        CatalogBase<String, Object, Context<String, Object>> clg = new CatalogBase<>(null);
        assertNull(clg);
    }


    // -------------------------------------------------------- Support Methods


    // Add an interesting set of commands to the catalog
    protected void addCommands() {
        this.catalog.addCommand("AddingCommand", new AddingCommand("", null));
        this.catalog.addCommand("DelegatingCommand", new DelegatingCommand(""));
        this.catalog.addCommand("DelegatingFilter", new DelegatingFilter("", ""));
        this.catalog.addCommand("ExceptionCommand", new ExceptionCommand(""));
        this.catalog.addCommand("ExceptionFilter", new ExceptionFilter("", ""));
        this.catalog.addCommand("NonDelegatingCommand", new NonDelegatingCommand(""));
        this.catalog.addCommand("NonDelegatingFilter", new NonDelegatingFilter("", ""));
        this.catalog.addCommand(
                "ChainBase", new ChainBase<String, Object, Context<String, Object>>());
    }

}
