/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.commons.chain2.web.servlet;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.chain2.web.MapEntry;

/**
 * <p>Private implementation of <code>Map</code> for servlet request
 * name-values[].</p>
 *
 * @version $Id$
 */
final class ServletHeaderValuesMap implements Map<String, String[]>, Serializable {

    /** serialVersionUID */
    private static final long serialVersionUID = 8688063591550062011L;

    private transient HttpServletRequest request = null;

    ServletHeaderValuesMap(final HttpServletRequest req) {
        this.request = req;
    }

    @Override
    public void clear() {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean containsKey(final Object key) {
        return this.request.getHeader(toKey(key)) != null;
    }

    @Override
    public boolean containsValue(final Object value) {
        if (value instanceof String[]) {
            String[] test = (String[]) value;
            for (final String[] actual : values()) {
                if (test.length == actual.length) {
                    boolean matched = true;
                    for (int i = 0; i < test.length; i++) {
                        if (!test[i].equals(actual[i])) {
                            matched = false;
                            break;
                        }
                    }
                    if (matched) {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    @Override
    public Set<Entry<String, String[]>> entrySet() {
        Set<Entry<String, String[]>> set = new HashSet<>();
        for (final String key : Collections.list(this.request.getHeaderNames())) {
            String[] valuesArray = enumerationToArray(this.request.getHeaders(key));
            /* Previously the API was returning an Set<Entry<String, Enumeration<String>>
             * I'm assuming that this was a bug because it violates the contract of how the
             * mapping class behaves. So, I fixed it right here.
             */
            set.add(new MapEntry<>(key, valuesArray, false));
        }
        return set;
    }

    @Override
    public boolean equals(final Object o) {
        if (o != this) {
            if (!ServletHeaderValuesMap.class.isInstance(o)) {
                return false;
            }
            ServletHeaderValuesMap map = (ServletHeaderValuesMap) o;
            return this.request.equals(map.request);
        }
        return true;
    }

    @Override
    public String[] get(final Object key) {
        String[] valuesArray = enumerationToArray(this.request.getHeaders(toKey(key)));
        return valuesArray;
    }

    @Override
    public int hashCode() {
        return this.request.hashCode();
    }

    @Override
    public boolean isEmpty() {
        return size() < 1;
    }

    @Override
    public Set<String> keySet() {
        return new HashSet<>(Collections.list(this.request.getHeaderNames()));
    }

    @Override
    public String[] put(final String key, final String[] value) {
        throw new UnsupportedOperationException();
    }

    @Override
    public void putAll(final Map<? extends String, ? extends String[]> map) {
        throw new UnsupportedOperationException();
    }

    @Override
    public String[] remove(final Object key) {
        throw new UnsupportedOperationException();
    }

    @Override
    public int size() {
        return Collections.list(this.request.getHeaderNames()).size();
    }

    @Override
    public Collection<String[]> values() {
        List<String[]> list = new ArrayList<>();
        for (final String key : Collections.list(this.request.getHeaderNames())) {
            list.add(enumerationToArray(this.request.getHeaders(key)));
        }
        return list;
    }

    /**
     * Simple utility method that converts a string enumeration into a string array.
     * @param values enumeration of strings
     * @return enumeration represented as a string array
     */
    private static String[] enumerationToArray(final Enumeration<String> values) {
        List<String> list = Collections.list(values);
        return list.toArray(new String[list.size()]);
    }

    private String toKey(final Object key) {
        if (key == null) {
            throw new IllegalArgumentException();
        }
        return key.toString();
    }

}
