/*
 * $Id: MockPageContext.java 471754 2006-11-06 14:55:09Z husted $
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.struts.mock;

import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

import javax.el.ELContext;
import javax.servlet.Servlet;
import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import javax.servlet.jsp.JspWriter;
import javax.servlet.jsp.PageContext;
import javax.servlet.jsp.el.ExpressionEvaluator;
import javax.servlet.jsp.el.VariableResolver;
import javax.servlet.jsp.tagext.BodyContent;

/**
 * <p>Mock <strong>ServletContext</strong> object for low-level unit tests of
 * Struts controller components.  Coarser grained tests should be implemented
 * in terms of the Cactus framework, instead of the mock object classes.</p>
 *
 * <p><strong>WARNING</strong> - Only the minimal set of methods needed to
 * create unit tests is provided, plus additional methods to configure this
 * object as necessary.  Methods for unsupported operations will throw
 * <code>UnsupportedOperationException</code>.</p>
 *
 * <p><strong>WARNING</strong> - Because unit tests operate in a single
 * threaded environment, no synchronization is performed.</p>
 *
 * @version $Rev: 471754 $ $Date: 2005-05-07 12:45:39 -0400 (Sat, 07 May 2005)
 *          $
 */
public class MockPageContext extends PageContext {

    // ----------------------------------------------------- Instance Variables

    private ServletContext application = null;
    // Page scope attributes
    private Map<String, Object> attributes = new HashMap<>();
    private ServletConfig config = null;
    private ServletRequest request = null;
    private ServletResponse response = null;
    private HttpSession session = null;
    private boolean throwIOException;
    private boolean returnBodyContent;

    // ----------------------------------------------------------- Constructors

    /**
     * MockPageContext
     */
    public MockPageContext() {
        super();
    }

    /**
     * @param cfg ServletConfig
     * @param req ServletRequest
     * @param res ServletResponse
     */
    public MockPageContext(final ServletConfig cfg, final ServletRequest req,
            final ServletResponse res) {
        super();
        setValues(cfg, req, res);
    }

    /**
     * <p> Construct a new PageContext impl. </p>
     *
     * @param exception Determines if the returned JspWriter should
     *                         throw an IOException on any method call.
     * @param bodyReturn       Determines if getOut() should return a new
     *                         <code>JspWriter</code> or a <code>BodyContent</code>.
     */
    public MockPageContext(final boolean exception, final boolean bodyReturn) {
        this.throwIOException = exception;
        this.returnBodyContent = bodyReturn;
    }

    void checkAndThrow() throws IOException {
        if (this.throwIOException) {
            throw new IOException();
        }
    }

    // --------------------------------------------------------- Public Methods

    /**
     * @param cfg ServletConfig
     * @param req ServletRequest
     * @param res ServletResponse
     */
    public void setValues(final ServletConfig cfg, final ServletRequest req,
            final ServletResponse res) {
        this.config = cfg;

        if (cfg != null) {
            this.application = cfg.getServletContext();
        } else {
            this.application = null;
        }

        this.request = req;
        this.response = res;

        if (req instanceof HttpServletRequest) {
            this.session = ((HttpServletRequest) req).getSession(false);
        } else {
            this.session = null;
        }
    }

    // ---------------------------------------------------- PageContext Methods

    /**
     * @see javax.servlet.jsp.JspContext#findAttribute(java.lang.String)
     */
    @Override
    public Object findAttribute(final String name) {
        Object value = getAttribute(name, PageContext.PAGE_SCOPE);

        if (value == null) {
            value = getAttribute(name, PageContext.REQUEST_SCOPE);
        }

        if (value == null) {
            value = getAttribute(name, PageContext.SESSION_SCOPE);
        }

        if (value == null) {
            value = getAttribute(name, PageContext.APPLICATION_SCOPE);
        }

        return value;
    }

    /**
     * @see javax.servlet.jsp.PageContext#forward(java.lang.String)
     */
    @Override
    public void forward(final String path) {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.jsp.JspContext#getAttribute(java.lang.String)
     */
    @Override
    public Object getAttribute(final String name) {
        return getAttribute(name, PageContext.PAGE_SCOPE);
    }

    /**
     * @see javax.servlet.jsp.JspContext#getAttribute(java.lang.String, int)
     */
    @Override
    public Object getAttribute(final String name, final int scope) {
        Object ret = null;
        if (scope == PageContext.PAGE_SCOPE) {
            ret = this.attributes.get(name);
        } else if (scope == PageContext.REQUEST_SCOPE) {
            if (this.request != null) {
                ret = this.request.getAttribute(name);
            }
        } else if (scope == PageContext.SESSION_SCOPE) {
            if (this.session != null) {
                ret = this.session.getAttribute(name);
            }
        } else if (scope == PageContext.APPLICATION_SCOPE) {
            if (this.application != null) {
                ret = this.application.getAttribute(name);
            }
        } else {
            throw new IllegalArgumentException("Invalid scope " + scope);
        }
        return ret;
    }

    /**
     * @see javax.servlet.jsp.JspContext#getAttributeNamesInScope(int)
     */
    @Override
    public Enumeration<String> getAttributeNamesInScope(final int scope) {
        Enumeration<String> ret = new MockEnumeration<>(Collections.emptyIterator());
        if (scope == PageContext.PAGE_SCOPE) {
            ret = new MockEnumeration<>(this.attributes.keySet().iterator());
        } else if (scope == PageContext.REQUEST_SCOPE) {
            if (this.request != null) {
                ret = this.request.getAttributeNames();
            }
        } else if (scope == PageContext.SESSION_SCOPE) {
            if (this.session != null) {
                ret = this.session.getAttributeNames();
            }
        } else if (scope == PageContext.APPLICATION_SCOPE) {
            if (this.application != null) {
                ret = this.application.getAttributeNames();
            }
        } else {
            throw new IllegalArgumentException("Invalid scope " + scope);
        }
        return ret;
    }

    /**
     * @see javax.servlet.jsp.JspContext#getAttributesScope(java.lang.String)
     */
    @Override
    public int getAttributesScope(final String name) {
        int ret = 0;
        if (this.attributes.get(name) != null) {
            ret = PageContext.PAGE_SCOPE;
        } else if ((this.request != null) && (this.request.getAttribute(name) != null)) {
            ret = PageContext.REQUEST_SCOPE;
        } else if ((this.session != null) && (this.session.getAttribute(name) != null)) {
            ret = PageContext.SESSION_SCOPE;
        } else if ((this.application != null)
            && (this.application.getAttribute(name) != null)) {
            ret = PageContext.APPLICATION_SCOPE;
        }
        return ret;
    }

    /**
     * @see javax.servlet.jsp.PageContext#getException()
     */
    @Override
    public Exception getException() {
        throw new UnsupportedOperationException();
    }

    /**
     * <p> Custom JspWriter that throws the specified exception (supplied on
     * the constructor...if any), else it simply returns. </p>
     * @return JspWriter
     */
    @Override
    public JspWriter getOut() {
        JspWriter jspWriter =
            new JspWriter(0, false) {
                @Override
                public void print(final String s) throws IOException {
                    checkAndThrow();
                }

                @Override
                public void newLine() throws IOException {
                    checkAndThrow();
                }

                @Override
                public void print(final boolean b) throws IOException {
                    checkAndThrow();
                }

                @Override
                public void print(final char c) throws IOException {
                    checkAndThrow();
                }

                @Override
                public void print(final int i) throws IOException {
                    checkAndThrow();
                }

                @Override
                public void print(final long l) throws IOException {
                    checkAndThrow();
                }

                @Override
                public void print(final float f) throws IOException {
                    checkAndThrow();
                }

                @Override
                public void print(final double d) throws IOException {
                    checkAndThrow();
                }

                @Override
                public void print(final char[] s) throws IOException {
                    checkAndThrow();
                }

                @Override
                public void print(final Object obj) throws IOException {
                    checkAndThrow();
                }

                @Override
                public void println() throws IOException {
                    checkAndThrow();
                }

                @Override
                public void println(final boolean x) throws IOException {
                    checkAndThrow();
                }

                @Override
                public void println(final char x) throws IOException {
                    checkAndThrow();
                }

                @Override
                public void println(final int x) throws IOException {
                    checkAndThrow();
                }

                @Override
                public void println(final long x) throws IOException {
                    checkAndThrow();
                }

                @Override
                public void println(final float x) throws IOException {
                    checkAndThrow();
                }

                @Override
                public void println(final double x) throws IOException {
                    checkAndThrow();
                }

                @Override
                public void println(final char[] x) throws IOException {
                    checkAndThrow();
                }

                @Override
                public void println(final String x) throws IOException {
                    checkAndThrow();
                }

                @Override
                public void println(final Object x) throws IOException {
                    checkAndThrow();
                }

                @Override
                public void clear() throws IOException {
                    checkAndThrow();
                }

                @Override
                public void clearBuffer() throws IOException {
                    checkAndThrow();
                }

                @Override
                public void flush() throws IOException {
                    checkAndThrow();
                }

                @Override
                public void close() throws IOException {
                    checkAndThrow();
                }

                @Override
                public int getRemaining() {
                    return 0;
                }

                @Override
                public void write(final char[] cbuf, final int off, final int len)
                        throws IOException {
                    checkAndThrow();
                }
            };

        if (this.returnBodyContent) {
            return new BodyContent(jspWriter) {
                    @Override
                    public Reader getReader() {
                        return null;
                    }

                    @Override
                    public String getString() {
                        return null;
                    }

                    @Override
                    public void writeOut(final Writer out) throws IOException {
                        checkAndThrow();
                    }

                    @Override
                    public void newLine() throws IOException {
                        checkAndThrow();
                    }

                    @Override
                    public void print(final boolean b) throws IOException {
                        checkAndThrow();
                    }

                    @Override
                    public void print(final char c) throws IOException {
                        checkAndThrow();
                    }

                    @Override
                    public void print(final int i) throws IOException {
                        checkAndThrow();
                    }

                    @Override
                    public void print(final long l) throws IOException {
                        checkAndThrow();
                    }

                    @Override
                    public void print(final float f) throws IOException {
                        checkAndThrow();
                    }

                    @Override
                    public void print(final double d) throws IOException {
                        checkAndThrow();
                    }

                    @Override
                    public void print(final char[] s) throws IOException {
                        checkAndThrow();
                    }

                    @Override
                    public void print(final String s) throws IOException {
                        checkAndThrow();
                    }

                    @Override
                    public void print(final Object obj) throws IOException {
                        checkAndThrow();
                    }

                    @Override
                    public void println() throws IOException {
                        checkAndThrow();
                    }

                    @Override
                    public void println(final boolean x) throws IOException {
                        checkAndThrow();
                    }

                    @Override
                    public void println(final char x) throws IOException {
                        checkAndThrow();
                    }

                    @Override
                    public void println(final int x) throws IOException {
                        checkAndThrow();
                    }

                    @Override
                    public void println(final long x) throws IOException {
                        checkAndThrow();
                    }

                    @Override
                    public void println(final float x) throws IOException {
                        checkAndThrow();
                    }

                    @Override
                    public void println(final double x) throws IOException {
                        checkAndThrow();
                    }

                    @Override
                    public void println(final char[] x) throws IOException {
                        checkAndThrow();
                    }

                    @Override
                    public void println(final String x) throws IOException {
                        checkAndThrow();
                    }

                    @Override
                    public void println(final Object x) throws IOException {
                        checkAndThrow();
                    }

                    @Override
                    public void clear() throws IOException {
                        checkAndThrow();
                    }

                    @Override
                    public void clearBuffer() throws IOException {
                        checkAndThrow();
                    }

                    @Override
                    public void close() throws IOException {
                        checkAndThrow();
                    }

                    @Override
                    public int getRemaining() {
                        return 0;
                    }

                    @Override
                    public void write(final char[] cbuf, final int off, final int len)
                            throws IOException {
                        checkAndThrow();
                    }
                };
        }

        return jspWriter;
    }

    /**
     * @see javax.servlet.jsp.PageContext#getPage()
     */
    @Override
    public Object getPage() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.jsp.PageContext#getRequest()
     */
    @Override
    public ServletRequest getRequest() {
        return this.request;
    }

    /**
     * @see javax.servlet.jsp.PageContext#getResponse()
     */
    @Override
    public ServletResponse getResponse() {
        return this.response;
    }

    /**
     * @see javax.servlet.jsp.PageContext#getServletConfig()
     */
    @Override
    public ServletConfig getServletConfig() {
        return this.config;
    }

    /**
     * @see javax.servlet.jsp.PageContext#getServletContext()
     */
    @Override
    public ServletContext getServletContext() {
        return this.application;
    }

    /**
     * @see javax.servlet.jsp.PageContext#getSession()
     */
    @Override
    public HttpSession getSession() {
        return this.session;
    }

    /**
     * @see javax.servlet.jsp.PageContext#handlePageException(java.lang.Exception)
     */
    @Override
    public void handlePageException(final Exception e) {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.jsp.PageContext#handlePageException(java.lang.Throwable)
     */
    @Override
    public void handlePageException(final Throwable t) {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.jsp.PageContext#include(java.lang.String)
     */
    @Override
    public void include(final String path) {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.jsp.PageContext#initialize(javax.servlet.Servlet,
     * javax.servlet.ServletRequest,
     * javax.servlet.ServletResponse, java.lang.String, boolean, int, boolean)
     */
    @Override
    public void initialize(final Servlet servlet, final ServletRequest req,
            final ServletResponse res, final String errorPageURL, final boolean needsSession,
            final int bufferSize, final boolean autoFlush) {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.jsp.JspContext#popBody()
     */
    @Override
    public JspWriter popBody() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.jsp.PageContext#pushBody()
     */
    @Override
    public BodyContent pushBody() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.jsp.PageContext#release()
     */
    @Override
    public void release() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see javax.servlet.jsp.JspContext#removeAttribute(java.lang.String)
     */
    @Override
    public void removeAttribute(final String name) {
        int scope = getAttributesScope(name);
        if (scope != 0) {
            removeAttribute(name, scope);
        }
    }

    /**
     * @see javax.servlet.jsp.JspContext#removeAttribute(java.lang.String, int)
     */
    @Override
    public void removeAttribute(final String name, final int scope) {
        if (scope == PageContext.PAGE_SCOPE) {
            this.attributes.remove(name);
        } else if (scope == PageContext.REQUEST_SCOPE) {
            if (this.request != null) {
                this.request.removeAttribute(name);
            }
        } else if (scope == PageContext.SESSION_SCOPE) {
            if (this.session != null) {
                this.session.removeAttribute(name);
            }
        } else if (scope == PageContext.APPLICATION_SCOPE) {
            if (this.application != null) {
                this.application.removeAttribute(name);
            }
        } else {
            throw new IllegalArgumentException("Invalid scope " + scope);
        }
    }

    /**
     * @see javax.servlet.jsp.JspContext#setAttribute(java.lang.String, java.lang.Object)
     */
    @Override
    public void setAttribute(final String name, final Object value) {
        setAttribute(name, value, PageContext.PAGE_SCOPE);
    }

    /**
     * @see javax.servlet.jsp.JspContext#setAttribute(java.lang.String, java.lang.Object, int)
     */
    @Override
    public void setAttribute(final String name, final Object value, final int scope) {
        if (scope == PageContext.PAGE_SCOPE) {
            this.attributes.put(name, value);
        } else if (scope == PageContext.REQUEST_SCOPE) {
            if (this.request != null) {
                this.request.setAttribute(name, value);
            }
        } else if (scope == PageContext.SESSION_SCOPE) {
            if (this.session != null) {
                this.session.setAttribute(name, value);
            }
        } else if (scope == PageContext.APPLICATION_SCOPE) {
            if (this.application != null) {
                this.application.setAttribute(name, value);
            }
        } else {
            throw new IllegalArgumentException("Invalid scope " + scope);
        }
    }

    /**
     * @see javax.servlet.jsp.PageContext#include(java.lang.String, boolean)
     */
    @Override
    public void include(final String arg0, final boolean arg1)
            throws ServletException, IOException {
        return;
    }

    /**
     * @see javax.servlet.jsp.JspContext#getExpressionEvaluator()
     * @deprecated deprecated
     */
    @Deprecated
    @Override
    public ExpressionEvaluator getExpressionEvaluator() {
        return null;
    }

    /**
     * @see javax.servlet.jsp.JspContext#getVariableResolver()
     * @deprecated deprecated
     */
    @Deprecated
    @Override
    public VariableResolver getVariableResolver() {
        return null;
    }

    /**
     * @see javax.servlet.jsp.JspContext#getELContext()
     */
    @Override
    public ELContext getELContext() {
        return null;
    }
}
