# Mekiku Viewer
#     A portable viewer for Mekiku communication

# TODO:
#     * send C
#     * send U if F9 is pressed
#     * attach logs of Cs once in three Hs
#     * support Android
#     * support iOS

import socket
import ipaddress
import time
import threading
import sys

import netifaces

import japanize_kivy

from kivy.app import App
from kivy.clock import Clock
from kivy.graphics import Color, Rectangle
from kivy.uix.boxlayout import BoxLayout
from kivy.uix.scrollview import ScrollView
from kivy.lang import Builder
from kivy.utils import platform

import IMETextInput


if(platform == "android"):
    from jnius import autoclass


class MekikuLines:
    def __init__(self, val, command=None, group=None):
        if(isinstance(val, str)):
            self.firstline = MekikuFirstLine(val, command, group)
            self.s = self.firstline.s
            self.loglines = None
            return
        elif(isinstance(val, bytes)):
            self.s = val.decode("utf-16-le")
        else:
            raise TypeError("MekikuLines can take bytes or str, not {}"
                            .format(type(val)))
        lines = self.s.splitlines()
        try:
            self.firstline = MekikuFirstLine(lines[0])
        except MekikuValueError as e:
            raise MekikuValueError("Error in the first line: {}".format(e))
        try:
            self.loglines = MekikuLogLines(lines[1:])
        except MekikuValueError as e:
            self.loglines = MekikuLogLines([])
            print("Error(s) in log lines:", e)

    def add_loglines(self, lls):
        if(self.loglines):
            raise Exception("cannot add loglines: already has one")
        if(isinstance(lls, MekikuLogLines)):
            self.loglines = lls
            self.s += str(lls)
        else:
            raise TypeError("not a MekikuLogLines: {}".format(type(lls)))

    def __str__(self):
        return self.s


class MekikuFirstLine:
    def __init__(self, val, command=None, group=None):
        if(command and group):
            self._create(val, command, group)
            return
        self.s = str(val)
        if len(val) < 14:
            raise MekikuValueError("too short: {}".format(val))
        self.group = val[0]
        self.timestamp = int(val[1:9])
        self.command = val[9]
        rawlen = int(val[10:14])
        if(self.command in "LHRMUF"):
            self.bytes = rawlen
            self.chars = int(rawlen / 2)
        elif(self.command in "DC"):
            self.bytes = rawlen * 2
            self.chars = rawlen
        else:
            raise MekikuValueError("unrecognized command '{}'"
                                   .format(self.command))
        self.data = val[14:15 + self.chars]
        if(self.command == "U"):
            udata = self.data.split(",")
            self.u_timestamp = str(int(udata[0])).zfill(8)
            # self.u_group = udata[1]
            self.u_command = udata[2]
            self.u_ip = udata[3]

    def _create(self, t, command, group):
        self.group = str(group)
        self.s = self.group
        self.timestamp = str(int(time.time() * 1000)).zfill(8)[-8:]
        self.s += self.timestamp
        self.command = str(command)
        self.s += self.command
        self.chars = len(str(t))
        self.bytes = self.chars * 2  # every UTF-16 char has 2bytes
        if(self.command in "LHRMF"):  # U is not supported
            self.s += str(self.bytes).zfill(4)
        elif(self.command in "DC"):
            self.s += str(self.chars).zfill(4)
        else:
            raise MekikuValueError("invalid command: {}".format(self.command))
        self.s += str(t)

    def __str__(self):
        return self.s


class MekikuLogLines:
    def __init__(self, lines):
        err = []
        self._list = []
        for line in lines:
            try:
                ll = MekikuLogLine(line)
                self._list.append(ll)
            except MekikuValueError as e:
                err.append(e)
        if(len(err)):
            em = ""
            for i in err:
                em += str(type(i)) + str(i) + "\n"
            raise MekikuValueError(em)

    def __iter__(self):
        self._cur = 0
        return self

    def __next__(self):
        if(self._cur < len(self._list)):
            self._cur += 1
            return self._list[self._cur - 1]
        else:
            raise StopIteration

    def append(self, ll):
        if(isinstance(ll, MekikuLogLine)):
            self._list.append(ll)
        else:
            raise TypeError("can only append MekikuLogLine, not {}"
                            .format(type(ll)))

    def __str__(self):
        s = ""
        for l in self._list:
            s += "\n" + str(l)
        return s


class MekikuLogLine:
    #  0- 7: remote ip address in hex
    #  8-15: remote timestamp
    # 16-23: local timestamp
    #    24: C or D
    #    25: flag for deleted
    # 26-29: number of characters
    # 30-  : data
    def __init__(self, val, remote_ip=None, remote_timestamp=None,
                 local_timestamp=None, command=None, deleted=None):
        if(command):
            self._create(val, remote_ip, remote_timestamp,
                         local_timestamp, command, deleted)
            return
        if(len(val) < (8 + 8 + 8 + 1 + 1 + 4)):
            MekikuValueError("logline too short: {}".format(val))
        self.s = val
        ip_hex = self.s[0:8]
        self.remote_ip = str(ipaddress.IPv4Address(int(ip_hex, 16)))
        self.remote_timestamp = self.s[8:16]
        self.local_timestamp = self.s[16:24]
        self.command = self.s[24]
        if(self.command in "DC"):
            self.deleted = int(self.s[25])  # 0 for normal, 1 for deleted
            mlen = int(self.s[26:30])
            self.data = self.s[30:31 + mlen]
        else:
            raise MekikuValueError("neither C nor D: {}".format(self.s))

    def _create(self, val, remote_ip, remote_timestamp,
                local_timestamp, command, deleted):
        self.remote_ip = remote_ip
        self.s = hex(int(ipaddress.IPv4Address(remote_ip)))[2:].zfill(8)
        self.remote_timestamp = remote_timestamp
        self.s += self.remote_timestamp
        self.local_timestamp = local_timestamp
        self.s += self.local_timestamp
        self.command = command
        self.s += self.command
        self.deleted = int(deleted)
        self.s += ("1" if deleted else "0")
        self.data = str(val)
        self.s += str(len(self.data)).zfill(4) + self.data

    def __str__(self):
        return self.s


class MekikuValueError(ValueError):
    # errors which you can simply ignore and go on
    pass


Builder.load_string("""
<RootWidget>:
    orientation: "vertical"

    MainView:
        id: mainview
        size_hint: 1, 0.7

    MonitorView:
        id: monitorview
        size_hint: 1, 0.2

    BoxLayout:
        size_hint: 1, 0.1
        orientation: "horizontal"
        ComView:
            id: comview
            size_hint: 0.6, 1
        Spinner:
            size_hint: 0.1, 1
            text: "ch"
            on_text: root.on_ch(self.text)
            values: [str(i) for i in range(1, 13)]
        Spinner:
            size_hint: 0.1, 1
            text: "gr"
            on_text: root.on_gr(self.text)
            values: [chr(i) for i in range(65, 65+26)]

        BoxLayout:
            size_hint: 0.2, 1
            orientation: "vertical"
            Spinner:
                size_hint: 1, 0.5
                text: "START"
                values: list(root.addrs())
                background_normal: ""
                background_color: [1, 0, 0, 1]
                on_text:
                    BN = "atlas://data/images/defaulttheme/button"
                    self.background_normal = BN
                    self.background_color = [1, 1, 1, 1]
                    root.on_net(self.text)
            IMETextInput:
                id: hostname
                size_hint: 1, 0.5
                multiline: False
                hint_text: "name"
                text_language: "ja"
                on_text_validate: root.on_name_enter(self.text)

<MainView>:
    Label:
        id: label
        size_hint: None, None
        size: self.texture_size
        text_size: root.width, None
        halign: "left"
        valign: "top"
        scroll_y: 0
        IMETextInput:
            id: textinput
            text_language: "ja"
            padding: [10, 5, 10, 5]
            multiline: False
            height: self.minimum_height
            width: root.width
            on_text_validate: root.on_enter(self.text)
            text_validate_unfocus: False

<MonitorView>:
    Label:
        id: label
        size_hint: None, None
        size: self.texture_size
        text_size: root.width, None
        halign: "left"
        valign: "top"

<ComView>:
    Label:
        id: label
        size_hint: None, None
        size: self.texture_size
        text_size: root.width, None
        halign: "left"
        valign: "top"
        # this would be better if it were a textinput
        Widget:
            id: combottom

<SpinnerOption>
    size_hint: 1, None
    height: "30dp"
    font_size: "12sp"
""")


class MainView(ScrollView):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.text = ""

    def on_kv_post(self, __):
        self.ids.label.text = "Mekiku Viewer\n\n\n"
        self.ids.label.texture_update()
        self.ids.textinput.bind(text=self.on_text)

    def add_text(self, t):
        self.text += "\n" + t
        self.ids.label.text = self.text + "\n\n"  # hack
        self.ids.label.texture_update()
        self.scroll_to(self.ids.textinput)

    def delete_latest(self):
        self.text = self.text.rsplit("\n", 1)[0]
        self.ids.label.text = self.text + "\n\n"  # hack
        self.ids.label.texture_update()
        self.scroll_to(self.ids.textinput)

    def on_enter(self, t):
        print("enter", t)
        if(not self.parent.sendnet):
            self.ids.textinput.text = "(set up the net first)"
            return
        msg = self.parent.createlines("D", t)
        self.parent.push(msg)
        self.ids.textinput.text = ""

    def on_text(self, instance, t):
        self.parent.mtext = t

    def reset_text(self):
        self.ids.textinput.text = ""
        self.ids.textinput.focus = True


class MonitorView(ScrollView):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.text = ""

    def on_kv_post(self, __):
        Clock.schedule_interval(self.update, 1.0)

    def update(self, dt):
        discon = self.parent.update_monitor(False)
        with self.canvas.before:
            if(discon):
                Color(0.4, 0, 0, mode="rgb")
            else:
                Color(0, 0.2, 0.2, mode="rgb")
            Rectangle(pos=self.pos, size=self.size)

    def add_text(self, t):
        self.text += t + "\n"

    def flush(self):
        if(len(self.text) == 0):
            self.text = "Monitor"
        self.ids.label.text = self.text
        self.text = ""


class ComView(ScrollView):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.text = ""
        self.last_text = ""

    def on_kv_post(self, __):
        Clock.schedule_interval(self.update, 1.0)

    def update(self, dt):
        self.parent.parent.update_com(False)

    def add_text(self, t):
        self.text += t + "\n"

    def flush(self):
        if(len(self.text) == 0):
            self.text = "Staff Communication"
        if(self.text != self.last_text):
            self.ids.label.text = self.text     # scroll up to the top
            self.scroll_to(self.ids.combottom)  # and down to the bottom
            self.last_text = self.text
        self.text = ""


class RootWidget(BoxLayout):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        if(platform == "android"):
            pyact = autoclass("org.kivy.android.PythonActivity")
            activity = pyact.mActivity
            context = autoclass("android.content.Context")
            self.wifi = activity.getSystemService(context.WIFI_SERVICE)

    def on_kv_post(self, __):
        self.ch = 1
        self.group = "A"
        self.sendnet = None
        self.sendpool = []
        self.socket = None
        self.HB_THRESHOLD = 10.0
        self.mtext = ""
        self.last_mtext = ""
        self.greet = False
        # self.rth = None
        # self.rth_stop = True
        self.th = None
        self.th_stop = True
        self.sth = None
        self.sth_stop = True
        # self.ith = None
        # self.ith_stop = True
        # self.iptalk_req = []
        self.ids.hostname.bind(text=self.on_name_change)
        Clock.schedule_interval(self.heartbeat, 2.0)

    def opensock(self, s, p):
        if((not s) or (s.fileno() < 0)):
            s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
            s.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
            s.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)
            s.settimeout(0.7)
            s.bind((self.myip, p))
            print("socket opened:", s)
        return s

    def on_ch(self, t):
        if(self.ch == int(t)):
            return
        self.ch = int(t)
        if(not self.sendnet):
            return
        self.stop_threads()
        self.start_threads()

    def on_gr(self, t):
        if(self.group == t):
            return
        self.group = t
        if(not self.sendnet):
            return
        self.stop_threads()
        self.start_threads()

    def addrs(self):
        d = {}
        for i in netifaces.interfaces():
            try:
                a = netifaces.ifaddresses(i)[netifaces.AF_INET][0]
            except KeyError:  # e.g. IPv6-only
                continue
            if(a["addr"] == "127.0.0.1"):
                continue
            if("broadcast" in a.keys()):
                d[a["addr"]] = a["broadcast"]
        return d

    def on_net(self, t):
        print("my ip:", t)
        self.myip = t
        print("bcast:", self.addrs()[t])
        self.sendnet = self.addrs()[t]
        # the only point to start the threads,
        # but the threads may start over here, too
        self.stop_threads()
        self.start_threads()
        self.ids.mainview.reset_text()

    # for the sake of UI
    # apply the change without an enter
    def on_name_change(self, instance, t):
        if((not t) or (len(t) == 0)):
            t = self.hostname = "viewer-" + str(time.time())
        self.on_name_enter(t)

    def on_name_enter(self, t):
        self.hostname = t

    # ??? tell me a best practice for these threads
    def stop_threads(self):
        # if(self.rth):
        #     self.rth_stop = True
        #     self.rth.join()
        # if(self.ith):
        #     self.ith_stop = True
        #     self.ith.join()
        if(self.sth):
            self.sth_stop = True
            self.sth.join()
        if(self.th):
            self.th_stop = True
            self.th.join()

    def start_threads(self):
        # self.rth = threading.Thread(target=self.reqloop)
        # self.rth_stop = False
        # self.rth.start()
        self.th = threading.Thread(target=self.listenloop)
        self.th_stop = False
        self.th.start()
        self.sth = threading.Thread(target=self.sendloop)
        self.sth_stop = False
        self.sth.start()
        # self.ith = threading.Thread(target=self.iptalkloop)
        # self.ith_stop = False
        # self.ith.start()

    def listenloop(self):
        self.ddic = {}  # (D)     ts_ip: (time, data)
        self.cdic = {}  # (C)     ts_ip: (time, data)
        self.tdic = {}  # (any)   ip: (ts, time)
        self.adic = {}  # (L/H/R) ip: name
        self.mdic = {}  # (M)     ip: data
        if(platform == "android"):
            self.multicast_lock = self.wifi.createMulticastLock("mekiku")
            self.multicast_lock.acquire()
        while(not self.th_stop):
            modified = False
            self.socket = self.opensock(self.socket, 6632 + self.ch)
            try:
                u16msg, (self.ip, port) = self.socket.recvfrom(4096)
            except socket.timeout:
                continue
            except Exception as e:
                print("interrupted:", e)
                break
            try:
                ml = MekikuLines(u16msg)
            except MekikuValueError as e:
                print("ignoring an error:", e)
                continue
            fl = ml.firstline
            ll = ml.loglines
            if(fl.group != self.group):
                continue
            self.tdic[self.ip] = (fl.timestamp, time.time())
            if(fl.command == "L"):
                if(self.ip != self.myip):
                    self.greet = True
                self.adic[self.ip] = fl.data
                self.mdic[self.ip] = ""
                modified = True
            elif(fl.command in "HR"):
                if((fl.command == "R") and (self.ip != self.myip)):
                    self.greet = True
                newalias = fl.data
                if(self.ip in self.adic.keys()):
                    oldalias = self.adic[self.ip]
                    if(newalias != oldalias):
                        self.adic[self.ip] = newalias
                        modified = True
                else:
                    self.adic[self.ip] = newalias
                    if(not (self.ip in self.mdic.keys())):
                        self.mdic[self.ip] = ""
                    modified = True
                modified |= self.readlog(ll)
            elif(fl.command == "M"):
                self.mdic[self.ip] = fl.data
                modified = True
                self.readlog(ll)
            elif(fl.command == "D"):
                key = str(fl.timestamp).zfill(8) + "_" + self.ip
                self.ddic[key] = (time.time(), fl.data)
                self.ids.mainview.add_text(fl.data)
                modified = True
                self.readlog(ll)
            elif(fl.command == "C"):
                key = str(fl.timestamp).zfill(8) + "_" + self.ip
                self.cdic[key] = (time.time(), fl.data)
                modified = True
                self.readlog(ll)
            elif(fl.command == "U"):
                if(fl.u_command == "D"):
                    key = fl.u_timestamp + "_" + fl.u_ip
                    if(key in self.ddic.keys()):
                        # U support is limited to the latest entry
                        last_d = sorted(self.ddic.items(),
                                        key=lambda t: t[1][0])[-1]
                        if(last_d[0] == key):
                            self.ids.mainview.delete_latest()
                        self.ddic[key] = (0, self.ddic[key][1])
                        modified = True
                    else:
                        print("tried to undo something not done yet: ", fl)
                modified |= self.readlog(ll)
            elif(fl.command == "F"):
                for d in [self.mdic, self.adic, self.tdic]:
                    try:
                        d.pop(self.ip)
                    except KeyError:
                        pass
                modified = True
                self.readlog(ll)
            if(modified):
                print("-------")
                # sort by time and print the latest 4 items
                for i in sorted(self.ddic.items(), key=lambda t: t[1][0])[-4:]:
                    ival = i[1]
                    if(ival[0]):
                        print(ival[1])
                self.update_monitor(True)
                self.update_com(True)
        if(platform == "android"):
            self.multicast_lock.release()
            self.multicast_lock = None

    def update_monitor(self, debug=False):
        if(not self.sendnet):
            mm = "Network has not been set up yet"
            self.ids.monitorview.add_text(mm)
            self.ids.monitorview.flush()
            return True  # paint it red
        discon = False
        for i in sorted(self.mdic.items()):
            iip, ival = i
            ian = iip
            if(iip in self.adic.keys()):
                ian = self.adic[iip]
            idelta = 0
            if(iip in self.tdic.keys()):
                idelta = int(time.time() - self.tdic[iip][1])
            if(idelta > self.HB_THRESHOLD):
                ian += " (" + str(idelta) + ")"
                discon = True
            mm = "[" + ian + "] " + ival
            if(debug):
                print(mm)
            self.ids.monitorview.add_text(mm)
        self.ids.monitorview.flush()
        return discon

    def update_com(self, debug=False):
        if(not self.sendnet):
            cm = "Select your network ->"
            self.ids.comview.add_text(cm)
            self.ids.comview.flush()
            return
        for i in sorted(self.cdic.items(), key=lambda t: t[1][0])[-4:]:
            its, ival = i
            cm = "! " + ival[1]
            if(debug):
                print(cm)
            self.ids.comview.add_text(cm)
        self.ids.comview.flush()

    def heartbeat(self, dt):
        if(not self.sendnet):
            return
        # not implemented:
        #     self.hb_count += 1
        #     self.hb_count %= 3
        #     if(not self.hb_count):
        #         if len(self.cdic):
        #             use_CLines_for_LogLines_instead_of_DLines
        self.push(self.createlines("H", self.hostname))
        if(self.greet):
            time.sleep(0.01)
            self.push(self.createlines("R", self.hostname))
            self.greet = False
        if(self.mtext != self.last_mtext):
            time.sleep(0.01)  # hack
            self.push(self.createlines("M", self.mtext))
            self.last_mtext = self.mtext

    def farewell(self):
        if(not self.sendnet):
            return
        self.push(self.createlines("F", self.hostname))
        # iptalk compat
        # fs = None
        # for ip in self.adic.keys():
        #     fs = self.opensock(fs, 6624 + (self.ch * 100))
        #     self.send_oneshot(6624, self.myip + ",", fs, ip)
        time.sleep(0.9)  # hack

    def createlines(self, command, t):
        ml = MekikuLines(t, command, self.group)
        mlls = MekikuLogLines([])
        # sort by time and iterate the latest 5
        # note: this viewer doesn't issue any Cs
        for i in sorted(self.ddic.items(),
                        key=lambda t: -t[1][0])[:5]:
            rts_ip, lts_data = i
            lts = int(lts_data[0] * 1000)
            if(lts):  # deleted items have zero timestamp
                mll = MekikuLogLine(lts_data[1], command="D", deleted=0,
                                    remote_ip=rts_ip[9:],
                                    remote_timestamp=rts_ip[:8],
                                    local_timestamp=str(lts).zfill(8)[-8:])
                mlls.append(mll)
        ml.add_loglines(mlls)
        return str(ml)

    def push(self, t):
        if(not self.sendnet):
            print("no broadcast address specified")
            return
        self.sendpool.append(t)

    def sendloop(self):
        self.push(self.createlines("L", self.hostname))
        time.sleep(0.015)  # hack
        starting_m = self.createlines("M", "")
        self.push(starting_m)
        self.push(starting_m)
        while(not self.sth_stop):
            if(len(self.sendpool)):
                self.socket = self.opensock(self.socket, 6632 + self.ch)
                self.send_sock_port(self.sendpool.pop(0).encode("utf-16-le"),
                                    self.socket, 6632 + self.ch)
            time.sleep(0.05)  # hack

    def send_sock_port(self, b, s, p, dest_ip=None):
        if(not dest_ip):
            dest_ip = self.sendnet
        try:
            c = s.sendto(b, (dest_ip, p))
        except Exception as e:
            print("sending aborted:", e)
            return 0
        return c

    # iptalk compatibility
    """
    def send_oneshot(self, baseport, txt, sock, dest_ip=None):
        p = baseport + (self.ch * 100)
        oneshot = False if sock else True
        sock = self.opensock(sock, p)
        self.send_sock_port(txt.encode("shift_jis"), sock, p, dest_ip)
        if(oneshot):
            sock.close()

    def iptalkloop(self):
        p = 6618
        waitingforans = []
        s = None
        while(not self.ith_stop):
            s = self.opensock(s, p + (self.ch * 100))
            if(len(self.iptalk_req)):
                d = self.iptalk_req.pop()
                t = "{},{},{},Ans".format(self.myip, d, self.hostname)
                self.send_oneshot(p, t, s, d)
                waitingforans.append(d)
                continue
            try:
                sjis, (ip, rp) = s.recvfrom(8192)
            except socket.timeout:
                continue
            except Exception as e:
                print("interrupted:", e)
                break
            m = sjis.decode("shift_jis")
            ml = m.split(",")
            try:
                ml_from_ip = ml[0]
                ml_to_ip = ml[1]
                ml_from_name = ml[2]
                ml_command = ml[3]
            except IndexError as e:
                print("broken ans data:", e)
                continue
            if(ml_from_ip == self.myip):
                continue
            if(ml_from_ip != ip):
                print("fake ip:", ml_from_ip, "from", ip)
                continue
            elif(ml_to_ip == self.myip):
                if(ml_command == "Ans"):
                    if(ml_from_ip in waitingforans):
                        waitingforans.remove(ml_from_ip)
                        # hmm... i'd rather ignore iptalk compat...
                        newalias = ml_from_name
                        if(ml_from_ip in self.adic.keys()):
                            oldalias = self.adic[ml_from_ip]
                            if(newalias != oldalias):
                                self.adic[ml_from_ip] = newalias
                        else:
                            self.adic[ml_from_ip] = newalias
                            if(not (ml_from_ip in self.mdic.keys())):
                                self.mdic[ml_from_ip] = ""
                    else:
                        t = "{},{},{},Ans".format(self.myip, ml_from_ip,
                                                  self.hostname)
                        self.send_oneshot(p, t, s, ml_from_ip)
                        # iptalk compat:
                        # send null to the monitor, and close the socket
                        self.send_oneshot(6621, self.myip + ",", None,
                                          ml_from_ip)
                        waitingforans.append(ml_from_ip)
                else:
                    print("don't know how to handle other than Ans:", m)
            else:
                # just ignore messages for others
                pass

    def reqloop(self):
        p = 6622
        s = self.opensock(None, p + (self.ch * 100))
        self.send_oneshot(p, self.myip + "," + self.hostname + ",Req,1", s)
        while(not self.rth_stop):
            s = self.opensock(s, p + (self.ch * 100))
            try:
                sjis, (ip, rp) = s.recvfrom(8192)
            except socket.timeout:
                continue
            except Exception as e:
                print("interrupted:", e)
                break
            m = sjis.decode("shift_jis")
            ml = m.split(",")
            try:
                ml_from_ip = ml[0]
                # ml_from_name = ml[1]
                ml_command = ml[2]
                ml_arg = ml[3]
            except IndexError as e:
                print("broken req data:", e)
                continue
            if(ml_from_ip == self.myip):
                continue
            if(ml_from_ip != ip):
                print("fake ip:", ml_from_ip, "from", ip)
                continue
            if(ml_command == "Req"):
                if(ml_arg == "1"):
                    self.iptalk_req.append(ml_from_ip)
                else:
                    print("don't know how to handle other than Req,1:", m)
            else:
                print("don't know how to handle other than Req:", m)
    """

    def readlog(self, lines):
        modified = False
        recovered = []
        for line in lines:
            key = line.remote_timestamp + "_" + line.remote_ip
            if(line.command == "D"):
                if(key in self.ddic.keys()):
                    if(self.ddic[key][1] == line.data):
                        if((line.deleted) and (self.ddic[key][0] != 0)):
                            self.ddic[key] = (0, line.data)
                            modified = True
                        continue
                    else:
                        print("either was invalid")
                        print("cache:", self.ddic[key])
                        print("input:", line.data)
                        if(line.deleted):
                            self.ddic[key] = (0, line.data)
                        else:
                            self.ddic[key] = (self.ddic[key][0], line.data)
                        modified = True
                        continue
                else:
                    if(line.deleted):
                        print("an unknown D is reported to have been deleted:")
                        print(line)
                        continue
                    print("recovered data:", line)
                    # calculate the time from the timestamp
                    #
                    # remote timestamps would raise an exception
                    # when a mekiku resends an old message
                    # previously sent from another mekiku which
                    # has not been registered to TDIC:
                    #   delta = (int(line.remote_timestamp)
                    #            - int(self.tdic[line.remote_ip][0])) / 1000
                    #
                    # therefore use local ones
                    delta = (int(line.local_timestamp)
                             - int(self.tdic[self.ip][0])) / 1000
                    # timestamps can overflow
                    # e.g. lts==99999999 and tdic[ip][0]==00000000
                    if(delta > 50000.000):
                        delta -= 100000.000
                    lt = self.tdic[self.ip][1] + delta
                    self.ddic[key] = (lt, line.data)
                    recovered.append(line.data)
                    print(key, ":", lt, line.data)
                    modified = True
            elif(line.command == "C"):
                if(key in self.cdic.keys()):
                    if(self.cdic[key][1] == line.data):
                        continue
                    else:
                        print("either was invalid")
                        print("cache:", self.cdic[key])
                        print("input:", line.data)
                        self.cdic[key] = (self.cdic[key][0], line.data)
                        modified = True
                        continue
                else:
                    print("recovered data:", line)
                    delta = (int(line.local_timestamp)
                             - int(self.tdic[self.ip][0])) / 1000
                    if(delta > 50000.000):
                        delta -= 100000.000
                    lt = self.tdic[self.ip][1] + delta
                    self.cdic[key] = (lt, line.data)
                    print(key, ":", lt, line.data)
                    modified = True
            else:
                print("other than C or D in cache line: ", line)
        for rd in reversed(recovered):
            self.ids.mainview.add_text(rd)
        return modified


class MekikuViewer(App):
    def build(self):
        return RootWidget()

    def on_stop(self):
        self.root.farewell()
        self.root.stop_threads()
        if(platform == "android"):
            if(self.root.multicast_lock):
                self.root.multicast_lock.release()

    def license(self):
        japanize_kivy.show_license()


MekikuViewer().run()
sys.exit()
