package jp.gr.java_conf.wutse.purewiki;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.gr.java_conf.wutse.purewiki.event.WikiSelectionEvent;
import jp.gr.java_conf.wutse.purewiki.event.WikiSelectionListener;
import jp.gr.java_conf.wutse.purewiki.exception.PureWikiValidateException;

public class PureWikiManager {
	private PureWikiEngine engine = null;
	private PureWikiData currentData = null;
	private List<PureWikiData> dataList = null;
	private Map<String, PureWikiData> dataMap = null;

	private List<WikiSelectionListener> selectionListenerList = new ArrayList<>();

	/**
	 * コンストラクタ
	 */
	public PureWikiManager() {
		engine = new PureWikiEngine();
		changeCurrentData(engine.getConfig().getDefaultWiki());
	}

	public PureWikiEngine getEngine() {
		return engine;
	}

	public void setEngine(PureWikiEngine engine) {
		this.engine = engine;
	}

	public PureWikiData getCurrentData() {
		return currentData;
	}

	public void createTemporaryData() {
		changeCurrentData(engine.createTemporaryWikiData(""));
	}

	/**
	 * 指定したwikiを選択する。
	 * 
	 * @param dataName
	 */
	public boolean changeCurrentData(String wikiName) {
		PureWikiData wikiData = getPureWikiData(wikiName);
		if (wikiData == null) {
			wikiData = engine.createTemporaryWikiData(wikiName);
		}
		return changeCurrentData(wikiData);
	}

	public void addWikiSelectionListener(WikiSelectionListener listener) {
		if (!selectionListenerList.contains(listener)) {
			selectionListenerList.add(listener);
		}
	}

	/**
	 * 指定したwikiを選択する。
	 * 
	 * @param dataName
	 */
	public boolean changeCurrentData(PureWikiData nextData) {
		if (nextData == null) {
			nextData = engine.createTemporaryWikiData("");
		}
		WikiSelectionEvent event = new WikiSelectionEvent(currentData, nextData);

		for (WikiSelectionListener listener : selectionListenerList) {
			listener.wikiChanging(event);
		}

		// もしキャンセルされていたら実施しない。
		if (event.isCanceled()) {
			return false;
		}

		// 新規でなく、未ロードであれば、ロードする。
		if (nextData != null && !nextData.isNew() && !nextData.isLoaded()) {
			engine.loadPureWikiData(nextData, false);
		}

		this.currentData = nextData;

		for (WikiSelectionListener listener : selectionListenerList) {
			listener.wikiChanged(event);
		}

		return true;
	}

	public PureWikiData getPureWikiData(String wikiName) {
		if (dataMap == null) {
			loadSrcDir();
		}
		return dataMap.get(wikiName);
	}

	public String loadWikiText() {
		engine.loadPureWikiData(currentData, true);
		return currentData.getWikiText();
	}

	public void storeWiki() throws PureWikiValidateException {
		storeWiki(currentData);
	}

	public void storeWiki(PureWikiData wikiData) throws PureWikiValidateException {
		try {
			boolean isNew = wikiData.isNew();
			engine.storePureWikiData(wikiData);
			engine.translateWiki(wikiData);
			if (isNew) {
				for (int i = 0; i < dataList.size(); i++) {
					if (wikiData.getWikiName().compareToIgnoreCase(dataList.get(i).getWikiName()) < 0) {
						dataList.add(i, wikiData);
						break;
					}
				}
				if (!dataList.contains(wikiData)) {
					dataList.add(wikiData);
				}
				dataMap.put(wikiData.getWikiName(), wikiData);
			}
		}
		catch (IOException ex) {

		}
	}

	/**
	 * 指定したwikiを削除する。
	 * 
	 * @param dataName
	 */
	public void removeWiki(String wikiName) {
		removeWiki(getPureWikiData(wikiName));
	}

	/**
	 * 指定したwikiを削除する。
	 * 
	 * @param dataName
	 */
	public void removeWiki(PureWikiData wikiData) {
		if (dataList.contains(wikiData)) {
			dataList.remove(wikiData);
			dataMap.remove(wikiData.getWikiName());
		}
		engine.removePureWikiData(wikiData);
		if (currentData == wikiData) {
			changeCurrentData(engine.getConfig().getDefaultWiki());
		}
	}

	public void exit() {
		engine.storePureWikiConfig();
	}

	public void translateWiki() throws IOException {
		translateWiki(currentData);
	}

	public void translateWiki(PureWikiData wikiData) throws IOException {
		engine.translateWiki(wikiData);
	}

	public void clearPureWikiDataList() {
		dataList = null;
		dataMap = null;
	}

	public List<PureWikiData> getPureWikiDataList() {
		if (dataList == null) {
			loadSrcDir();
		}
		List<PureWikiData> wikiDataList = new ArrayList<>();
		for (PureWikiData wikiData : dataList) {
			wikiDataList.add(wikiData);
		}
		return wikiDataList;
	}

	/**
	 * srcディレクトリ以下のwikiリストをロードします。
	 */
	private void loadSrcDir() {
		dataList = engine.getWikiDataList();
		dataMap = new HashMap<String, PureWikiData>();
		for (PureWikiData wikiData : dataList) {
			dataMap.put(wikiData.getWikiName(), wikiData);
		}
	}

	public void removePureWikiData(String wikiName) {
		PureWikiData wikiData = getPureWikiData(wikiName);
		// 実ファイル削除
		engine.removePureWikiData(wikiData);

		// キャッシュから削除
		dataList.remove(wikiData.getWikiName());
		dataMap.remove(wikiData.getWikiName());
	}

}
