/*
 * CAN Target Simulator
 * for M16C/5M family + NC30WA v5.45
 *
 * [ CAN ]
 */

#include "common.h"
#include "target.h"
#include "debug.h"
#include "sfr5m.h"
#include "timer.h"
#include "log.h"
#include "led.h"
#include "list.h"
#include "can.h"

/*
 * CAN
 * p[^ݒ
 * ݂16TqATv|Cg75%Arbg[g125kbps/250kbps/500kbps/1Mbpsz肵Ă
 */
#define CAN_BITRATE					(500)
										/* 500kbps */
#define CAN_SS						(1)
										/* 1Tq(ŒASS+TSEG+TSEG2=16) */
#define CAN_TSEG1					(11)
										/* 4Tq-16Tq */
#define CAN_TSEG2					(4)
										/* 2Tq-8Tq */
#define CAN_SJW						(1)
										/* 1Tq-4Tq */
#define CAN_TQ						(CAN_SS + CAN_TSEG1 + CAN_TSEG2)
										/* 1rbg^CTQ */

/*
 * CAN
 * G[|[OԊu
 * ܂葬ƗƂrbgclearedƌoĂ܂߁A߂Ɏ
 */
#define CAN_ERR_POLL_INTERVAL		(10)
										/* 10msɃ|[O */

/*
 * CAN
 * `l
 */
static can_channel_info can_channel;
										/* `l\ */

/*
 * CAN
 * {[[g
 */
static void can_init_baudrate(void)
{
	uint16 bcrl;
	uint8 bcrhl;

	/* p[^ݒG[̌o */
#if (CAN_TQ != 8) && (CAN_TQ != 16)
#error CAN_TQ must be 8 or 16
#endif /* CAN_TQ */
#if CAN_SS != 1
#error CAN_SS must be 1
#endif /* CAN_SS */
#if (CAN_TSEG1 < 4) || (CAN_TSEG1 > 16)
#error CAN_TSEG1 must be in the range between 4 to 16
#endif /* CAN_TSEG1 */
#if (CAN_TSEG2 < 2) || (CAN_TSEG2 > 8)
#error CAN_TSEG2 must be in the range between 2 to 8
#endif /* CAN_TSEG2 */
#if (CAN_SJW < 1) || (CAN_SJW > 4)
#error CAN_SJW must be in the range between 1 to 4
#endif /* CAN_SJW */
#if CAN_TSEG1 <= CAN_TSEG2
#error CAN_TSEG2 must be greater than CAN_TSEG1
#endif /* CAN_TSEG1,CAN_TSEG2 */
#if CAN_TSEG2 < CAN_SJW
#error CAN_TSEG2 must be equal or greater than CAN_SJW
#endif /* CAN_SJW, CAN_TSEG2 */
#if (CAN_SJW == 1) && (CAN_TSEG2 < 2)
#error CAN_TSEG2 must be equal or greater than 2 in case of CAN_SJW=1
#endif /* CAN_SJW, CAN_TSEG2 */

	/* rbgRtBM[VWX^̐ݒ(BRP, TSEG1) */
	bcrl = (uint16)((32000 / (CAN_BITRATE * CAN_TQ)) - 1);
	bcrl |= (uint16)((CAN_TSEG1 - 1) << 12);
	c0bcrll = (uint8)bcrl;
	c0bcrlh = (uint8)(bcrl >> 8);

	/* rbgRtBM[VWX^̐ݒ(TSEG2, SJW) */
	bcrhl = (uint8)(CAN_TSEG2 - 1);
	bcrhl |= (uint8)((CAN_SJW - 1) << 4);
	c0bcrhl = bcrhl;

	/* NbNIWX^̐ݒ:NbN\[XƂPLLNbNI */
	 c0clkr = 0x00;
}

/*
 * CAN
 * 荞ݏ
 * 荞݋֎~ŌĂ΂
 */
static void can_init_intr(void)
{
	uint16 mierh;

	/* [{bNX荞݋WX^ */
	mierh = 0x0000;

	/* MFIFO荞݋:荞݋ */
	mierh |= 0x0100;

	/* MFIFO荞ݔ^C~O:MɂđMFIFOɂȂƂ */
	mierh |= 0x0200;

	/* MFIFO荞݋:荞݋ */
	mierh |= 0x1000;

	/* MFIFO荞ݔ^C~O:MɂĎMFIFOobt@[jOɂȂƂ */
	mierh |= 0x2000;

	/* ʏ탁[{bNXׂ͂Ċ荞݋֎~ */
	c0mierl = 0x0000;

	/* FIFO[{bNX͊荞݋ */
	c0mierh = mierh;

	/* EFCNAbv荞݂֎~ */
	c0wic = 0x00;

	/* G[荞݂UART3M荞݂ƃxN^L邽߁Aׂċ֎~ */
	c0eier = 0x00;

	/* M荞݂֎~ */
	c0ric = 0x00;

	/* M荞݂֎~ */
	c0tic = 0x00;

	/* MFIFO荞݂֎~ */
	c0fric = 0x00;

	/* MFIFO荞݂֎~ */
	c0ftic = 0x00;

	/* G[荞݂UART3M荞݂ƌp̂ߐݒ肵Ȃ */
}

/*
 * CAN
 * }XN
 */
static void can_init_mask(void)
{
	/* }XNWX^:W[{bNXSID,EIDrȂ */
	c0mkr0l = 0x0000;
	c0mkr0h = 0x0000;
	c0mkr1l = 0x0000;
	c0mkr1h = 0x0000;
	c0mkr2l = 0x0000;
	c0mkr2h = 0x0000;
	c0mkr3l = 0x0000;
	c0mkr3h = 0x0000;
	c0mkr4l = 0x0000;
	c0mkr4h = 0x0000;
	c0mkr5l = 0x0000;
	c0mkr5h = 0x0000;

	/* }XNWX^:FIFO[{bNXSID,EIDrȂ */
	c0mkr6l = 0x0000;
	c0mkr6h = 0x0000;
	c0mkr7l = 0x0000;
	c0mkr7h = 0x0000;

	/* }XNWX^:ׂẴ}XNL */
	c0mkivlrl = 0x0000;
	c0mkivlrh = 0x0000;

	/* FIFOMrWX^0:ׂẴf[^t[M */
	c0fidcr0l = 0x0000;
	c0fidcr0h = 0x0000;

	/* FIFOMrWX^1:ׂẴf[^t[M */
	c0fidcr1l = 0x0000;
	c0fidcr1h = 0x0000;
}

/*
 * CAN
 * Zbg[h
 * 荞݋֎~ŌĂ΂
 */
static void can_init_reset(void)
{
	uint16 ctlr;
	uint8 pd;

	/* X[v[h */
	slpm_c0ctlr = 0;

	/* X[v[h܂ő҂ */
	cpu_nop();
	while (slpst_c0str == 1)
	{
		cpu_nop();
	}

	/* Zbg[h֑Jڂ */
	ctlr = c0ctlr;
	ctlr &= (uint16)(~0x0003);
	ctlr |= 0x0001;
	c0ctlr = ctlr;

	/* Zbg[hɑJڂ܂ő҂ */
	cpu_nop();
	while ((c0strl & 0x07) != 0x01)
	{
		cpu_nop();
	}

	/* ēxA䃌WX^ǂݒ */
	ctlr = c0ctlr;

	/* oXItA[h:ISO11898-1dl */
	ctlr &= (uint16)(~0x0018);

	/* CAN[{bNX[h:FIFO[{bNX[h */
	ctlr |= 0x0100;

	/* IDtH[}bg[h:WID[h */
	ctlr &= (uint16)(~0x0060);

	/* bZ[WXg[h:I[o[[h */
	ctlr |= 0x0800;

	/* MD揇ʃ[h:IDD摗M[h(ISO11898-1dl) */
	ctlr &= (uint16)(~0x1000);

	/* ^CX^vvXP[:8rbg^C */
	ctlr |= 0xc000;

	/* CAN|[g:CANo͂Ƃċ@\ */
	ctlr |= 0x0080;

	/* ps͂̕Ƃ(P9_2CRX0pAP9_3CTX1p) */
	pd = pd9;
	pd &= (uint8)(~0x0c);

	/* PRC2(veNgrbg)"1"() */
	prc2 = 1;

	/* PD9WX^PRC2rbgŕی삳Ă */
	pd9 = pd;

	/* 䃌WX^ݒ */
	c0ctlr = ctlr;

	/* G[R[hi[WX^:ݐσ[h */
	edpm_c0ecsr = 1;

	/* CANoX^C~Oƃ{[[g̐ݒ */
	can_init_baudrate();

	/* 荞݃WX^̐ݒ */
	can_init_intr();

	/* }XNWX^̐ݒ */
	can_init_mask();
}

/*
 * CAN
 * Halt[h
 */
static void can_init_halt(uint8 testmode)
{
	uint16 ctlr;

	ASSERT(testmode <= 0x03);

	/* Halt[h֑Jڂ */
	ctlr = c0ctlr;
	ctlr &= (uint16)(~0x0003);
	ctlr |= 0x0002;
	c0ctlr = ctlr;

	/* Halt[hɑJڂ܂ő҂ */
	cpu_nop();
	while ((c0strl & 0x07) != 0x02)
	{
		cpu_nop();
	}

	/* eXg[h */
	if (testmode != 0x00)
	{
		/* eXg䃌WX^:eXg[h */
		c0tcr = (uint8)((testmode << 1) | 0x01);
	}
	else
	{
		/* eXg䃌WX^:eXg[h֎~ */
		c0tcr = 0x00;
	}
}

/*
 * CAN
 * Iy[V[h
 */
static void can_init_oper(void)
{
	uint16 ctlr;
	uint8 rfcr;
	uint8 NEAR *mctl;
	uint8 loop;

	/* Iy[V[h֑Jڂ */
	ctlr = c0ctlr;
	ctlr &= (uint16)(~0x0003);
	c0ctlr = ctlr;

	/* Iy[V[hɑJڂ܂ő҂ */
	cpu_nop();
	while ((c0strl & 0x07) != 0x00)
	{
		cpu_nop();
	}

	/* bZ[W䃌WX^:ʏ탁[{bNX */
	mctl = &c0mctl0;
	for (loop=0; loop<24; loop++)
	{
		*mctl++ = 0x00;
	}

	/* MFIFO䃌WX^ǂݍ */
	rfcr = c0rfcr;

	/* MFIFObZ[WXg:XgȂ */
	rfcr &= (uint8)(~0x10);

	/* MFIFO: */
	rfcr |= 0x01;

	/* MFIFO䃌WX^֏ */
	c0rfcr = rfcr;

	/* MFIFO䃌WX^(MFIFO̂) */
	tfe_c0tfcr = 1;

	/* MFIFO荞݂ */
	c0fric = INTR_DEF_LEVEL;
}

/*
 * CAN
 * PDU
 */
static void can_init_pdu(can_channel_info NEAR *channel)
{
	uint8 loop;

	/* PDUׂď */
	for (loop=0; loop<_countof(channel->pdu); loop++)
	{
		/* oXg */
		list_init((list_node NEAR *)&(channel->pdu[loop]));

		/* [{bNXNo. */
		channel->pdu[loop].mb = loop;

		/*  */
		channel->pdu[loop].state = CAN_STATE_NONE;
	}

	/* empty͐擪PDUɃ|Cg */
	channel->empty = &(channel->pdu[0]);

	/* 1ȍ~ǉ */
	for (loop=1; loop<_countof(channel->pdu); loop++)
	{
		list_insert((list_node NEAR *)&(channel->pdu[0]), (list_node NEAR *)&(channel->pdu[loop]));
	}

	/* MPDU݂͑Ȃ(NULL) */
	channel->tx = NULL;

	/* MPDU݂͑Ȃ(NULL) */
	channel->rx = NULL;
}

/*
 * CAN
 * G[񏉊
 */
static void can_init_err(can_error_info NEAR *err)
{
	/* o */
	err->be = 0;
	err->ew = 0;
	err->ep = 0;
	err->boe = 0;
	err->bor = 0;
	err->ror = 0;
	err->ol = 0;
	err->bl = 0;
	err->se = 0;
	err->fe = 0;
	err->ae = 0;
	err->ce = 0;
	err->be1 = 0;
	err->be0 = 0;
	err->ade = 0;

	/* t[^C} */
	err->freerun = timer_freerun();

	/* G[荞ݗvWX^ */
	err->eifr = c0eifr;

	/* G[R[hi[WX^ */
	err->ecsr = (uint8)(c0ecsr & 0x7f);
}

/*
 * CAN
 * Xe[^X񏉊
 */
static void can_init_status(can_status_info *stat)
{
	/* o */
	stat->ep = 0;
	stat->bo = 0;
	stat->nml = 0;
	stat->fml = 0;

	/* Xe[^XWX^ */
	stat->str = c0str;
}

/*
 * CAN
 * 
 */
void can_init(uint8 test_mode)
{
	/* Zbg[h */
	can_init_reset();

	/* Halt[h */
	can_init_halt(test_mode);

	/* Iy[V[h */
	can_init_oper();

	/* PDU */
	can_init_pdu(&can_channel);

	/* G[񏉊 */
	can_init_err(&can_channel.error);

	/* Xe[^X񏉊 */
	can_init_status(&can_channel.status);
}

/*
 * CAN
 * X[v
 */
void can_sleep(void)
{
	uint16 ctlr;

	/* Zbg[h֑Jڂ */
	ctlr = c0ctlr;
	ctlr &= (uint16)(~0x0003);
	ctlr |= 0x0001;
	c0ctlr = ctlr;

	/* Zbg[hɑJڂ܂ő҂ */
	cpu_nop();
	while ((c0strl & 0x03) != 0x01)
	{
		cpu_nop();
	}

	/* X[v[h֑Jڂ */
	slpm_c0ctlr = 1;

	/* X[v[hɑJڂ܂ő҂ */
	cpu_nop();
	while (slpst_c0str == 0)
	{
		cpu_nop();
	}
}

/*
 * CAN
 * PDU擾
 * ׂĂPDUgp̏ꍇNULLԂ
 */
static can_pdu NEAR *can_pdu_get(void)
{
	uint8 flg;
	can_pdu NEAR *pdu;

	/* 荞݋֎~ */
	flg = cpu_di();

	/* empty1ȏ̃Gg */
	if (can_channel.empty != NULL)
	{
		/* ߂l݂͌empty */
		pdu = can_channel.empty;

		/* emptyXV */
		can_channel.empty = (can_pdu NEAR *)list_get_next((list_node NEAR *)can_channel.empty);

		/* Opdu */
		list_init((list_node NEAR *)pdu);

		/* ԂȂ */
		pdu->state = CAN_STATE_NONE;
	}
	else
	{
		/* ߂lNULL */
		pdu = NULL;
	}

	/* 荞ݕA */
	cpu_ei(flg);

	return pdu;
}

/*
 * CAN
 * PDU֒ǉ
 */
static void can_pdu_free(can_pdu NEAR *pdu)
{
	uint8 flg;

	ASSERT(pdu != NULL);

	/* 荞݋֎~ */
	flg = cpu_di();

	/* ԂȂ */
	pdu->state = CAN_STATE_NONE;

	/* empty1ȏ̃Gg */
	if (can_channel.empty != NULL)
	{
		/* empty̒ɑ} */
		list_insert((list_node NEAR *)can_channel.empty, (list_node NEAR *)pdu);
	}
	else
	{
		/* oXg */
		list_init((list_node NEAR *)pdu);

		/* emptyƂĐݒ */
		can_channel.empty = pdu;
	}

	/* 荞ݕA */
	cpu_ei(flg);
}

/*
 * CAN
 * MPDU擾
 * ׂĂPDUgp̏ꍇNULLԂ
 */
can_pdu NEAR *can_tx_get(void)
{
	/* PDU擾 */
	return can_pdu_get();
}

/*
 * CAN
 * M|[O
 */
static void can_send_poll(void)
{
	can_pdu NEAR *pdu;
	can_pdu NEAR *next;
	uint8 mctl;

	/* MPDU𓾂 */
	pdu = can_channel.tx;

	/* NULLłȂΒTs */
	while (pdu != NULL)
	{
		/* PDU擾Ă */
		next = (can_pdu NEAR *)list_get_next((list_node NEAR *)pdu);

		/* ԂM[{bNXi[ł */
		if (pdu->state == CAN_STATE_TX_MB)
		{
			/* bZ[W䃌WX^擾 */
			mctl = *(&c0mctl0 + pdu->mb);

			/* SENTDATA, TRMABT̂ݎc */
			mctl &= 0x05;

			/* SENTDATA=1őMATRMABT=1őMA{[g */
			if (mctl != 0)
			{
				if ((mctl & 0x01) != 0)
				{
					/* SENTDATA=1:Ԃ𑗐MɕύX */
					pdu->state = CAN_STATE_TX_COMPLETED;
				}
				else
				{
					/* TRMABT=1:Ԃ𑗐MA{[gɕύX */
					pdu->state = CAN_STATE_TX_ABORTED;
				}

				/* PDU؂藣 */
				next = (can_pdu NEAR *)list_remove((list_node NEAR *)pdu);

				/* can_channel.txpduvĂ΁AAčXV */
				if (can_channel.tx == pdu)
				{
					can_channel.tx = next;
				}

				/* PDU֒ǉ */
				can_pdu_free(pdu);
			}
		}

		/* nextT */
		pdu = next;
	}
}

/*
 * CAN
 * MvPDU[{bNX
 */
static void can_send_mb(void)
{
	can_pdu NEAR *pdu;
	volatile union CANMB_DEF NEAR *mailbox;
	uint8 loop;
	uint8 NEAR *mctl;

	/* MPDU𓾂 */
	pdu = can_channel.tx;

	/* NULLłȂΒTs */
	while (pdu != NULL)
	{
		/* ԂMvłΑM */
		if (pdu->state == CAN_STATE_TX_REQ)
		{
			/* bZ[W䃌WX^̃AhX𓾂 */
			mctl = (&c0mctl0) + pdu->mb;

			/* bZ[W䃌WX^NA */
			while (*mctl != 0x00)
			{
				/* 0x00 */
				*mctl = 0x00;

				/* ҂ */
				cpu_nop();
			}

			/* [{bNX̃AhX𓾂 */
			mailbox = &c0mb0_addr + pdu->mb;

			/* gprbgNA */
			mailbox->word[0] = 0x0000;
			mailbox->word[1] = 0x0000;
			mailbox->word[2] = 0x0000;

			/* DATA */
			for (loop=0; loop<8; loop++)
			{
				mailbox->mc.data[loop] = pdu->data[loop];
			}

			/* DLC */
			mailbox->mc.dlc = pdu->dlc;

			/* SID */
			mailbox->ml.sid = pdu->sid;

			/* Mv(TRMREQ=1) */
			*mctl = 0x80;

			/* Ԃ𑗐M[{bNXi[Ƃ */
			pdu->state = CAN_STATE_TX_MB;

			/* LED_ */
			led_oneshot(LED_CAN_TX, LED_ONESHOT_MS);
		}

		/* PDU擾 */
		pdu = (can_pdu NEAR *)list_get_next((list_node NEAR *)pdu);
	}
}

/*
 * CAN
 * M^XN
 */
static void can_task_send(void)
{
	/* M|[O */
	can_send_poll();

	/* oXItԂłȂ */
	if (bost_c0str == 0)
	{
		/* MvPDU[{bNX */
		can_send_mb();
	}
}

/*
 * CAN
 * Mv
 */
BOOL can_tx_req(can_pdu NEAR *pdu)
{
	ASSERT(pdu != NULL);
	ASSERT(pdu->state == CAN_STATE_NONE);

	/* Ԃ𑗐Mvɐݒ */
	pdu->state = CAN_STATE_TX_REQ;

	/* MPDU֒ǉ */
	if (can_channel.tx != NULL)
	{
		/* tx̖ɒǉ */
		list_add((list_node NEAR *)can_channel.tx, (list_node NEAR *)pdu);
	}
	else
	{
		/* oXg */
		list_init((list_node NEAR *)pdu);

		/* txƂĐݒ */
		can_channel.tx = pdu;
	}

	/* M^XNĂ */
	can_task_send();
}

/*
 * CAN
 * [{bNXMPDU
 */
static void can_recv_mb(can_pdu NEAR *pdu)
{
	uint8 loop;

	ASSERT(pdu != NULL);

	/* [{bNX28擾(SID) */
	pdu->sid = (uint16)sid_c0mb28;

	/* [{bNX28擾(DLC) */
	pdu->dlc = dlc_c0mb28;

	/* [{bNX28擾(DATA) */
	for (loop=0; loop<8; loop++)
	{
		pdu->data[loop] = c0mb28.mc.data[loop];
	}

	/* M */
	pdu->state = CAN_STATE_RX_COMPLETED;

	/* MPDU֒ǉ */
	if (can_channel.rx != NULL)
	{
		/* rx̖ɒǉ */
		list_add((list_node NEAR *)can_channel.rx, (list_node NEAR *)pdu);
	}
	else
	{
		/* oXg */
		list_init((list_node NEAR *)pdu);

		/* rxƂĐݒ */
		can_channel.rx = pdu;
	}
}

/*
 * CAN
 * M^XN
 */
static void can_task_recv(void)
{
	uint8 flg;
	can_pdu NEAR *pdu;
	BOOL recv;

	/* MȂ */
	recv = FALSE;

	/* 荞݋֎~ */
	flg = cpu_di();

	/* MFIFOɖǃbZ[WAbZ[WMs */
	while (rfest_c0rfcr == 0)
	{
		/* M */
		recv = TRUE;

		/* MFIFObZ[WXgĂ΃NA */
		/* (MFIFObZ[WXg̓Xe[^X^XNŃJEg) */
		if (rfmlf_c0rfcr == 1)
		{
			rfmlf_c0rfcr = 0;
		}

		/* PDU擾 */
		pdu = can_pdu_get();

		/* PDU擾ł */
		if (pdu != NULL)
		{
			/* 擾ł:[{bNXMPDU */
			can_recv_mb(pdu);
		}

		/* MFIFOi߂ */
		c0rfpcr = 0xff;
	}

	/* 荞ݕA */
	cpu_ei(flg);

	/* M΁ALEDVbg_ */
	if (recv == TRUE)
	{
		flg = cpu_di();
		led_oneshot(LED_CAN_RX, LED_ONESHOT_MS);
		cpu_ei(flg);
	}
}

/*
 * CAN
 * MPDU擾
 * MPDU݂ȂꍇNULLԂ
 */
can_pdu NEAR *can_rx_get(void)
{
	uint8 flg;
	can_pdu NEAR *pdu;

	/* 荞݋֎~ */
	flg = cpu_di();

	/* MPDU̐擪Ԃ */
	pdu = can_channel.rx;

	/* 荞ݕA */
	cpu_ei(flg);

	return pdu;
}

/*
 * CAN
 * MPDU
 */
void can_rx_free(can_pdu NEAR *pdu)
{
	uint8 flg;
	can_pdu NEAR *next;

	ASSERT(pdu != NULL);

	/* 荞݋֎~ */
	flg = cpu_di();

	/* PDU؂藣 */
	next = (can_pdu NEAR *)list_remove((list_node NEAR *)pdu);

	/* can_channel.rxpduvĂ΁AAčXV */
	if (can_channel.rx == pdu)
	{
		can_channel.rx = next;
	}

	/* PDU֒ǉ */
	can_pdu_free(pdu);

	/* 荞ݕA */
	cpu_ei(flg);
}

/*
 * CAN
 * G[^XN(荞ݗv背WX^)
 */
static void can_task_eifr(can_error_info NEAR *err)
{
	uint8 eifr;
	uint8 eifr_xor;

	/* G[荞ݗv背WX^ǂݏo */
	eifr = c0eifr;

	/* otOꊇŃNA */
	c0eifr = (uint8)~eifr;

	/* OlƂ̍o */
	eifr_xor = eifr ^ err->eifr;

	/* oXG[otO */
	if ((eifr_xor & 0x01) != 0)
	{
		if ((eifr & 0x01) != 0)
		{
			/* rbg 0->1 */
			err->be++;
			LOG_U2("[CAN]detect bus error", err->be);
		}
		else
		{
			/* rbg 1->0 */
			LOG_U2("[CAN]clear  bus error", err->be);
		}
	}

	/* G[[jOotO */
	if ((eifr_xor & 0x02) != 0)
	{
		if ((eifr & 0x02) != 0)
		{
			/* rbg 0->1 */
			err->ew++;
			LOG_U2("[CAN]detect error warning", err->ew);
		}
		else
		{
			/* rbg 1->0 */
			LOG_U2("[CAN]clear  error warning", err->ew);
		}
	}

	/* G[pbVuotO */
	if ((eifr_xor & 0x04) != 0)
	{
		/* G[pbVuotO */
		if ((eifr & 0x04) != 0)
		{
			/* rbg 0->1 */
			err->ep++;
			LOG_U2("[CAN]detect error passive", err->ep);
		}
		else
		{
			/* rbg 1->0 */
			LOG_U2("[CAN]clear  error passive", err->ep);
		}
	}

	/* oXItJnotO */
	if ((eifr_xor & 0x08) != 0)
	{
		if ((eifr & 0x08) != 0)
		{
			/* rbg 0->1 */
			err->boe++;
			LOG_U2("[CAN]detect enter bus-off", err->boe);
		}
		else
		{
			/* rbg 1->0 */
			LOG_U2("[CAN]clear  enter bus-off", err->boe);
		}
	}

	/* oXItAotO */
	if ((eifr_xor & 0x10) != 0)
	{
		if ((eifr & 0x10) != 0)
		{
			/* rbg 0->1 */
			err->bor++;
			LOG_U2("[CAN]detect restore bus-off", err->bor);
		}
		else
		{
			/* rbg 1->0 */
			LOG_U2("[CAN]clear  restore bus-off", err->bor);
		}
	}

	/* MI[ootO */
	if ((eifr_xor & 0x20) != 0)
	{
		if ((eifr & 0x20) != 0)
		{
			/* rbg 0->1 */
			err->ror++;
			LOG_U2("[CAN]detect receive overrun", err->ror);
		}
		else
		{
			/* rbg 1->0 */
			LOG_U2("[CAN]clear  receive overrun", err->ror);
		}
	}

	/* I[o[ht[MotO */
	if ((eifr_xor & 0x40) != 0)
	{
		if ((eifr & 0x40) != 0)
		{
			/* rbg 0->1 */
			err->ol++;
			LOG_U2("[CAN]detect send overload", err->ol);
		}
		else
		{
			/* rbg 1->0 */
			LOG_U2("[CAN]clear  send overload", err->ol);
		}
	}

	/* oXbNotO */
	if ((eifr_xor & 0x80) != 0)
	{
		if ((eifr & 0x80) != 0)
		{
			/* rbg 0->1 */
			err->bl++;
			LOG_U2("[CAN]detect bus lock", err->bl);
		}
		else
		{
			/* rbg 1->0 */
			LOG_U2("[CAN]clear  bus lock", err->bl);
		}
	}

	/* ̒lL */
	err->eifr = eifr;
}

/*
 * CAN
 * G[^XN(G[R[hi[WX^)
 */
static void can_task_ecsr(can_error_info NEAR *err)
{
	uint8 ecsr;
	uint8 ecsr_xor;

	/* G[R[hi[WX^ǂݏo */
	ecsr = c0ecsr;
	ecsr &= 0x7f;

	/* otOꊇăNA */
	c0ecsr = (uint8)~ecsr;

	/* OlƂ̍o */
	ecsr_xor = ecsr ^ err->ecsr;

	/* X^btG[otO */
	if ((ecsr_xor & 0x01) != 0)
	{
		if ((ecsr & 0x01) != 0)
		{
			/* rbg 0->1 */
			err->se++;
			LOG_U2("[CAN]detect stuff error", err->se);
		}
		else
		{
			/* rbg 1->0 */
			LOG_U2("[CAN]clear  stuff error", err->se);
		}
	}

	/* tH[G[otO */
	if ((ecsr_xor & 0x02) != 0)
	{
		if ((ecsr & 0x02) != 0)
		{
			/* rbg 0->1 */
			err->fe++;
			LOG_U2("[CAN]detect form error", err->fe);
		}
		else
		{
			/* rbg 1->0 */
			LOG_U2("[CAN]clear  form error", err->fe);
		}
	}

	/* ACKG[otO */
	if ((ecsr_xor & 0x04) != 0)
	{
		if ((ecsr & 0x04) != 0)
		{
			/* rbg 0->1 */
			err->ae++;
			LOG_U2("[CAN]detect ack error", err->ae);
		}
		else
		{
			/* rbg 1->0 */
			LOG_U2("[CAN]clear  ack error", err->ae);
		}
	}

	/* CRCG[otO */
	if ((ecsr_xor & 0x08) != 0)
	{
		if ((ecsr & 0x08) != 0)
		{
			/* rbg 0->1 */
			err->ce++;
			LOG_U2("[CAN]detect crc error", err->ce);
		}
		else
		{
			/* rbg 1->0 */
			LOG_U2("[CAN]clear  crc error", err->ce);
		}
	}

	/* rbgG[(ZVu)otO */
	if ((ecsr_xor & 0x10) != 0)
	{
		if ((ecsr & 0x10) != 0)
		{
			/* rbg 0->1 */
			err->be1++;
			LOG_U2("[CAN]detect bit error(recessive)", err->be1);
		}
		else
		{
			/* rbg 1->0 */
			LOG_U2("[CAN]clear  bit error(recessive)", err->be1);
		}
	}

	/* rbgG[(h~ig)otO */
	if ((ecsr_xor & 0x20) != 0)
	{
		if ((ecsr & 0x20) != 0)
		{
			/* rbg 0->1 */
			err->be0++;
			LOG_U2("[CAN]detect bit error(dominant)", err->be0);
		}
		else
		{
			/* rbg 1->0 */
			LOG_U2("[CAN]clear  bit error(dominant)", err->be0);
		}
	}

	/* ACKf~^G[otO */
	if ((ecsr_xor & 0x40) != 0)
	{
		if ((ecsr & 0x40) != 0)
		{
			/* rbg 0->1 */
			err->ade++;
			LOG_U2("[CAN]detect ack delimiter error", err->ade);
		}
		else
		{
			/* rbg 1->0 */
			LOG_U2("[CAN]clear  ack delimiter error", err->ade);
		}
	}

	/* ̒lL */
	err->ecsr = ecsr;
}

/*
 * CAN
 * G[^XN
 */
static void can_task_error(can_error_info NEAR *err)
{
	uint16 freerun;
	uint16 diff;

	/* t[^C}擾 */
	freerun = timer_freerun();

	/* O񎞊ԂƂ̍ */
	diff = (uint16)(freerun - err->freerun);

	/* ȏ゠ */
	if (diff >= CAN_ERR_POLL_INTERVAL)
	{
		/* ̎Ԃɐi߂ */
		err->freerun += CAN_ERR_POLL_INTERVAL;

		/* 荞ݗv背WX^ */
		can_task_eifr(err);

		/* G[R[hi[WX^ */
		can_task_ecsr(err);
	}
}

/*
 * CAN
 * Xe[^X^XN
 */
static void can_task_status(can_status_info NEAR *stat)
{
	uint16 str;
	uint16 str_xor;

	/* Xe[^XWX^ǂݏo */
	str = c0str;

	/* svȃrbg}XN */
	str &= 0x3018;

	/* OlƂ̍o */
	str_xor = str ^ stat->str;

	/* G[pbVuXe[^XtO */
	if ((str_xor & 0x0008) != 0)
	{
		if ((str & 0x0008) != 0)
		{
			/* rbg 0->1 */
			stat->ep++;
			LOG_U2("[CAN]detect error passive status", stat->ep);
		}
		else
		{
			/* rbg 1->0 */
			LOG_U2("[CAN]clear  error passive status", stat->ep);
		}
	}

	/* oXItXe[^XtO */
	if ((str_xor & 0x0010) != 0)
	{
		if ((str & 0x0010) != 0)
		{
			/* rbg 0->1 */
			stat->bo++;
			LOG_U2("[CAN]detect bus-off status", stat->bo);
		}
		else
		{
			/* rbg 1->0 */
			LOG_U2("[CAN]clear  bus-off status", stat->bo);
		}
	}

	/* ʏ탁[{bNX bZ[WXgotO */
	if ((str_xor & 0x1000) != 0)
	{
		if ((str & 0x1000) != 0)
		{
			/* rbg 0->1 */
			stat->nml++;
			LOG_U2("[CAN]detect message lost (normal)", stat->nml);
		}
		else
		{
			/* rbg 1->0 */
			LOG_U2("[CAN]clear  message lost (normal)", stat->nml);
		}
	}

	/* FIFO[{bNX bZ[WXgotO */
	if ((str_xor & 0x2000) != 0)
	{
		if ((str & 0x2000) != 0)
		{
			/* rbg 0->1 */
			stat->fml++;
			LOG_U2("[CAN]detect message lost (FIFO)", stat->fml);
		}
		else
		{
			/* rbg 1->0 */
			LOG_U2("[CAN]clear  message lost (FIFO)", stat->fml);
		}
	}

	/* ̒lL */
	stat->str = str;
}

/*
 * CAN
 * ^XN
 */
void can_task(void)
{
	can_task_send();
	can_task_recv();
	can_task_error(&can_channel.error);
	can_task_status(&can_channel.status);
}

/*
 * CAN
 * oXItXe[^X擾
 */
BOOL can_is_busoff(void)
{
	/* oXItԂ */
	if (bost_c0str == 1)
	{
		/* oXIt */
		return TRUE;
	}
	else
	{
		/* oXItԂłȂ */
		return FALSE;
	}
}

/*
 * CAN
 * MFIFO荞݃nh
 */
#pragma INTERRUPT /B can_rxfifo_isr(vect=55)
void can_rxfifo_isr(void)
{
	/* M^XN̏s */
	can_task_recv();
}
