//! @file		pf_gpio.h
//! @brief		プラットフォーム(GPIO)ヘッダファイル

// The MIT License (MIT)
// Copyright (c) 2023 @xm6_original
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#ifndef PF_GPIO_H
#define PF_GPIO_H

#include "pf_types.h"

//! @brief		GPIOピンのID
typedef enum PF_GPIO_ID_Tag
{
	PF_GPIO_ID_UART_TXD,					//!< UART TxD
	PF_GPIO_ID_UART_RXD,					//!< UART RxD
	PF_GPIO_ID_ROW1,						//!< Display ROW1
	PF_GPIO_ID_ROW2,						//!< Display ROW2
	PF_GPIO_ID_ROW3,						//!< Display ROW3
	PF_GPIO_ID_ROW4,						//!< Display ROW4
	PF_GPIO_ID_ROW5,						//!< Display ROW5
	PF_GPIO_ID_COL1,						//!< Display COL1
	PF_GPIO_ID_COL2,						//!< Display COL2
	PF_GPIO_ID_COL3,						//!< Display COL3
	PF_GPIO_ID_COL4,						//!< Display COL4
	PF_GPIO_ID_COL5,						//!< Display COL5
	PF_GPIO_ID_BUTTON_A,					//!< BUTTON A
	PF_GPIO_ID_BUTTON_B,					//!< BUTTON B
	PF_GPIO_ID_INTERNAL_SCL,				//!< 内部I2Cバス SCL
	PF_GPIO_ID_INTERNAL_SDA,				//!< 内部I2Cバス SDA
	PF_GPIO_ID_EXTERNAL_SCL,				//!< 外部I2Cバス SCL
	PF_GPIO_ID_EXTERNAL_SDA,				//!< 外部I2Cバス SDA
	PF_GPIO_ID_MAQUEEN_LED_L,				//!< MAQUEEN LED L
	PF_GPIO_ID_MAQUEEN_LED_R,				//!< MAQUEEN LED R
	PF_GPIO_ID_MAQUEEN_PATROL_L,			//!< MAQUEEN PATROL L
	PF_GPIO_ID_MAQUEEN_PATROL_R,			//!< MAUQEEN PATROL R
	PF_GPIO_ID_MAX,							//!< (IDの個数を表す)
} PF_GPIO_ID;

//! @brief		GPIO初期化
//! @remarks	プラットフォーム初期化処理から呼び出すこと
void pf_gpio_init(void);

//! @brief		GPIOポート番号を取得
//! @param		[in] id			GPIOピンのID
//! @return		GPIOポート番号(0 or 1)
u4 pf_gpio_get_port(PF_GPIO_ID id);

//! @brief		GPIOピン番号を取得
//! @param		[in] id			GPIOピンのID
//! @return		GPIOピン番号(0～31)
u4 pf_gpio_get_pin(PF_GPIO_ID id);

//! @brief		GPIOから入力
//! @param		[in] id			GPIOピンのID
//! @result		入力レベル(TRUE='H'レベル/FALSE='L'レベル)
BOOL pf_gpio_input(PF_GPIO_ID id);

//! @brief		GPIOへ出力
//! @param		[in] id			GPIOピンのID
//! @param		[in] level		出力レベル
void pf_gpio_output(PF_GPIO_ID id, BOOL level);

#endif  // PF_GPIO_H
