//! @file		pf_timer.h
//! @brief		プラットフォーム(Timer)ヘッダファイル

// The MIT License (MIT)
// Copyright (c) 2023 @xm6_original
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#ifndef PF_TIMER_H
#define PF_TIMER_H

#include "pf_types.h"

//! @brief		us→コンペア値変換
#define MICROSEC_TO_TIMERCC(us)		(us)

//! @brief		Timerコールバック関数
//! @param		[in] eventbit	イベントビットパターン
typedef void (*PF_TIMER_CALLBACK)(u4 eventbit);

//! @brief		TimerのID
typedef enum PF_TIMER_ID_Tag
{
	PF_TIMER_ID_DISPLAY = 0,				//!< ディスプレイタイマ
	PF_TIMER_ID_MAX							//!< (IDの個数を表す)
} PF_TIMER_ID;

//! @brief		Timer初期化
//! @remarks	プラットフォーム初期化処理から呼び出すこと
void pf_timer_init(void);

//! @brief		Timer開始
//! @param		[in] id			TimerのID
void pf_timer_start(PF_TIMER_ID id);

//! @brief		Timer停止
//! @param		[in] id			TimerのID
void pf_timer_stop(PF_TIMER_ID id);

//! @brief		Timerコンペアカウンタ設定
//! @param		[in] id			TimerのID
//! @param		[in] cc			コンペアカウンタ配列
void pf_timer_cc(PF_TIMER_ID id, u4 *cc);

//! @brief		Timerコールバック関数設定
//! @param		[in] id			TimerのID
//! @param		[in] func		Timerコールバック関数へのポインタ
//! @attention	Timerコールバック関数は割り込みコンテキストで呼び出される
void pf_timer_callback(PF_TIMER_ID id, PF_TIMER_CALLBACK func);

//! @brief		Timer0割り込みハンドラ
//! @attention	データ競合(割り込み干渉)に注意する
void TIMER0_IRQHandler(void);

//! @brief		Timer1割り込みハンドラ
//! @attention	データ競合(割り込み干渉)に注意する
void TIMER1_IRQHandler(void);

//! @brief		Timer2割り込みハンドラ
//! @attention	データ競合(割り込み干渉)に注意する
void TIMER2_IRQHandler(void);

//! @brief		Timer3割り込みハンドラ
//! @attention	データ競合(割り込み干渉)に注意する
void TIMER3_IRQHandler(void);

//! @brief		Timer4割り込みハンドラ
//! @attention	データ競合(割り込み干渉)に注意する
void TIMER4_IRQHandler(void);

#endif  // PF_TIMER_H
