//! @file		main.c
//! @brief		メインプログラム

// The MIT License (MIT)
// Copyright (c) 2023 @xm6_original
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include "pf_types.h"
#include "pf_clock.h"
#include "pf_gpio.h"
#include "pf_systick.h"
#include "pf_timer.h"
#include "pf_uart.h"
#include "pf_display.h"
#include "pf_i2c.h"
#include "pf_power.h"
#include "pf_button.h"
#include "pf_led.h"
#include "pf_patrol.h"
#include "pf_motor.h"

//! @brief		プラットフォーム初期化
//! @attention	初期化順序に注意する
static void pf_init(void)
{
	// Step1: クロックをHFXO(外部発振子)に切り替える
	pf_clock_init();

	// Step2: 低位モジュールを初期化する
	pf_gpio_init();
	pf_systick_init();
	pf_timer_init();

	// Step3: 中位モジュール(低位モジュールに依存するモジュール)を初期化する
	pf_uart_init();
	pf_display_init();
	pf_i2c_init();

	// Step4: 高位モジュール(低位モジュールまたは中位モジュールに依存するモジュール)を初期化する
	pf_power_init();
	pf_button_init();
	pf_led_init();
	pf_patrol_init();
	pf_motor_init();
}

//! @brief		定期タスク(入力系)
static void pf_input_task(void)
{
	pf_power_task();
	pf_uart_task();
	pf_button_task();
	pf_patrol_task();
}

//! @brief		定期タスク(出力系)
static void pf_output_task(void)
{
	pf_led_task();
	pf_motor_task();
}

static BOOL button_prev[2];
static PF_DISPLAY_ID display_id;
static char display_ch = 0x20;
static PF_MOTOR_DIR motor_dir = 0;

//! @brief		アプリケーションタスク
static void app_task(void)
{
	BOOL button_now[2];

	button_now[0] = pf_button_get(PF_BUTTON_ID_BUTTON_A);
	button_now[1] = pf_button_get(PF_BUTTON_ID_BUTTON_B);

	if (button_now[0] != button_prev[0])
	{
		if (button_now[0])
		{
			pf_uart_log("Button A: OFF -> ON");

			// アイコン系
			pf_display_id(display_id);
			display_id++;
			if (display_id >= PF_DISPLAY_ID_MAX)
			{
				display_id = 0;
			}
		}
		else
		{
			pf_uart_log("Button A: ON  -> OFF\n");
		}
	}

	if (button_now[1] != button_prev[1])
	{
		if (button_now[1])
		{
			pf_uart_log("Button B: OFF -> ON");

			// キャラクタ系
			pf_display_ch(display_ch);
			if (display_ch == 0x7F)
			{
				display_ch = 0x20;
			}
			else
			{
				display_ch++;
			}

			// モータ駆動
			pf_motor_drive(motor_dir);
			motor_dir++;
			if (motor_dir >= PF_MOTOR_DIR_MAX)
			{
				motor_dir = 0;
			}
		}
		else
		{
			pf_uart_log("Button B: ON  -> OFF\n");
		}
	}

	// ラインセンサとLEDの連携制御(黒色で点灯、白色で消灯)
	if (PF_PATROL_COLOR_BLACK == pf_patrol_get(PF_PATROL_ID_L))
	{
		pf_led_ctrl(PF_LED_ID_L, TRUE);
	}
	else
	{
		pf_led_ctrl(PF_LED_ID_L, FALSE);
	}

	if (PF_PATROL_COLOR_BLACK == pf_patrol_get(PF_PATROL_ID_R))
	{
		pf_led_ctrl(PF_LED_ID_R, TRUE);
	}
	else
	{
		pf_led_ctrl(PF_LED_ID_R, FALSE);
	}

	// ボタン情報を更新
	button_prev[0] = button_now[0];
	button_prev[1] = button_now[1];
}

//! @brief		メインプログラム
//! @remarks	マイコンリセット後、startup.cの_start()に制御が移り、その後main()が呼び出される
int main(void)
{
	// プラットフォーム初期化
	pf_init();

	// 無限ループ
	while (1)
	{
		// 制御周期まで待つ
		pf_systick_sync();

		// 定期タスク(入力系)
		pf_input_task();

		// 定期タスク(アプリケーション)
		app_task();

		// 定期タスク(出力系)
		pf_output_task();
	}
}
