//! @file		pf_motor.c
//! @brief		プラットフォーム(モータ)実装ファイル

// The MIT License (MIT)
// Copyright (c) 2023 @xm6_original
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include "pf_types.h"
#include "pf_interrupt.h"
#include "pf_i2c.h"
#include "pf_motor.h"

//! @brief		モータ速度(デフォルト)
#define PF_MOTOR_DEFAULT_SPEED		((u1)100U)

//! @brief		チャネル(左)
#define PF_MOTOR_CHANNEL_LEFT		((u1)0x00U)

//! @brief		チャネル(左)
#define PF_MOTOR_CHANNEL_RIGHT		((u1)0x02U)

//! @brief		チャネル数
#define PF_MOTOR_CHANNEL_MAX		((u4)2U)

//! @brief		回転方向(前回転)
#define PF_MOTOR_MOVE_FORWARD		((u1)0x00U)

//! @brief		回転方向(後回転)
#define PF_MOTOR_MOVE_BACKWARD		((u1)0x01U)

//! @brief		動作速度(駆動)
#define PF_MOTOR_SPEED_DRIVE		((u1)0x01U)

//! @brief		動作速度(停止)
#define PF_MOTOR_SPEED_STOP			((u1)0x00U)

//! @brief		I2Cバッファ要素定義
typedef enum PF_MOTOR_I2C_Tag
{
	PF_MOTOR_I2C_CHANNEL = 0,				//!< チャネル
	PF_MOTOR_I2C_MOVE,						//!< 回転方向
	PF_MOTOR_I2C_SPEED,						//!< 速度
	PF_MOTOR_I2C_MAX,						//!< (バッファのサイズを表す)
} PF_MOTOR_I2C;

//! @brief		モータ駆動情報構造体
typedef struct PF_MOTOR_DIR_INFO_Tag
{
	u1				channel;				//!< チャネル
	u1				direction;				//!< 駆動方向
	u1				speed;					//!< 動作速度
} PF_MOTOR_DIR_INFO;

//! @brief		モータ駆動方向→モータ駆動情報テーブル
static const PF_MOTOR_DIR_INFO pf_motor_dir_to_info[PF_MOTOR_DIR_MAX][PF_MOTOR_CHANNEL_MAX] =
{
	// PF_MOTOR_DIR_FORWARD
	{
		// LEFT
		{
			PF_MOTOR_CHANNEL_LEFT,
			PF_MOTOR_MOVE_FORWARD,
			PF_MOTOR_SPEED_DRIVE,
		},

		// RIGHT
		{
			PF_MOTOR_CHANNEL_RIGHT,
			PF_MOTOR_MOVE_FORWARD,
			PF_MOTOR_SPEED_DRIVE,
		},
	},

	// PF_MOTOR_DIR_LEFT
	{
		// LEFT
		{
			PF_MOTOR_CHANNEL_LEFT,
			PF_MOTOR_MOVE_BACKWARD,
			PF_MOTOR_SPEED_DRIVE,
		},

		// RIGHT
		{
			PF_MOTOR_CHANNEL_RIGHT,
			PF_MOTOR_MOVE_FORWARD,
			PF_MOTOR_SPEED_DRIVE,
		},
	},

	// PF_MOTOR_DIR_RIGHT
	{
		// LEFT
		{
			PF_MOTOR_CHANNEL_LEFT,
			PF_MOTOR_MOVE_FORWARD,
			PF_MOTOR_SPEED_DRIVE,
		},

		// RIGHT
		{
			PF_MOTOR_CHANNEL_RIGHT,
			PF_MOTOR_MOVE_BACKWARD,
			PF_MOTOR_SPEED_DRIVE,
		},
	},

	// PF_MOTOR_DIR_BACKWORD
	{
		// LEFT
		{
			PF_MOTOR_CHANNEL_LEFT,
			PF_MOTOR_MOVE_BACKWARD,
			PF_MOTOR_SPEED_DRIVE,
		},

		// RIGHT
		{
			PF_MOTOR_CHANNEL_RIGHT,
			PF_MOTOR_MOVE_BACKWARD,
			PF_MOTOR_SPEED_DRIVE,
		},
	},

	// PF_MOTOR_DIR_STOP
	{
		// LEFT
		{
			PF_MOTOR_CHANNEL_LEFT,
			PF_MOTOR_MOVE_FORWARD,
			PF_MOTOR_SPEED_STOP,
		},

		// RIGHT
		{
			PF_MOTOR_CHANNEL_RIGHT,
			PF_MOTOR_MOVE_FORWARD,
			PF_MOTOR_SPEED_STOP,
		},
	},
};

//! @brief		I2C通信バッファ
static u1 pf_motor_buf[PF_MOTOR_CHANNEL_MAX][PF_MOTOR_I2C_MAX];

//! @brief		現在のモータ駆動方向
static PF_MOTOR_DIR pf_motor_current_dir;

//! @brief		次回のモータ駆動方向
static PF_MOTOR_DIR pf_motor_next_dir;

//! @brief		現在のモータ速度
static u1 pf_motor_current_speed;

//! @brief		現在の送信中チャネル
static u4 pf_motor_current_channel;

//! @brief		I2Cからのコールバック関数
//! @param		[in] status		通信ステータス(TRUE=成功/FALSE=失敗)
static void pf_motor_callback(BOOL status)
{
	// 成功の場合
	if (TRUE == status)
	{
		// 送信中チャネルをインクリメント
		pf_motor_current_channel++;

		// 送信中チャネルが最大値に達しているか
		if (pf_motor_current_channel < PF_MOTOR_CHANNEL_MAX)
		{
			// 次の送信をスタート
			(void)pf_i2c_send(PF_I2C_ID_MAQUEEN_MOTOR, &pf_motor_buf[pf_motor_current_channel][0],
							PF_MOTOR_I2C_MAX);
		}
		else
		{
			// すべてのチャネルを送信完了したので、更新
			pf_motor_current_dir = pf_motor_next_dir;
		}
	}
}

//! @brief		モータ初期化
//! @remarks	プラットフォーム初期化処理から呼び出すこと
void pf_motor_init(void)
{
	// モータ駆動方向: 停止
	pf_motor_current_dir = PF_MOTOR_DIR_STOP;
	pf_motor_next_dir = PF_MOTOR_DIR_STOP;

	// モータ速度: デフォルト
	pf_motor_current_speed = PF_MOTOR_DEFAULT_SPEED;

	// I2Cコールバック関数を設定
	pf_i2c_callback(PF_I2C_ID_MAQUEEN_MOTOR, pf_motor_callback);
}

//! @brief		モータ駆動情報送信
static void pf_motor_send(void)
{
	u4 loop;

	// オート変数初期化
	loop = 0;

	// チャネル数ループ
	for (loop = 0; loop < PF_MOTOR_CHANNEL_MAX; loop++)
	{
		// チャネル
		pf_motor_buf[loop][PF_MOTOR_I2C_CHANNEL] =
						pf_motor_dir_to_info[pf_motor_next_dir][loop].channel;

		// 駆動方向
		pf_motor_buf[loop][PF_MOTOR_I2C_MOVE] =
						pf_motor_dir_to_info[pf_motor_next_dir][loop].direction;

		// 動作速度(駆動の場合は現在の速度を設定する)
		if (PF_MOTOR_SPEED_STOP == pf_motor_dir_to_info[pf_motor_next_dir][loop].speed)
		{
			// 停止
			pf_motor_buf[loop][PF_MOTOR_I2C_SPEED] = PF_MOTOR_SPEED_STOP;
		}
		else
		{
			// 現在の速度で駆動
			pf_motor_buf[loop][PF_MOTOR_I2C_SPEED] = pf_motor_current_speed;
		}
	}

	// 送信中チャネルを初期化
	pf_motor_current_channel = 0;

	// 最初のチャネルの送信をスタート
	(void)pf_i2c_send(PF_I2C_ID_MAQUEEN_MOTOR, &pf_motor_buf[0][0], PF_MOTOR_I2C_MAX);
}

//! @brief		モータ定期タスク
//! @remarks	プラットフォーム定期タスク(出力系)処理から呼び出すこと
void pf_motor_task(void)
{
	u4 enable;
	PF_MOTOR_DIR current;

	// オート変数初期化
	enable = 0;
	current = PF_MOTOR_DIR_STOP;

	// 現在の駆動方向を取得(グローバル割り込みを禁止して行う)
	enable = pf_interrupt_global_disable();
	current = pf_motor_current_dir;
	pf_interrupt_global_restore(enable);

	// 駆動方向を比較
	if (current != pf_motor_next_dir)
	{
		// 一致するまで送信
		pf_motor_send();
	}
}

//! @brief		モータ駆動
//! @details	モータ定期タスクで実際のモータ出力が行われる
//! @param		[in] dir		モータ駆動方向
void pf_motor_drive(PF_MOTOR_DIR dir)
{
	// パラメータチェック
	if (dir < PF_MOTOR_DIR_MAX)
	{
		pf_motor_next_dir = dir;
	}
}
