//! @file		pf_motor.h
//! @brief		プラットフォーム(モータ)ヘッダファイル

// The MIT License (MIT)
// Copyright (c) 2023 @xm6_original
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#ifndef PF_MOTOR_H
#define PF_MOTOR_H

#include "pf_types.h"

//! @brief		モータ駆動方向
typedef enum PF_MOTOR_DIR_Tag
{
	PF_MOTOR_DIR_FORWARD,					//!< 前進
	PF_MOTOR_DIR_LEFT,						//!< 左回転
	PF_MOTOR_DIR_RIGHT,						//!< 右回転
	PF_MOTOR_DIR_BACKWARD,					//!< 後進
	PF_MOTOR_DIR_STOP,						//!< 停止
	PF_MOTOR_DIR_MAX,						//!< (方向の個数を表す)
} PF_MOTOR_DIR;

//! @brief		モータ初期化
//! @remarks	プラットフォーム初期化処理から呼び出すこと
void pf_motor_init(void);

//! @brief		モータ定期タスク
//! @remarks	プラットフォーム定期タスク(出力系)処理から呼び出すこと
void pf_motor_task(void);

//! @brief		モータ駆動
//! @details	モータ定期タスクで実際のモータ出力が行われる
//! @details	pf_motor_freeとは排他的に動作する
//! @param		[in] dir		モータ駆動方向
void pf_motor_drive(PF_MOTOR_DIR dir);

//! @brief		モータ制御
//! @details	モータ定期タスクで実際のモータ出力が行われる
//! @details	pf_motor_driveとは排他的に動作する
//! @param		[in] left		左モータ駆動速度(0=停止)
//! @param		[in] right		右モータ駆動速度(0=停止)
void pf_motor_ctrl(u1 left, u1 right);

//! @brief		モータ速度設定
//! @param		[in] speed		モータ駆動速度(デフォルト=100)
void pf_motor_speed(u1 speed);

#endif  // PF_MOTOR_H
