//! @file		pf_button.c
//! @brief		プラットフォーム(ボタン)実装ファイル

// The MIT License (MIT)
// Copyright (c) 2023 @xm6_original
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include "pf_types.h"
#include "pf_gpio.h"
#include "pf_button.h"

//! @brief		チャタリング防止のためのフィルタ段数(最低2段以上にすること)
#define PF_BUTTON_FILTER_MAX		((u4)5U)

//! @brief		ボタン→GPIO IDテーブル
static const PF_GPIO_ID pf_button_to_gpio[PF_BUTTON_ID_MAX] =
{
	PF_GPIO_ID_BUTTON_A,					//!< BUTTON A(Display面から見て左側)
	PF_GPIO_ID_BUTTON_B,					//!< BUTTON B(Display面から見て右側)
};

//! @brief		ボタン情報構造体
typedef struct PF_BUTTON_INFO_Tag
{
	BOOL			prev[PF_BUTTON_FILTER_MAX];	//!< 過去データ
	BOOL			now;						//!< 現在データ
} PF_BUTTON_INFO;

//! @brief		ボタン情報テーブル
static PF_BUTTON_INFO pf_button_info[PF_BUTTON_ID_MAX];

//! @brief		ボタン初期化(単一のID)
//! @param		[in] id			ボタンのID
static void pf_button_init_id(PF_BUTTON_ID id)
{
	u4 loop;

	// オート変数初期化
	loop = 0;

	// 過去データをすべてFALSE(押されていない)に初期化
	for (loop = 0; loop < PF_BUTTON_FILTER_MAX; loop++)
	{
		pf_button_info[id].prev[loop] = FALSE;
	}

	// 現在データをFALSE(押されていない)に初期化
	pf_button_info[id].now = FALSE;
}

//! @brief		ボタン初期化
//! @remarks	プラットフォーム初期化処理から呼び出すこと
void pf_button_init(void)
{
	PF_BUTTON_ID id;

	// オート変数初期化
	id = 0;

	// すべてのIDをループ
	for (id = 0; id < PF_BUTTON_ID_MAX; id++)
	{
		// 1つのIDを初期化
		pf_button_init_id(id);
	}
}

//! @brief		ボタン定期タスク(単一のID)
//! @param		[in] id			ボタンのID
static void pf_button_task_id(PF_BUTTON_ID id)
{
	PF_GPIO_ID gpio;
	u4 loop;
	BOOL result;
	BOOL now;

	// オート変数初期化
	gpio = pf_button_to_gpio[id];
	loop = 0;
	result = FALSE;
	now = FALSE;

	// 現在のボタン押下情報を取得(GPIOからの入力が'L'で押されている)
	if (FALSE == pf_gpio_input(gpio))
	{
		// 押されている
		result = TRUE;
	}
	else
	{
		// 押されていない
		result = FALSE;
	}

	// prev配列を1つ前へ移動
	for (loop = 0; loop < (PF_BUTTON_FILTER_MAX - 1); loop++)
	{
		pf_button_info[id].prev[loop] = pf_button_info[id].prev[loop + 1];
	}

	// 最後に要素にresultを格納
	pf_button_info[id].prev[PF_BUTTON_FILTER_MAX - 1] = result;

	// いったん、現在の状態を'押されている'に設定する
	now = TRUE;

	// すべてのprevを検索し、一つでも'押されていない'であればFALSE
	for (loop = 0; loop < PF_BUTTON_FILTER_MAX; loop++)
	{
		if (FALSE == pf_button_info[id].prev[loop])
		{
			now = FALSE;
			break;
		}
	}

	// 現在の押下状態を更新
	pf_button_info[id].now = now;
}

//! @brief		ボタン定期タスク
//! @remarks	プラットフォーム定期タスク(入力系)処理から呼び出すこと
void pf_button_task(void)
{
	PF_BUTTON_ID id;

	// オート変数初期化
	id = 0;

	// すべてのIDをループ
	for (id = 0; id < PF_BUTTON_ID_MAX; id++)
	{
		// 1つのIDを処理
		pf_button_task_id(id);
	}
}

//! @brief		ボタンの押下状態を取得
//! @details	モジュール内部でチャタリング除去処理が行われる
//! @param		[in] id			ボタンのID
//! @return		押下状態(TRUE=押されている/FALSE=押されていない)
BOOL pf_button_get(PF_BUTTON_ID id)
{
	BOOL result;

	// オート変数初期化
	result = FALSE;

	// パラメータチェック
	if (id < PF_BUTTON_ID_MAX)
	{
		result = pf_button_info[id].now;
	}

	return result;
}
