//! @file		pf_motor.c
//! @brief		プラットフォーム(モータ)実装ファイル

// The MIT License (MIT)
// Copyright (c) 2023 @xm6_original
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include "pf_types.h"
#include "pf_interrupt.h"
#include "pf_i2c.h"
#include "pf_motor.h"

//! @brief		I2C常時送信動作(TRUE=常時送信/FALSE=キャッシュ動作)
#define PF_MOTOR_I2C_ANY			(TRUE)

//! @brief		モータ速度(デフォルト)
#define PF_MOTOR_DEFAULT_SPEED		((u1)100U)

//! @brief		チャネル(左)
#define PF_MOTOR_CHANNEL_LEFT		((u1)0x00U)

//! @brief		チャネル(左)
#define PF_MOTOR_CHANNEL_RIGHT		((u1)0x02U)

//! @brief		チャネル数
#define PF_MOTOR_CHANNEL_MAX		((u4)2U)

//! @brief		回転方向(前回転)
#define PF_MOTOR_MOVE_FORWARD		((u1)0x00U)

//! @brief		回転方向(後回転)
#define PF_MOTOR_MOVE_BACKWARD		((u1)0x01U)

//! @brief		回転方向(不定値)
#define PF_MOTOR_MOVE_UNFORMATTED	((u1)0xFFU)

//! @brief		動作速度(駆動)
#define PF_MOTOR_SPEED_DRIVE		((u1)0x01U)

//! @brief		動作速度(停止)
#define PF_MOTOR_SPEED_STOP			((u1)0x00U)

//! @brief		I2Cバッファ要素定義
typedef enum PF_MOTOR_I2C_Tag
{
	PF_MOTOR_I2C_CHANNEL = 0,				//!< チャネル
	PF_MOTOR_I2C_MOVE,						//!< 回転方向
	PF_MOTOR_I2C_SPEED,						//!< 速度
	PF_MOTOR_I2C_MAX,						//!< (バッファのサイズを表す)
} PF_MOTOR_I2C;

//! @brief		I2C通信パケット構造体
typedef struct PF_MOTOR_I2C_PACKET_Tag
{
	u1				channel;				//!< チャネル
	u1				direction;				//!< 駆動方向
	u1				speed;					//!< 動作速度
} PF_MOTOR_I2C_PACKET;

//! @brief		駆動方向→I2C通信パケットテーブル
static const PF_MOTOR_I2C_PACKET pf_motor_dir_to_packet[PF_MOTOR_DIR_MAX][PF_MOTOR_CHANNEL_MAX] =
{
	// PF_MOTOR_DIR_FORWARD
	{
		// LEFT
		{
			PF_MOTOR_CHANNEL_LEFT,
			PF_MOTOR_MOVE_FORWARD,
			PF_MOTOR_SPEED_DRIVE,
		},

		// RIGHT
		{
			PF_MOTOR_CHANNEL_RIGHT,
			PF_MOTOR_MOVE_FORWARD,
			PF_MOTOR_SPEED_DRIVE,
		},
	},

	// PF_MOTOR_DIR_LEFT
	{
		// LEFT
		{
			PF_MOTOR_CHANNEL_LEFT,
			PF_MOTOR_MOVE_BACKWARD,
			PF_MOTOR_SPEED_DRIVE,
		},

		// RIGHT
		{
			PF_MOTOR_CHANNEL_RIGHT,
			PF_MOTOR_MOVE_FORWARD,
			PF_MOTOR_SPEED_DRIVE,
		},
	},

	// PF_MOTOR_DIR_RIGHT
	{
		// LEFT
		{
			PF_MOTOR_CHANNEL_LEFT,
			PF_MOTOR_MOVE_FORWARD,
			PF_MOTOR_SPEED_DRIVE,
		},

		// RIGHT
		{
			PF_MOTOR_CHANNEL_RIGHT,
			PF_MOTOR_MOVE_BACKWARD,
			PF_MOTOR_SPEED_DRIVE,
		},
	},

	// PF_MOTOR_DIR_BACKWORD
	{
		// LEFT
		{
			PF_MOTOR_CHANNEL_LEFT,
			PF_MOTOR_MOVE_BACKWARD,
			PF_MOTOR_SPEED_DRIVE,
		},

		// RIGHT
		{
			PF_MOTOR_CHANNEL_RIGHT,
			PF_MOTOR_MOVE_BACKWARD,
			PF_MOTOR_SPEED_DRIVE,
		},
	},

	// PF_MOTOR_DIR_STOP
	{
		// LEFT
		{
			PF_MOTOR_CHANNEL_LEFT,
			PF_MOTOR_MOVE_FORWARD,
			PF_MOTOR_SPEED_STOP,
		},

		// RIGHT
		{
			PF_MOTOR_CHANNEL_RIGHT,
			PF_MOTOR_MOVE_FORWARD,
			PF_MOTOR_SPEED_STOP,
		},
	},
};

//! @brief		モータ動作情報構造体
typedef struct PF_MOTOR_INFO_Tag
{
	PF_MOTOR_DIR	dir;					//!< 現在の駆動方向
	u4				channel;				//!< 送信中チャネル
	BOOL			free;					//!< フリー動作フラグ
	u1				speed;					//!< 駆動速度(フリー動作以外)
	u1				left;					//!< 駆動速度(フリー動作L)
	u1				right;					//!< 駆動速度(フリー動作R)
	u1				buf[PF_MOTOR_CHANNEL_MAX][PF_MOTOR_I2C_MAX];	//!< I2C通信バッファ(送信中)
	u1				i2c[PF_MOTOR_CHANNEL_MAX][PF_MOTOR_I2C_MAX];	//!< I2C通信バッファ(送信完了)
} PF_MOTOR_INFO;

// モータ動作情報
static PF_MOTOR_INFO pf_motor_info;

//! @brief		I2Cからのコールバック関数
//! @param		[in] status		通信ステータス(TRUE=成功/FALSE=失敗)
static void pf_motor_callback(BOOL status)
{
	u4 loop;

	// オート変数初期化
	loop = 0;

	// 成功の場合
	if (TRUE == status)
	{
		// 送信中チャネルをインクリメント
		pf_motor_info.channel++;

		// 送信中チャネルが最大値に達しているか
		if (pf_motor_info.channel < PF_MOTOR_CHANNEL_MAX)
		{
			// 次のI2C送信をスタート
			(void)pf_i2c_send(PF_I2C_ID_MAQUEEN_MOTOR, &pf_motor_info.buf[pf_motor_info.channel][0],
							PF_MOTOR_I2C_MAX);
		}
		else
		{
			// すべてのチャネルが正常送信できた
			if (FALSE == pf_motor_info.free)
			{
				// チャネル数ループ
				for (loop = 0; loop < PF_MOTOR_CHANNEL_MAX; loop++)
				{
					// pf_motor_info.bufをpf_motor_info.i2cへコピー(回転方向と速度のみ)
					pf_motor_info.i2c[loop][PF_MOTOR_I2C_MOVE] =
									pf_motor_info.buf[loop][PF_MOTOR_I2C_MOVE];
					pf_motor_info.i2c[loop][PF_MOTOR_I2C_SPEED] =
									pf_motor_info.buf[loop][PF_MOTOR_I2C_SPEED];
				}
			}
		}
	}
}

//! @brief		モータ初期化
//! @remarks	プラットフォーム初期化処理から呼び出すこと
void pf_motor_init(void)
{
	u4 loop;

	// オート変数初期化
	loop = 0;

	// モータ駆動方向
	pf_motor_info.dir = PF_MOTOR_DIR_STOP;

	// 送信中チャネル
	pf_motor_info.channel = 0;

	// フリー動作フラグ
	pf_motor_info.free = FALSE;

	// 駆動速度(フリー動作以外)
	pf_motor_info.speed = PF_MOTOR_DEFAULT_SPEED;

	// 駆動速度(フリー動作)
	pf_motor_info.left = PF_MOTOR_DEFAULT_SPEED;
	pf_motor_info.right = PF_MOTOR_DEFAULT_SPEED;

	// I2C通信バッファ(送信完了)
	for (loop = 0; loop < PF_MOTOR_CHANNEL_MAX; loop++)
	{
		// 回転方向を不定値とすることで、初回送信が確実に行われるようにする
		pf_motor_info.i2c[loop][PF_MOTOR_I2C_MOVE] = PF_MOTOR_MOVE_UNFORMATTED;
	}

	// I2Cコールバック関数を設定
	pf_i2c_callback(PF_I2C_ID_MAQUEEN_MOTOR, pf_motor_callback);
}

//! @brief		I2C送信
static void pf_motor_send(void)
{
#if PF_MOTOR_I2C_ANY == TRUE
	// 送信中チャネルをクリア
	pf_motor_info.channel = 0;

	// 常に送信開始
	(void)pf_i2c_send(PF_I2C_ID_MAQUEEN_MOTOR, &pf_motor_info.buf[pf_motor_info.channel][0],
						PF_MOTOR_I2C_MAX);
#else
	u4 enable;
	u4 loop;
	BOOL request;

	// オート変数初期化
	enable = 0;
	loop = 0;
	request = FALSE;

	// I2C割り込み(外部)を禁止
	enable = pf_interrupt_local_disable(PF_INTERRUPT_PRI_I2C_EXT);

	// 左右のモータを調べ、双方とも既に目的の動作になっていればI2C送信をスキップする
	for (loop = 0; loop < PF_MOTOR_CHANNEL_MAX; loop++)
	{
		// 回転方向が異なっていたら送信を要求
		if (pf_motor_info.buf[loop][PF_MOTOR_I2C_MOVE]
						!= pf_motor_info.i2c[loop][PF_MOTOR_I2C_MOVE])
		{
			request = TRUE;
		}

		// 速度が異なっていたら送信を要求
		if (pf_motor_info.buf[loop][PF_MOTOR_I2C_SPEED]
						!= pf_motor_info.i2c[loop][PF_MOTOR_I2C_SPEED])
		{
			request = TRUE;
		}
	}

	// 送信中チャネルをクリア
	pf_motor_info.channel = 0;

	// I2C割り込み(外部)を復元
	pf_interrupt_local_restore(PF_INTERRUPT_PRI_I2C_EXT, enable);

	// 要求があれば、送信開始
	if (TRUE == request)
	{
		(void)pf_i2c_send(PF_I2C_ID_MAQUEEN_MOTOR, &pf_motor_info.buf[pf_motor_info.channel][0],
						PF_MOTOR_I2C_MAX);
	}
#endif // PF_MOTOR_I2C_ANY
}

//! @brief		I2Cバッファ構成(駆動方向)
static void pf_motor_build_dir(void)
{
	u4 loop;

	// オート変数初期化
	loop = 0;

	// チャネル数ループ
	for (loop = 0; loop < PF_MOTOR_CHANNEL_MAX; loop++)
	{
		// チャネル
		pf_motor_info.buf[loop][PF_MOTOR_I2C_CHANNEL] =
						pf_motor_dir_to_packet[pf_motor_info.dir][loop].channel;

		// 回転方向
		pf_motor_info.buf[loop][PF_MOTOR_I2C_MOVE] =
						pf_motor_dir_to_packet[pf_motor_info.dir][loop].direction;

		// 速度(駆動の場合は現在の速度を設定する)
		if (PF_MOTOR_SPEED_STOP == pf_motor_dir_to_packet[pf_motor_info.dir][loop].speed)
		{
			// 停止
			pf_motor_info.buf[loop][PF_MOTOR_I2C_SPEED] = PF_MOTOR_SPEED_STOP;
		}
		else
		{
			// 現在の速度で駆動
			pf_motor_info.buf[loop][PF_MOTOR_I2C_SPEED] = pf_motor_info.speed;
		}
	}
}

//! @brief		I2Cバッファ構成(フリー)
static void pf_motor_build_free(void)
{
	u4 loop;

	// オート変数初期化
	loop = 0;

	// チャネル数ループ
	for (loop = 0; loop < PF_MOTOR_CHANNEL_MAX; loop++)
	{
		// チャネル
		pf_motor_info.buf[loop][PF_MOTOR_I2C_CHANNEL] =
						pf_motor_dir_to_packet[pf_motor_info.dir][loop].channel;

		// 回転方向(前回転固定)
		pf_motor_info.buf[loop][PF_MOTOR_I2C_MOVE] = PF_MOTOR_MOVE_FORWARD;

		// 速度
		if (0 == loop)
		{
			pf_motor_info.buf[loop][PF_MOTOR_I2C_SPEED] = pf_motor_info.left;
		}
		else
		{
			pf_motor_info.buf[loop][PF_MOTOR_I2C_SPEED] = pf_motor_info.right;
		}
	}
}

//! @brief		モータ定期タスク
//! @remarks	プラットフォーム定期タスク(出力系)処理から呼び出すこと
void pf_motor_task(void)
{
	// フリー種別で分ける
	if (FALSE == pf_motor_info.free)
	{
		// I2Cバッファ構成(駆動方向)
		pf_motor_build_dir();
	}
	else
	{
		// I2Cバッファ構成(フリー)
		pf_motor_build_free();
	}

	// I2C送信(必要に応じて行う)
	pf_motor_send();
}

//! @brief		モータ駆動
//! @details	モータ定期タスクで実際のモータ出力が行われる
//! @details	pf_motor_freeとは排他的に動作する
//! @param		[in] dir		モータ駆動方向
void pf_motor_drive(PF_MOTOR_DIR dir)
{
	// パラメータチェック
	if (dir < PF_MOTOR_DIR_MAX)
	{
		// 方向を設定
		pf_motor_info.dir = dir;

		// フリー動作なし
		pf_motor_info.free = FALSE;
	}
}

//! @brief		モータ制御
//! @details	モータ定期タスクで実際のモータ出力が行われる
//! @details	pf_motor_driveとは排他的に動作する
//! @param		[in] left		左モータ駆動速度(0=停止)
//! @param		[in] right		右モータ駆動速度(0=停止)
void pf_motor_ctrl(u1 left, u1 right)
{
	// 速度を設定
	pf_motor_info.left = left;
	pf_motor_info.right = right;

	// フリー動作あり
	pf_motor_info.free = TRUE;
}

//! @brief		モータ速度設定
//! @param		[in] speed		モータ駆動速度(デフォルト=100)
void pf_motor_speed(u1 speed)
{
	pf_motor_info.speed = speed;
}
