//! @file		pf_display.h
//! @brief		プラットフォーム(Display)ヘッダファイル

// The MIT License (MIT)
// Copyright (c) 2023 @xm6_original
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#ifndef PF_DISPLAY_H
#define PF_DISPLAY_H

#include "pf_types.h"

//! @brief		行の最大数
#define PF_DISPLAY_ROW_MAX			((u4)5U)

//! @brief		列の最大数
#define PF_DISPLAY_COL_MAX			((u4)5U)

//! @brief		明るさの最小値
#define PF_DISPLAY_BRIGHTNESS_MIN	((u4)4U)

//! @brief		明るさの最大値
#define PF_DISPLAY_BRIGHTNESS_MAX	((u4)96U)

//! @brief		イメージのID
typedef enum PF_DISPLAY_ID_Tag
{
	// デフォルト
	PF_DISPLAY_ID_HEART = 0,				//!< ハート

	// 感情
	PF_DISPLAY_ID_HAPPY,					//!< HAPPY(嬉しい)
	PF_DISPLAY_ID_SMILE,					//!< SMILE(笑い)
	PF_DISPLAY_ID_SAD,						//!< SAD(悲しい)
	PF_DISPLAY_ID_CONFUSED,					//!< CONFUSED(混乱)
	PF_DISPLAY_ID_ANGRY,					//!< ANGRY(怒り)

	// YES/NO
	PF_DISPLAY_ID_YES,						//!< YES(チェックマーク)
	PF_DISPLAY_ID_NO,						//!< NO(×マーク)

	// 矢印
	PF_DISPLAY_ID_ARROW_N,					//!< 矢印(上向き)
	PF_DISPLAY_ID_ARROW_NE,					//!< 矢印(右上向き)
	PF_DISPLAY_ID_ARROW_E,					//!< 矢印(右向き)
	PF_DISPLAY_ID_ARROW_SE,					//!< 矢印(右下向き)
	PF_DISPLAY_ID_ARROW_S,					//!< 矢印(下向き)
	PF_DISPLAY_ID_ARROW_SW,					//!< 矢印(左下向き)
	PF_DISPLAY_ID_ARROW_W,					//!< 矢印(左向き)
	PF_DISPLAY_ID_ARROW_NW,					//!< 矢印(左上向き)

	// その他の図形
	PF_DISPALY_ID_TRIANGLE,					//!< 三角形
	PF_DISPLAY_ID_CHESSBOARD,				//!< チェスボード(縞模様)
	PF_DISPLAY_ID_DIAMOND,					//!< ダイアモンド(菱形)
	PF_DISPLAY_ID_SQUARE,					//!< 四角形
	PF_DISPLAY_ID_MUSIC,					//!< 音符

	PF_DISPLAY_ID_MAX,						//!< (IDの個数を表す)
} PF_DISPLAY_ID;

//! @brief		Display初期化
//! @remarks	プラットフォーム初期化処理から呼び出すこと
//! @attention	GPIO初期化およびTimer初期化の後に呼び出すこと
void pf_display_init(void);

//! @brief		Display表示(文字)
//! @param		[in] ch			ASCII文字(0x20～0x7F)
void pf_display_ch(char ch);

//! @brief		Display表示(ID)
//! @param		[in] id			イメージのID
void pf_display_id(PF_DISPLAY_ID id);

//! @brief		Display表示(イメージ)
//! @param		[in] image		イメージへのポインタ(0=OFF/1=ON)
void pf_display_image(const u1 *image);

//! @brief		Display停止
void pf_display_off(void);

//! @brief		Display明るさ指定
//! @param		[in] brightness	明るさ(PF_DISPLAY_BRIGHTNESS_MIN～PF_DISPLAY_BRIGHTNESS_MAX)
//! @attention	範囲外の値は設定禁止
void pf_display_brightness(u4 brightness);

//! @brief		Display表示(電圧低下)
//! @attention	これ以降、Display系のAPIは使用できない
void pf_display_powerdown(void);

#endif  // PF_DISPLAY_H
