//! @file		pf_systick.c
//! @brief		プラットフォーム(SysTick)実装ファイル

// The MIT License (MIT)
// Copyright (c) 2023 @xm6_original
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include "pf_types.h"
#include "nrf52833.h"
#include "pf_interrupt.h"
#include "pf_systick.h"

//! @brief		フリーランカウンタ[ms]
//! @remarks	通常コンテキストで変化を監視するためvolatile修飾を付与する
static volatile u4 pf_systick_freerun_ms;

//! @brief		制御周期カウンタ[ms]
static u4 pf_systick_sync_ms;

//! @brief		SysTick初期化
//! @remarks	プラットフォーム初期化処理から呼び出すこと
void pf_systick_init(void)
{
	// SysTickタイマを停止
	SysTick->CTRL = 0;

	// 割り込み優先度を設定
	NVIC_SetPriority(SysTick_IRQn, PF_INTERRUPT_PRI_SYSTICK);

	// フリーランタイマをリセット
	pf_systick_freerun_ms = 0;

	// 1msのタイマを生成させるため、SystemCoreClockの1/1000を指定する
	// ・24bitの範囲に収まっていることが必要
	// ・狙い値の-1を設定する
	SysTick->LOAD = ((SystemCoreClock / 1000) - 1);

	// カウントリセット(この際に割り込みは発生しない)
	SysTick->VAL = 0;

	// 動作モードを指定する
	// ・SysClockタイマ動作
	// ・割り込みモードを指定
	// ・クロックソースとしてプロセッサクロックを指定
	SysTick->CTRL = SysTick_CTRL_ENABLE_Msk | SysTick_CTRL_TICKINT_Msk | SysTick_CTRL_CLKSOURCE_Msk;

	// 制御周期カウンタをクリア
	pf_systick_sync_ms = 0;
}

//! @brief		次の制御周期まで待つ
//! @attention	グローバル割り込み禁止状態で呼び出さないこと
void pf_systick_sync(void)
{
	BOOL quit;
	u4 freerun;

	// オート変数初期化
	quit = FALSE;
	freerun = 0;

	// 時間が経過するまでループ
	while (FALSE == quit)
	{
		// フリーランmsカウンタを取得
		freerun = pf_systick_freerun();

		// 差分が上回っていればSYNC完了
		if ((u4)(freerun - pf_systick_sync_ms) >= PF_SYSTICK_SYNC_MS)
		{
			// 制御周期カウンタを次の周期へ
			pf_systick_sync_ms += PF_SYSTICK_SYNC_MS;

			// 脱出フラグセット
			quit = TRUE;
		}
	}
}

//! @brief		SysTick時刻を取得
//! @param		[out] timebuf	SysTick時刻情報構造体へのポインタ
//! @attention	グローバル割り込み禁止状態で呼び出さないこと
void pf_systick_time(PF_SYSTICK_TIME *timebuf)
{
	u4 primask;
	u4 freerun[2];
	u4 val[2];
	u4 sec;
	u4 ms;
	u4 us;

	// オート変数初期化
	primask = 0;
	freerun[0] = 0;
	freerun[1] = 0;
	val[0] = 0;
	val[1] = 0;
	sec = 0;
	ms = 0;
	us = 0;

	// パラメータチェック
	if (NULL != timebuf)
	{
		// 初回読み出し(グローバル割り込み禁止＋読み出し+グローバル割り込み許可)
		primask = pf_interrupt_global_disable();
		val[0] = SysTick->VAL;
		freerun[0] = pf_systick_freerun_ms;
		pf_interrupt_global_restore(primask);

		// while文が必ず成立するよう条件設定
		freerun[1] = freerun[0] + 1;
		val[1] = val[0];

		// 2回読み出して、msが一致かつVALの順序が正しくなるまで続ける
		while ((freerun[0] != freerun[1]) || (val[0] > val[1]))
		{
			// 移動
			freerun[1] = freerun[0];
			val[1] = val[0];

			// グローバル割り込み禁止
			primask = pf_interrupt_global_disable();

			// 取得
			val[0] = SysTick->VAL;
			freerun[0] = pf_systick_freerun_ms;

			// グローバル割り込み復元
			pf_interrupt_global_restore(primask);
		}

		// sec作成
		sec = freerun[0] / 1000;

		// ms作成
		ms = freerun[0] % 1000;

		// us作成
		us = (SystemCoreClock / 1000) - val[0];
		us = (us * 1000) / (SystemCoreClock / 1000);

		// 格納
		timebuf->sec = sec;
		timebuf->ms = ms;
		timebuf->us = us;
	}
}

//! @brief		フリーランmsカウンタを取得
//! @return		フリーランmsカウンタ(32bit)
u4 pf_systick_freerun(void)
{
	u4 primask;
	u4 freerun;

	// オート変数初期化
	primask = 0;
	freerun = 0;

	// グローバル割り込み禁止
	primask = pf_interrupt_global_disable();

	// 取得
	freerun = pf_systick_freerun_ms;

	// グローバル割り込み復元
	pf_interrupt_global_restore(primask);

	return freerun;
}

//! @brief		SysTick割り込みハンドラ
void SysTick_Handler(void)
{
	// 49日と17時間2分47.295秒でラップアラウンドする
	pf_systick_freerun_ms++;
}
