//! @file		pf_uart.c
//! @brief		プラットフォーム(UART)実装ファイル

// The MIT License (MIT)
// Copyright (c) 2023 @xm6_original
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include <stdio.h>
#include <stdarg.h>

#include "pf_types.h"
#include "nrf52833.h"
#include "nrf52833_bitfields.h"
#include "pf_interrupt.h"
#include "pf_gpio.h"
#include "pf_systick.h"
#include "pf_uart.h"

//! @brief		バッファサイズ(2の倍数とすること)
#define PF_UART_BUF_BYTES			(0x1000)

//! @brief		受信除外文字(この文字は受信できない)
#define PF_UART_EMPTY_CODE			((u1)0xFFU)

//! @brief		UARTバッファ管理構造体
typedef struct PF_UART_BUF_Tag
{
	u4				read_pos;				//!< 読み取り位置
	u4				write_pos;				//!< 書き込み位置
	u4				valid_bytes;			//!< 有効バイト数
	u4				overrun_count;			//!< オーバーラン発生回数
	u4				uart_process_count;		//!< UART動作カウント
	u4				uart_read_bytes;		//!< 現在のUARTバッファから読み出したバイト数
	u1				buf[PF_UART_BUF_BYTES];	//!< リングバッファ
} PF_UART_BUF;

//! @brief		UARTバッファのID
typedef enum PF_UART_BUF_ID_Tag
{
	PF_UART_BUF_ID_TXD = 0,					//!< 送信バッファ
	PF_UART_BUF_ID_RXD,						//!< 受信バッファ
	PF_UART_BUF_ID_MAX,						//!< (IDの個数を表す)
} PF_UART_BUF_ID;

//! @brief		UARTバッファ本体
static PF_UART_BUF pf_uart_buf[PF_UART_BUF_ID_MAX];

//! @brief		DMAバッファ本体
static u1 pf_uart_dma[PF_UART_BUF_BYTES];

//! @brief		UARTエラー情報本体
static PF_UART_ERROR pf_uart_rxderr;

//! @brief		UARTバッファ初期化
//! @param		[in] id			UARTバッファのID
static void pf_uart_init_buf(PF_UART_BUF_ID id)
{
	pf_uart_buf[id].read_pos = 0;
	pf_uart_buf[id].write_pos = 0;
	pf_uart_buf[id].valid_bytes = 0;
	pf_uart_buf[id].overrun_count = 0;
	pf_uart_buf[id].uart_process_count = 0;
	pf_uart_buf[id].uart_read_bytes = 0;
}

//! @brief		DMAバッファ初期化
static void pf_uart_init_dma(void)
{
	u4 loop;

	// オート変数初期化
	loop = 0;

	// DMAバッファを除外文字コードで埋める
	for (loop = 0; loop < sizeof(pf_uart_dma); loop++)
	{
		pf_uart_dma[loop] = PF_UART_EMPTY_CODE;
	}
}

//! @brief		UARTエラー情報初期化
static void pf_uart_init_error(void)
{
	pf_uart_rxderr.overrun = 0;
	pf_uart_rxderr.parity = 0;
	pf_uart_rxderr.framing = 0;
	pf_uart_rxderr.brk = 0;
}

//! @brief		UART送信ポーリング
//! @attention	通常コンテキストと、割り込みコンテキストの双方から呼び出される
static void pf_uart_poll_txd(void)
{
	u4 txd_bytes;

	// オート変数初期化
	txd_bytes = 0;

	// TASKS_STARTTXを掛けられるか？
	if (0 == pf_uart_buf[PF_UART_BUF_ID_TXD].uart_process_count)
	{
		// 送信バッファに送信すべきデータがあるか？
		if (pf_uart_buf[PF_UART_BUF_ID_TXD].valid_bytes > 0)
		{
			// バッファをまたぐか？
			txd_bytes = sizeof(pf_uart_buf[PF_UART_BUF_ID_TXD].buf)
							- pf_uart_buf[PF_UART_BUF_ID_TXD].read_pos;
			if (pf_uart_buf[PF_UART_BUF_ID_TXD].valid_bytes < txd_bytes)
			{
				// バッファをまたがない(valid_bytesの方が小さい)
				txd_bytes = pf_uart_buf[PF_UART_BUF_ID_TXD].valid_bytes;
			}

			// 送信指示
			NRF_UARTE0->TXD.PTR =
							(u4)&(pf_uart_buf[PF_UART_BUF_ID_TXD].buf[pf_uart_buf[PF_UART_BUF_ID_TXD].read_pos]);
			NRF_UARTE0->TXD.MAXCNT = txd_bytes;
			NRF_UARTE0->TASKS_STARTTX =
			UARTE_TASKS_STARTTX_TASKS_STARTTX_Trigger << UARTE_TASKS_STARTTX_TASKS_STARTTX_Pos;

			// 1回のUART送信を予約した
			pf_uart_buf[PF_UART_BUF_ID_TXD].uart_process_count++;

			// read_posを進める
			pf_uart_buf[PF_UART_BUF_ID_TXD].read_pos += txd_bytes;
			pf_uart_buf[PF_UART_BUF_ID_TXD].read_pos &= (sizeof(pf_uart_buf[PF_UART_BUF_ID_TXD].buf)
							- 1);

			// valid_bytesを減らす
			pf_uart_buf[PF_UART_BUF_ID_TXD].valid_bytes -= txd_bytes;
		}
	}
}

//! @brief		UART受信ポーリング
//! @attention	通常コンテキストと、割り込みコンテキストの双方から呼び出される
static void pf_uart_poll_rxd(void)
{
	while (PF_UART_EMPTY_CODE != pf_uart_dma[pf_uart_buf[PF_UART_BUF_ID_RXD].uart_read_bytes])
	{
		// 現在の読み取り位置をバッファに挿入する
		if (pf_uart_buf[PF_UART_BUF_ID_RXD].valid_bytes
						<= sizeof(pf_uart_buf[PF_UART_BUF_ID_RXD].buf))
		{
			// データ転送
			pf_uart_buf[PF_UART_BUF_ID_RXD].buf[pf_uart_buf[PF_UART_BUF_ID_RXD].write_pos] =
							pf_uart_dma[pf_uart_buf[PF_UART_BUF_ID_RXD].uart_read_bytes];

			// write_posを進める
			pf_uart_buf[PF_UART_BUF_ID_RXD].write_pos++;
			pf_uart_buf[PF_UART_BUF_ID_RXD].write_pos &=
							(sizeof(pf_uart_buf[PF_UART_BUF_ID_RXD].buf) - 1);

			// valid_bytesをインクリメント
			pf_uart_buf[PF_UART_BUF_ID_RXD].valid_bytes++;
		}
		else
		{
			// リングバッファのオーバーラン
			pf_uart_buf[PF_UART_BUF_ID_RXD].overrun_count++;
		}

		// このDMAバッファを受信除外文字に復元する
		pf_uart_dma[pf_uart_buf[PF_UART_BUF_ID_RXD].uart_read_bytes] =
		PF_UART_EMPTY_CODE;

		// UARTバッファから読み出したバイト数を進める
		pf_uart_buf[PF_UART_BUF_ID_RXD].uart_read_bytes++;
		pf_uart_buf[PF_UART_BUF_ID_RXD].uart_read_bytes &=
						(sizeof(pf_uart_buf[PF_UART_BUF_ID_RXD].buf) - 1);
	}
}

//! @brief		UART初期化
//! @remarks	プラットフォーム初期化処理から呼び出すこと
//! @attention	GPIO初期化の後で呼び出すこと
void pf_uart_init(void)
{
	// UART無効化
	NRF_UARTE0->ENABLE = UARTE_ENABLE_ENABLE_Disabled << UARTE_ENABLE_ENABLE_Pos;

	// バッファ初期化
	pf_uart_init_buf(PF_UART_BUF_ID_TXD);
	pf_uart_init_buf(PF_UART_BUF_ID_RXD);
	pf_uart_init_dma();
	pf_uart_init_error();

	// イベントのショートカット(ENDRXと同時にSTARTRXを行う)
	NRF_UARTE0->SHORTS = UARTE_SHORTS_ENDRX_STARTRX_Enabled << UARTE_SHORTS_ENDRX_STARTRX_Pos;

	// 割り込み有効化
	NRF_UARTE0->INTEN = (UARTE_INTEN_ENDRX_Enabled << UARTE_INTEN_ENDRX_Pos)
					| (UARTE_INTEN_ENDTX_Enabled << UARTE_INTEN_ENDTX_Pos)
					| (UARTE_INTEN_ERROR_Enabled << UARTE_INTEN_ERROR_Pos)
					| (UARTE_INTEN_RXSTARTED_Enabled << UARTE_INTEN_RXSTARTED_Pos);

	// 割り込みセットアップ
	pf_interrupt_local_setup(PF_INTERRUPT_PRI_UART);

	// 受信エラーをすべてクリア
	NRF_UARTE0->ERRORSRC = (UARTE_ERRORSRC_OVERRUN_NotPresent << UARTE_ERRORSRC_OVERRUN_Pos)
					| (UARTE_ERRORSRC_PARITY_NotPresent << UARTE_ERRORSRC_PARITY_Pos)
					| (UARTE_ERRORSRC_FRAMING_NotPresent << UARTE_ERRORSRC_FRAMING_Pos)
					| (UARTE_ERRORSRC_BREAK_NotPresent << UARTE_ERRORSRC_BREAK_Pos);

	// RTSピン設定(使用しない)
	NRF_UARTE0->PSEL.RTS |= UARTE_PSEL_RTS_CONNECT_Disconnected << UARTE_PSEL_RTS_CONNECT_Pos;

	// TXDピン設定(pf_gpioで管理しているポート番号・ピン番号に接続する)
	NRF_UARTE0->PSEL.TXD = (pf_gpio_get_pin(PF_GPIO_ID_UART_TXD) << UARTE_PSEL_TXD_PIN_Pos)
					| (pf_gpio_get_port(PF_GPIO_ID_UART_TXD) << UARTE_PSEL_TXD_PORT_Pos)
					| (UARTE_PSEL_TXD_CONNECT_Connected << UARTE_PSEL_TXD_CONNECT_Pos);

	//  CTSピン設定(使用しない)
	NRF_UARTE0->PSEL.CTS |= UARTE_PSEL_CTS_CONNECT_Disconnected << UARTE_PSEL_CTS_CONNECT_Pos;

	// RXDピン設定(pf_gpioで管理しているポート番号・ピン番号に接続する)
	NRF_UARTE0->PSEL.RXD = (pf_gpio_get_pin(PF_GPIO_ID_UART_RXD) << UARTE_PSEL_RXD_PIN_Pos)
					| (pf_gpio_get_port(PF_GPIO_ID_UART_RXD) << UARTE_PSEL_RXD_PORT_Pos)
					| (UARTE_PSEL_RXD_CONNECT_Connected << UARTE_PSEL_RXD_CONNECT_Pos);

	// ボーレート115200bps(実際のボーレートは115108bps、誤差-0.08%)
	NRF_UARTE0->BAUDRATE = UARTE_BAUDRATE_BAUDRATE_Baud115200 << UARTE_BAUDRATE_BAUDRATE_Pos;

	// ハードウェアフロー無し、パリティ無し、ストップビット1bit
	NRF_UARTE0->CONFIG = (UARTE_CONFIG_HWFC_Disabled << UARTE_CONFIG_HWFC_Pos)
					| (UARTE_CONFIG_PARITY_Excluded << UARTE_CONFIG_PARITY_Pos)
					| (UARTE_CONFIG_STOP_One << UARTE_CONFIG_STOP_Pos)
					| (UARTE_CONFIG_PARITYTYPE_Even << UARTE_CONFIG_PARITYTYPE_Pos);

	// UART有効化
	NRF_UARTE0->ENABLE = UARTE_ENABLE_ENABLE_Enabled << UARTE_ENABLE_ENABLE_Pos;

	// 受信開始(EasyDMA)
	NRF_UARTE0->RXD.PTR = (u4)pf_uart_dma;
	NRF_UARTE0->RXD.MAXCNT = sizeof(pf_uart_dma);
	NRF_UARTE0->TASKS_STARTRX = UART_TASKS_STARTRX_TASKS_STARTRX_Trigger
					<< UART_TASKS_STARTRX_TASKS_STARTRX_Pos;

	// 1回のUART受信を予約した
	pf_uart_buf[PF_UART_BUF_ID_RXD].uart_process_count++;
}

//! @brief		UART定期タスク
//! @remarks	プラットフォーム定期タスク(入力系)処理から呼び出すこと
void pf_uart_task(void)
{
	u4 enable;

	// オート変数初期化
	enable = 0;

	// UART割り込み禁止
	enable = pf_interrupt_local_disable(PF_INTERRUPT_PRI_UART);

	// ポーリング
	pf_uart_poll_txd();
	pf_uart_poll_rxd();

	// UART割り込み復元
	pf_interrupt_local_restore(PF_INTERRUPT_PRI_UART, enable);
}

//! @brief		UART送信
//! @param		[in] buf		送信バッファへのポインタ
//! @param		[in] bytes		送信バイト数
void pf_uart_send(const u1 *buf, u4 bytes)
{
	u4 enable;

	// オート変数初期化
	enable = 0;

	// パラメータチェック
	if ((NULL != buf) && (bytes > 0))
	{
		// UART割り込み禁止
		enable = pf_interrupt_local_disable(PF_INTERRUPT_PRI_UART);

		// 全てのバイト数を処理し終わるまで
		while (bytes > 0)
		{
			// 送信バッファが一杯？
			if (pf_uart_buf[PF_UART_BUF_ID_TXD].valid_bytes
							< sizeof(pf_uart_buf[PF_UART_BUF_ID_TXD].buf))
			{
				// 送信バッファへ1バイト挿入
				pf_uart_buf[PF_UART_BUF_ID_TXD].buf[pf_uart_buf[PF_UART_BUF_ID_TXD].write_pos] =
								*buf;
				buf++;

				// write_posを進める
				pf_uart_buf[PF_UART_BUF_ID_TXD].write_pos++;
				pf_uart_buf[PF_UART_BUF_ID_TXD].write_pos &=
								(sizeof(pf_uart_buf[PF_UART_BUF_ID_TXD].buf) - 1);

				// valid_bytesをインクリメント
				pf_uart_buf[PF_UART_BUF_ID_TXD].valid_bytes++;
			}
			else
			{
				// 送信バッファが一杯なので、送信データを捨てる
			}

			// 1バイト処理した
			bytes--;
		}

		// 送信可能であれば、送信する
		pf_uart_poll_txd();

		// UART割り込み復元
		pf_interrupt_local_restore(PF_INTERRUPT_PRI_UART, enable);
	}
}

//! @brief		UART受信
//! @param		[out] buf		受信バッファへのポインタ
//! @param		[in] bytes		受信バッファの最大数
//! @return		受信したバイト数
u4 pf_uart_recv(u1 *buf, u4 bytes)
{
	u4 enable;
	u4 recv;

	// オート変数初期化
	enable = 0;
	recv = 0;

	// パラメータチェック
	if ((NULL != buf) && (bytes > 0))
	{
		// UART割り込み禁止
		enable = pf_interrupt_local_disable(PF_INTERRUPT_PRI_UART);

		// 全てのバイト数を処理し終わるまで
		while ((recv < bytes) && (pf_uart_buf[PF_UART_BUF_ID_RXD].valid_bytes > 0))
		{
			// 受信バッファへ1バイト転送
			*buf = pf_uart_buf[PF_UART_BUF_ID_RXD].buf[pf_uart_buf[PF_UART_BUF_ID_RXD].read_pos];
			buf++;

			// read_posを進める
			pf_uart_buf[PF_UART_BUF_ID_RXD].read_pos++;
			pf_uart_buf[PF_UART_BUF_ID_RXD].read_pos &= (sizeof(pf_uart_buf[PF_UART_BUF_ID_RXD].buf)
							- 1);

			// valid_bytesをデクリメント
			pf_uart_buf[PF_UART_BUF_ID_RXD].valid_bytes--;

			// 受信したバイト数をインクリメント
			recv++;
		}

		// UART割り込み復元
		pf_interrupt_local_restore(PF_INTERRUPT_PRI_UART, enable);
	}

	return recv;
}

//! @brief		UARTエラー情報取得
//! @param		[out] error		エラー情報構造体へのポインタ
//! @remarks	プラットフォーム内部のエラー情報はクリアされる
void pf_uart_error(PF_UART_ERROR *error)
{
	u4 enable;

	// オート変数初期化
	enable = 0;

	// パラメータチェック
	if (NULL != error)
	{
		// UART割り込み禁止
		enable = pf_interrupt_local_disable(PF_INTERRUPT_PRI_UART);

		// エラー情報を転送
		error->overrun = pf_uart_rxderr.overrun;
		error->parity = pf_uart_rxderr.parity;
		error->framing = pf_uart_rxderr.framing;
		error->brk = pf_uart_rxderr.brk;

		// プラットフォーム内部のエラー情報をクリア
		pf_uart_rxderr.overrun = 0;
		pf_uart_rxderr.parity = 0;
		pf_uart_rxderr.framing = 0;
		pf_uart_rxderr.brk = 0;

		// UART割り込み復元
		pf_interrupt_local_restore(PF_INTERRUPT_PRI_UART, enable);
	}
}

//! @brief		UART1文字出力
//! @param		[in] ch			送信する文字(0xFF以外)
void pf_uart_putc(u1 ch)
{
	// 受信除外文字でない場合
	if (PF_UART_EMPTY_CODE != ch)
	{
		// UART送信
		pf_uart_send(&ch, 1);
	}
}

//! @brief		UART1文字入力
//! @return		受信した文字(受信していない場合、0xFF)
u1 pf_uart_getc(void)
{
	u1 ch;

	// オート変数初期化
	ch = PF_UART_EMPTY_CODE;

	// UART受信(戻り値は捨てる)
	(void)pf_uart_recv(&ch, 1);

	// 受信バッファが空の場合、chは変化しない
	return ch;
}

//! @brief		システム時間のフォーマット
//! @param		[out] buf		格納バッファ
//! @param		[in] format		printf形式の書式指定文字列
//! @attention	グローバル割り込み禁止状態で呼び出さないこと
static void pf_uart_log_time(char *buf, const char *format, ...)
{
	va_list ap;

	// 可変長引数の処理を開始する
	va_start(ap, format);

	// フォーマット
	vsnprintf(buf, 15, format, ap);

	// 可変長引数の処理を終了する
	va_end(ap);
}

//! @brief		UARTログ出力
//! @param		[in] format		printf形式の書式指定文字列(末尾の改行はあってもなくてもよい)
//! @attention	グローバル割り込み禁止状態で呼び出さないこと
void pf_uart_log(const char *format, ...)
{
	PF_SYSTICK_TIME timebuf;
	char buf[0x100];
	u4 bytes;
	va_list ap;

	// オート変数初期化
	timebuf.sec = 0;
	timebuf.ms = 0;
	timebuf.sec = 0;
	bytes = 0;

	// 可変長引数の処理を開始する
	va_start(ap, format);

	// SysTick時刻を取得
	pf_systick_time(&timebuf);

	// SysTick時刻をフォーマット
	pf_uart_log_time(buf, "%05d,%03d,%03d,", timebuf.sec, timebuf.ms, timebuf.us);

	// 本体のフォーマット
	bytes = (u4)vsnprintf(&buf[14], sizeof(buf) - 16, format, ap);
	bytes += 14;

	// 末尾に改行文字があるか？
	if (buf[bytes - 1] == '\n')
	{
		// 改行文字を削る
		bytes--;
	}

	// 改行文字を追加
	buf[bytes] = '\r';
	bytes++;
	buf[bytes] = '\n';
	bytes++;

	// UART送信
	pf_uart_send((const u1*)buf, bytes);

	// 可変長引数の処理を終了する
	va_end(ap);
}

//! @brief		UART割り込みハンドラ(ENDTX)
//! @attention	データ競合(割り込み干渉)に注意する
static void pf_uart_isr_endtx(void)
{
	// 1回のUART送信が完了した
	pf_uart_buf[PF_UART_BUF_ID_TXD].uart_process_count--;

	// 通常コンテキストの動作と同一
	pf_uart_poll_txd();

	// UART送信が必要ない場合、省電力のため送信を止める
	if (0 == pf_uart_buf[PF_UART_BUF_ID_TXD].uart_process_count)
	{
		NRF_UARTE0->TASKS_STOPTX = UART_TASKS_STOPTX_TASKS_STOPTX_Trigger
						<< UART_TASKS_STOPTX_TASKS_STOPTX_Pos;
	}
}

//! @brief		UART割り込みハンドラ(ERROR)
//! @attention	データ競合(割り込み干渉)に注意する
static void pf_uart_isr_error(void)
{
	u4 overrun;
	u4 parity;
	u4 framing;
	u4 brk;

	// オート変数初期化
	overrun = UART_ERRORSRC_OVERRUN_Present << UART_ERRORSRC_OVERRUN_Pos;
	parity = UART_ERRORSRC_PARITY_Present << UART_ERRORSRC_PARITY_Pos;
	framing = UART_ERRORSRC_FRAMING_Present << UART_ERRORSRC_FRAMING_Pos;
	brk = UART_ERRORSRC_BREAK_Present << UART_ERRORSRC_BREAK_Pos;

	// オーバーランエラー
	if (overrun == (NRF_UARTE0->ERRORSRC & overrun))
	{
		NRF_UARTE0->ERRORSRC &= (u4)(~UART_ERRORSRC_OVERRUN_Msk);
		pf_uart_rxderr.overrun++;
	}

	// パリティエラー
	if (parity == (NRF_UARTE0->ERRORSRC & parity))
	{
		NRF_UARTE0->ERRORSRC &= (u4)(~UART_ERRORSRC_PARITY_Msk);
		pf_uart_rxderr.parity++;
	}

	// フレーミングエラー
	if (framing == (NRF_UARTE0->ERRORSRC & framing))
	{
		NRF_UARTE0->ERRORSRC &= (u4)(~UART_ERRORSRC_FRAMING_Msk);
		pf_uart_rxderr.framing++;
	}

	// ブレーク信号受信
	if (brk == (NRF_UARTE0->ERRORSRC & brk))
	{
		NRF_UARTE0->ERRORSRC &= (u4)(~UART_ERRORSRC_BREAK_Msk);
		pf_uart_rxderr.brk++;
	}
}

//! @brief		UART割り込みハンドラ(RXSTARTED)
//! @attention	データ競合(割り込み干渉)に注意する
static void pf_uart_isr_rxstarted(void)
{
	// 受信バッファを再セットできるか？
	if (pf_uart_buf[PF_UART_BUF_ID_RXD].uart_process_count < 2)
	{
		// 受信バッファを再セット
		NRF_UARTE0->RXD.PTR = (u4)pf_uart_dma;

		// 1回のUART受信を予約した
		pf_uart_buf[PF_UART_BUF_ID_RXD].uart_process_count++;
	}
}

//! @brief		UART割り込みハンドラ(ENDRX)
//! @attention	データ競合(割り込み干渉)に注意する
static void pf_uart_isr_endrx(void)
{
	// 受信バッファの再セット
	pf_uart_isr_rxstarted();

	// 1回のUART受信が終了した
	pf_uart_buf[PF_UART_BUF_ID_RXD].uart_process_count--;

	// 通常コンテキストの動作と同一
	pf_uart_poll_rxd();
}

//! @brief		UART割り込みハンドラ
//! @attention	データ競合(割り込み干渉)に注意する
void UARTE0_UART0_IRQHandler(void)
{
	// ENDRX
	if ((UARTE_EVENTS_ENDRX_EVENTS_ENDRX_Generated << UARTE_EVENTS_ENDRX_EVENTS_ENDRX_Pos)
					== NRF_UARTE0->EVENTS_ENDRX)
	{
		NRF_UARTE0->EVENTS_ENDRX = UARTE_EVENTS_ENDRX_EVENTS_ENDRX_NotGenerated
						<< UARTE_EVENTS_ENDRX_EVENTS_ENDRX_Pos;
		pf_uart_isr_endrx();
	}

	// ENDTX
	if ((UARTE_EVENTS_ENDTX_EVENTS_ENDTX_Generated << UARTE_EVENTS_ENDTX_EVENTS_ENDTX_Pos)
					== NRF_UARTE0->EVENTS_ENDTX)
	{
		NRF_UARTE0->EVENTS_ENDTX = UARTE_EVENTS_ENDTX_EVENTS_ENDTX_NotGenerated
						<< UARTE_EVENTS_ENDTX_EVENTS_ENDTX_Pos;
		pf_uart_isr_endtx();
	}

	// ERROR
	if ((UARTE_EVENTS_ERROR_EVENTS_ERROR_Generated << UARTE_EVENTS_ERROR_EVENTS_ERROR_Pos)
					== NRF_UARTE0->EVENTS_ERROR)
	{
		NRF_UARTE0->EVENTS_ERROR = UARTE_EVENTS_ERROR_EVENTS_ERROR_NotGenerated
						<< UARTE_EVENTS_ERROR_EVENTS_ERROR_Pos;
		pf_uart_isr_error();
	}

	// RXSTARTED
	if ((UARTE_EVENTS_RXSTARTED_EVENTS_RXSTARTED_Generated
					<< UARTE_EVENTS_RXSTARTED_EVENTS_RXSTARTED_Pos) == NRF_UARTE0->EVENTS_RXSTARTED)
	{
		NRF_UARTE0->EVENTS_RXSTARTED =
		UARTE_EVENTS_RXSTARTED_EVENTS_RXSTARTED_NotGenerated
						<< UARTE_EVENTS_RXSTARTED_EVENTS_RXSTARTED_Pos;
		pf_uart_isr_rxstarted();
	}
}
