//! @file		pf_i2c.c
//! @brief		プラットフォーム(I2C)実装ファイル

// The MIT License (MIT)
// Copyright (c) 2023 @xm6_original
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include "pf_types.h"
#include "nrf52833.h"
#include "nrf52833_bitfields.h"
#include "pf_gpio.h"
#include "pf_interrupt.h"
#include "pf_i2c.h"

//! @brief		I2Cチャネルの最大数
#define PF_I2C_CHANNEL_MAX			((u4)2U)

//! @brief		I2Cアドレス情報構造体
typedef struct PF_I2C_ADDR_Tag
{
	u4				addr;					//!< I2Cアドレス
	u4				channel;				//!< I2Cチャネル(0=内部/1=外部)
} PF_I2C_ADDR;

//! @brief		I2Cアドレス情報テーブル
static const PF_I2C_ADDR pf_i2c_addr_table[PF_I2C_ID_MAX] =
{
	// PF_I2C_ID_ACCELEROMETER
	{
		0x19,
		0,
	},

	// PF_I2C_ID_MAGNETOMETER
	{
		0x1E,
		0,
	},

	// PF_I2C_ID_MAQUEEN_MOTOR
	{
		0x10,
		1,
	},
};

//! @brief		I2C動作情報構造体
typedef struct PF_I2C_INFO_Tag
{
	PF_I2C_CALLBACK	callback;				//!< コールバック関数
	BOOL			busy;					//!< BUSYフラグ
	BOOL			status;					//!< 通信ステータス(TRUE=成功/FALSE=失敗)
	u4				overrun;				//!< オーバーラン発生回数(受信動作が不正)
	u4				anack;					//!< アドレスNACK発生回数
	u4				dnack;					//!< データNACK発生回数
	u1				reg;					//!< 受信レジスタ
} PF_I2C_INFO;

//! @brief		I2C動作情報テーブル
static PF_I2C_INFO pf_i2c_info[PF_I2C_ID_MAX];

//! @brief		I2Cチャネル→デバイステーブル
static NRF_TWIM_Type* const pf_i2c_channel_to_dev[PF_I2C_CHANNEL_MAX] =
{
	NRF_TWIM0,								//!< チャネル0(内部)
	NRF_TWIM1,								//!< チャネル1(外部)
};

//! @brief		I2Cチャネル→割り込み優先度テーブル
static const PF_INTERRUPT_PRI pf_i2c_channel_to_pri[PF_I2C_CHANNEL_MAX] =
{
	PF_INTERRUPT_PRI_I2C_INT,				//!< チャネル0(内部)
	PF_INTERRUPT_PRI_I2C_EXT,				//!< チャネル1(外部)
};

//! @brief		I2C割り込み禁止
//! @param		[in] channel	I2Cチャネル(0=内部/1=外部)
//! @return		直前のI2C割り込み禁止状態(1:割り込み禁止/0:割り込み許可)
static u4 pf_i2c_interrupt_disable(u4 channel)
{
	u4 enable;
	PF_INTERRUPT_PRI pri;

	// オート変数初期化
	enable = 0;
	pri = pf_i2c_channel_to_pri[channel];

	// ローカル割り込み禁止
	enable = pf_interrupt_local_disable(pri);

	return enable;
}

//! @brief		I2C割り込み復元
//! @param		[in] channel	I2Cチャネル(0=内部/1=外部)
//! @param		[in] enable		pf_i2c_interrupt_disable(channel)の返り値
static void pf_i2c_interrupt_restore(u4 channel, u4 enable)
{
	PF_INTERRUPT_PRI pri;

	// オート変数初期化
	pri = pf_i2c_channel_to_pri[channel];

	// ローカル割り込み復元
	pf_interrupt_local_restore(pri, enable);
}

//! @brief		I2CデバイスBUSYチェック
//! @param		[in] channel	I2Cチャネル(0=内部/1=外部)
//! @return		デバイスBUSY状態(TRUE=デバイスBUSY/FALSE=デバイスREADY)
//! @attention	I2C割り込み禁止状態で呼び出すこと
static BOOL pf_i2c_get_busy(u4 channel)
{
	BOOL busy;
	PF_I2C_ID id;

	// オート変数初期化
	busy = FALSE;
	id = 0;

	// すべてのIDをループ
	for (id = 0; id < PF_I2C_ID_MAX; id++)
	{
		// 当該チャネルに該当するか
		if (channel == pf_i2c_addr_table[id].channel)
		{
			// BUSYであれば
			if (TRUE == pf_i2c_info[id].busy)
			{
				// そのチャネルはBUSY
				busy = TRUE;
				break;
			}
		}
	}

	return busy;
}

//! @brief		動作中のIDを取得
//! @param		[in] channel	I2Cチャネル(0=内部/1=外部)
//! @return		I2C通信先のID
//! @attention	I2C割り込み禁止状態で呼び出すこと
static PF_I2C_ID pf_i2c_get_id(u4 channel)
{
	PF_I2C_ID id;

	// オート変数初期化
	id = 0;

	// すべてのIDをループ
	for (id = 0; id < PF_I2C_ID_MAX; id++)
	{
		// 当該チャネルに該当するか
		if (channel == pf_i2c_addr_table[id].channel)
		{
			// BUSYであれば
			if (TRUE == pf_i2c_info[id].busy)
			{
				// このIDが該当する
				break;
			}
		}
	}

	return id;
}

//! @brief		I2C初期化(単一のID)
//! @param		[in] id			I2C通信先のID
static void pf_i2c_init_id(PF_I2C_ID id)
{
	// コールバック関数なし
	pf_i2c_info[id].callback = NULL;

	// デバイスREADY
	pf_i2c_info[id].busy = FALSE;

	// エラーなし
	pf_i2c_info[id].overrun = 0;
	pf_i2c_info[id].anack = 0;
	pf_i2c_info[id].dnack = 0;

	// 受信レジスタ0
	pf_i2c_info[id].reg = 0;
}

//! @brief		I2C初期化(デバイス)
//! @param		[in] channel	I2Cチャネル(0=内部/1=外部)
static void pf_i2c_init_dev(u4 channel)
{
	NRF_TWIM_Type *dev;
	u4 scl_port;
	u4 scl_pin;
	u4 sda_port;
	u4 sda_pin;

	// オート変数初期化
	dev = pf_i2c_channel_to_dev[channel];
	scl_port = 0;
	scl_pin = 0;
	sda_port = 0;
	sda_pin = 0;

	// I2C無効化
	dev->ENABLE = TWIM_ENABLE_ENABLE_Disabled << TWIM_ENABLE_ENABLE_Pos;

	// 割り込み有効化(転送完了およびエラー)
	dev->INTEN = (TWIM_INTEN_STOPPED_Enabled << TWIM_INTEN_STOPPED_Pos)
					| (TWIM_INTEN_ERROR_Enabled << TWIM_INTEN_ERROR_Pos);

	// 割り込みセットアップ
	pf_interrupt_local_setup(pf_i2c_channel_to_pri[channel]);

	// 転送速度(400kbps)
	dev->FREQUENCY = TWIM_FREQUENCY_FREQUENCY_K400 << TWIM_FREQUENCY_FREQUENCY_Pos;

	// pf_gpioよりポート番号およびピン番号を取得
	if (0 == channel)
	{
		// SCL(チャネル0)
		scl_port = pf_gpio_get_port(PF_GPIO_ID_INTERNAL_SCL);
		scl_pin = pf_gpio_get_pin(PF_GPIO_ID_INTERNAL_SCL);

		// SDA(チャネル0)
		sda_port = pf_gpio_get_port(PF_GPIO_ID_INTERNAL_SDA);
		sda_pin = pf_gpio_get_pin(PF_GPIO_ID_INTERNAL_SDA);
	}
	else
	{
		// SCL(チャネル1)
		scl_port = pf_gpio_get_port(PF_GPIO_ID_EXTERNAL_SCL);
		scl_pin = pf_gpio_get_pin(PF_GPIO_ID_EXTERNAL_SCL);

		// SDA(チャネル1)
		sda_port = pf_gpio_get_port(PF_GPIO_ID_EXTERNAL_SDA);
		sda_pin = pf_gpio_get_pin(PF_GPIO_ID_EXTERNAL_SDA);
	}

	// SCLピン設定(pf_gpioで管理しているポート番号・ピン番号に接続する)
	dev->PSEL.SCL = (scl_pin << TWIM_PSEL_SCL_PIN_Pos) | (scl_port << TWIM_PSEL_SCL_PORT_Pos)
					| (TWIM_PSEL_SCL_CONNECT_Connected << TWIM_PSEL_SCL_CONNECT_Pos);

	// SDAピン設定
	dev->PSEL.SDA = (sda_pin << TWIM_PSEL_SDA_PIN_Pos) | (sda_port << TWIM_PSEL_SDA_PORT_Pos)
					| (TWIM_PSEL_SDA_CONNECT_Connected << TWIM_PSEL_SDA_CONNECT_Pos);

	// Array Listは使用しない
	dev->RXD.LIST = TWIM_RXD_LIST_LIST_Disabled << TWIM_RXD_LIST_LIST_Pos;
	dev->TXD.LIST = TWIM_TXD_LIST_LIST_Disabled << TWIM_TXD_LIST_LIST_Pos;

	// I2C有効化
	dev->ENABLE = TWIM_ENABLE_ENABLE_Enabled << TWIM_ENABLE_ENABLE_Pos;
}

//! @brief		I2C初期化
//! @remarks	プラットフォーム初期化処理から呼び出すこと
void pf_i2c_init(void)
{
	PF_I2C_ID id;

	// オート変数初期化
	id = 0;

	// すべてのIDをループ
	for (id = 0; id < PF_I2C_ID_MAX; id++)
	{
		// 1つのIDを初期化
		pf_i2c_init_id(id);
	}

	// デバイス初期化(内部)
	pf_i2c_init_dev(0);

	// デバイス初期化(外部)
	pf_i2c_init_dev(1);
}

//! @brief		I2C送信(デバイス)
//! @param		[in] channel	I2Cチャネル(0=内部/1=外部)
//! @param		[in] addr		I2Cアドレス
//! @param		[in] buf		送信バッファへのポインタ
//! @param		[in] bytes		送信バイト数
//! @attention	デバイスREADYの場合に限り呼び出すこと
static void pf_i2c_send_dev(u4 channel, u4 addr, const u1 *buf, u4 bytes)
{
	NRF_TWIM_Type *dev;

	// オート変数初期化
	dev = pf_i2c_channel_to_dev[channel];

	// イベントのショートカット
	dev->SHORTS = TWIM_SHORTS_LASTTX_STOP_Enabled << TWIM_SHORTS_LASTTX_STOP_Pos;

	// 通信先I2Cアドレス
	dev->ADDRESS = addr;

	// 送信バッファ
	dev->TXD.PTR = (u4)buf;

	// 送信バイト数
	dev->TXD.MAXCNT = bytes;

	// 送信開始
	dev->TASKS_STARTTX = TWIM_TASKS_STARTTX_TASKS_STARTTX_Trigger
					<< TWIM_TASKS_STARTTX_TASKS_STARTTX_Pos;
}

//! @brief		I2C送信
//! @param		[in] id			I2C通信先のID
//! @param		[in] buf		送信バッファへのポインタ
//! @param		[in] bytes		送信バイト数
//! @return		送信ステータス(TRUE=送信開始/FALSE=デバイスBUSY)
//! @attention	呼び出し側で送信バッファのライフタイムを保証すること
BOOL pf_i2c_send(PF_I2C_ID id, const u1 *buf, u4 bytes)
{
	u4 channel;
	u4 enable;
	BOOL busy;
	BOOL result;

	// オート変数初期化
	channel = 0;
	enable = 0;
	busy = FALSE;
	result = FALSE;

	// パラメータチェック
	if ((id < PF_I2C_ID_MAX) && (NULL != buf) && (bytes > 0))
	{
		// チャネル取得
		channel = pf_i2c_addr_table[id].channel;

		// I2C割り込み禁止
		enable = pf_i2c_interrupt_disable(channel);

		// デバイスBUSYチェック
		busy = pf_i2c_get_busy(channel);

		// READYであれば送信開始
		if (FALSE == busy)
		{
			// 通信成功に初期化
			pf_i2c_info[id].status = TRUE;

			// 送信開始
			pf_i2c_send_dev(channel, pf_i2c_addr_table[id].addr, buf, bytes);

			// BUSYへ変更
			pf_i2c_info[id].busy = TRUE;

			// 送信開始に成功した
			result = TRUE;
		}

		// I2C割り込み復元
		pf_i2c_interrupt_restore(channel, enable);
	}

	return result;
}

//! @brief		I2C受信(デバイス)
//! @param		[in] channel	I2Cチャネル(0=内部/1=外部)
//! @param		[in] addr		I2Cアドレス
//! @param		[in] reg		受信レジスタへのポインタ
//! @param		[in] buf		受信バッファへのポインタ
//! @param		[in] bytes		受信バイト数
//! @attention	デバイスREADYの場合に限り呼び出すこと
static void pf_i2c_recv_dev(u4 channel, u4 addr, const u1 *reg, u1 *buf, u4 bytes)
{
	NRF_TWIM_Type *dev;

	// オート変数初期化
	dev = pf_i2c_channel_to_dev[channel];

	// イベントのショートカット
	dev->SHORTS = (TWIM_SHORTS_LASTTX_STARTRX_Enabled << TWIM_SHORTS_LASTTX_STARTRX_Pos)
					| (TWIM_SHORTS_LASTRX_STOP_Enabled << TWIM_SHORTS_LASTRX_STOP_Pos);

	// 通信先I2Cアドレス
	dev->ADDRESS = addr;

	// 送信バッファ(受信レジスタ)
	dev->TXD.PTR = (u4)reg;

	// 送信バイト数
	dev->TXD.MAXCNT = 1;

	// 受信バッファ
	dev->RXD.PTR = (u4)buf;

	// 送信バイト数
	dev->RXD.MAXCNT = bytes;

	// レジスタ送信開始(レジスタ送信完了後、受信動作に移行する)
	dev->TASKS_STARTTX = TWIM_TASKS_STARTTX_TASKS_STARTTX_Trigger
					<< TWIM_TASKS_STARTTX_TASKS_STARTTX_Pos;
}

//! @brief		I2C受信
//! @param		[in] id			I2C通信先のID
//! @param		[in] reg		受信レジスタ
//! @param		[in] buf		受信バッファへのポインタ
//! @param		[in] bytes		受信バイト数
//! @return		受信ステータス(TRUE=受信開始/FALSE=デバイスBUSY)
BOOL pf_i2c_recv(PF_I2C_ID id, u1 reg, u1 *buf, u4 bytes)
{
	u4 channel;
	u4 enable;
	BOOL busy;
	BOOL result;

	// オート変数初期化
	channel = 0;
	enable = 0;
	busy = FALSE;
	result = FALSE;

	// パラメータチェック
	if ((id < PF_I2C_ID_MAX) && (NULL != buf) && (bytes > 0))
	{
		// チャネル取得
		channel = pf_i2c_addr_table[id].channel;

		// I2C割り込み禁止
		enable = pf_i2c_interrupt_disable(channel);

		// デバイスBUSYチェック
		busy = pf_i2c_get_busy(channel);

		// READYであれば送受信開始
		if (FALSE == busy)
		{
			// 受信レジスタを記憶
			pf_i2c_info[id].reg = reg;

			// 通信成功に初期化
			pf_i2c_info[id].status = TRUE;

			// 送受信開始
			pf_i2c_recv_dev(channel, pf_i2c_addr_table[id].addr, &pf_i2c_info[id].reg, buf, bytes);

			// BUSYへ変更
			pf_i2c_info[id].busy = TRUE;

			// 送受信開始に成功した
			result = TRUE;
		}

		// I2C割り込み復元
		pf_i2c_interrupt_restore(channel, enable);
	}

	return result;
}

//! @brief		I2Cエラー情報取得
//! @param		[in] id			I2C通信先のID
//! @param		[out] error		エラー情報構造体へのポインタ
//! @remarks	プラットフォーム内部のエラー情報はクリアされる
void pf_i2c_error(PF_I2C_ID id, PF_I2C_ERROR *error)
{
	u4 channel;
	u4 enable;

	// オート変数初期化
	channel = 0;
	enable = 0;

	// パラメータチェック
	if ((id < PF_I2C_ID_MAX) && (NULL != error))
	{
		// チャネル取得
		channel = pf_i2c_addr_table[id].channel;

		// I2C割り込み禁止
		enable = pf_i2c_interrupt_disable(channel);

		// エラー情報を転送
		error->overrun = pf_i2c_info[id].overrun;
		error->anack = pf_i2c_info[id].anack;
		error->dnack = pf_i2c_info[id].dnack;

		// プラットフォーム内部のエラー情報をクリア
		pf_i2c_info[id].overrun = 0;
		pf_i2c_info[id].anack = 0;
		pf_i2c_info[id].dnack = 0;

		// I2C割り込み復元
		pf_i2c_interrupt_restore(channel, enable);
	}
}

//! @brief		I2Cコールバック関数設定
//! @param		[in] id			I2C通信先のID
//! @param		[in] func		I2Cコールバック関数へのポインタ
//! @attention	I2Cコールバック関数は割り込みコンテキストで呼び出される
void pf_i2c_callback(PF_I2C_ID id, PF_I2C_CALLBACK func)
{
	u4 channel;
	u4 enable;

	// オート変数初期化
	channel = 0;
	enable = 0;

	// IDチェック
	if (id < PF_I2C_ID_MAX)
	{
		// チャネル取得
		channel = pf_i2c_addr_table[id].channel;

		// I2C割り込み禁止
		enable = pf_i2c_interrupt_disable(channel);

		// コールバック関数設定
		pf_i2c_info[id].callback = func;

		// I2C割り込み復元
		pf_i2c_interrupt_restore(channel, enable);
	}
}

//! @brief		I2C割り込みハンドラ(STOPPED)
//! @param		[in] channel	I2Cチャネル(0=内部/1=外部)
//! @attention	データ競合(割り込み干渉)に注意する
static void pf_i2c_isr_stopped(u4 channel)
{
	PF_I2C_ID id;

	// オート変数初期化
	id = PF_I2C_ID_ACCELEROMETER;

	// チャネルから通信先IDを逆引きする
	id = pf_i2c_get_id(channel);

	// 通信終了
	pf_i2c_info[id].busy = FALSE;

	// I2Cコールバック関数が登録されていれば、呼び出す
	if (NULL != pf_i2c_info[id].callback)
	{
		pf_i2c_info[id].callback(pf_i2c_info[id].status);
	}
}

//! @brief		I2C割り込みハンドラ(ERROR)
//! @param		[in] channel	I2Cチャネル(0=内部/1=外部)
//! @attention	データ競合(割り込み干渉)に注意する
static void pf_i2c_isr_error(u4 channel)
{
	PF_I2C_ID id;
	NRF_TWIM_Type *dev;
	u4 overrun;
	u4 anack;
	u4 dnack;

	// オート変数初期化
	id = PF_I2C_ID_ACCELEROMETER;
	dev = pf_i2c_channel_to_dev[channel];
	overrun = TWIM_ERRORSRC_OVERRUN_Received << TWIM_ERRORSRC_OVERRUN_Pos;
	anack = TWIM_ERRORSRC_ANACK_Received << TWIM_ERRORSRC_ANACK_Pos;
	dnack = TWIM_ERRORSRC_DNACK_Received << TWIM_ERRORSRC_DNACK_Pos;

	// チャネルから通信先IDを逆引きする
	id = pf_i2c_get_id(channel);

	// 通信エラーあり(送信失敗 or 受信失敗)
	pf_i2c_info[id].status = FALSE;

	// エラー種別判定(ANACK)
	if (anack == (dev->ERRORSRC & anack))
	{
		// ANACK発生。'1'を書き込むことでクリアする
		pf_i2c_info[id].anack++;
		dev->ERRORSRC = anack;
	}

	// エラー種別判定(DNACK)
	if (dnack == (dev->ERRORSRC & dnack))
	{
		// DNACK発生。'1'を書き込むことでクリアする
		pf_i2c_info[id].dnack++;
		dev->ERRORSRC = dnack;
	}

	// エラー種別判定(OVERRUN)
	if (overrun == (dev->ERRORSRC & overrun))
	{
		// オーバーラン発生。'0'を書き込むことでクリアする
		pf_i2c_info[id].overrun++;
		dev->ERRORSRC = TWIM_ERRORSRC_OVERRUN_NotReceived << TWIM_ERRORSRC_OVERRUN_Pos;
	}

	// ここでI2C通信を打ち切る(この後、pf_i2c_isr_stoppedが呼ばれる)
	dev->TASKS_STOP = TWIM_TASKS_STOP_TASKS_STOP_Trigger << TWIM_TASKS_STOP_TASKS_STOP_Pos;
}

//! @brief		共通割り込みハンドラ
//! @param		[in] channel	I2Cチャネル(0=内部/1=外部)
//! @attention	データ競合(割り込み干渉)に注意する
static void pf_i2c_isr(u4 channel)
{
	NRF_TWIM_Type *dev;

	// オート変数初期化
	dev = pf_i2c_channel_to_dev[channel];

	// STOPPED
	if ((TWIM_EVENTS_STOPPED_EVENTS_STOPPED_Generated << TWIM_EVENTS_STOPPED_EVENTS_STOPPED_Pos)
					== dev->EVENTS_STOPPED)
	{
		dev->EVENTS_STOPPED = TWIM_EVENTS_STOPPED_EVENTS_STOPPED_NotGenerated
						<< TWIM_EVENTS_STOPPED_EVENTS_STOPPED_Pos;
		pf_i2c_isr_stopped(channel);
	}

	// ERROR
	if ((TWIM_EVENTS_ERROR_EVENTS_ERROR_Generated << TWIM_EVENTS_ERROR_EVENTS_ERROR_Pos)
					== dev->EVENTS_ERROR)
	{
		dev->EVENTS_ERROR = TWIM_EVENTS_ERROR_EVENTS_ERROR_NotGenerated
						<< TWIM_EVENTS_ERROR_EVENTS_ERROR_Pos;
		pf_i2c_isr_error(channel);
	}
}

//! @brief		I2C割り込みハンドラ(チャネル0=内部I2Cバス)
//! @attention	データ競合(割り込み干渉)に注意する
void SPIM0_SPIS0_TWIM0_TWIS0_SPI0_TWI0_IRQHandler(void)
{
	pf_i2c_isr(0);
}

//! @brief		I2C割り込みハンドラ(チャネル1=外部I2Cバス)
//! @attention	データ競合(割り込み干渉)に注意する
void SPIM1_SPIS1_TWIM1_TWIS1_SPI1_TWI1_IRQHandler(void)
{
	pf_i2c_isr(1);
}
