//! @file		pf_neopixel.c
//! @brief		プラットフォーム(NeoPixel)実装ファイル

// The MIT License (MIT)
// Copyright (c) 2023 @xm6_original
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include "pf_types.h"
#include "nrf52833.h"
#include "pf_gpio.h"
#include "pf_interrupt.h"
#include "pf_neopixel.h"

//! @brief		NepPixelあたりの送信バイト数
#define PF_NEOPIXEL_SEND_BYTES			((u4)3U)

//! @brief		R/G/Bの階調マスク値
#define PF_NEOPIXEL_COLOR_MASK			((u1)0x7FU)

//! @brief		NeoPixel動作情報構造体
typedef struct PF_NEOPIXEL_INFO_Tag
{
	PF_NEOPIXEL_COLOR	current[PF_NEOPIXEL_ID_MAX];	//!< 現在のRGB
	PF_NEOPIXEL_COLOR	target[PF_NEOPIXEL_ID_MAX];		//!< 目的とするRGB
	u1 buf[PF_NEOPIXEL_ID_MAX][PF_NEOPIXEL_SEND_BYTES];	//!< 送信バッファ
} PF_NEOPIXEL_INFO;

//! @brief		NeoPixel動作情報
static PF_NEOPIXEL_INFO pf_neopixel_info;

//! @brief		NeoPixel初期化(単一のID)
//! @param		[in] id			NeoPixelのID
static void pf_neopixel_init_id(PF_NEOPIXEL_ID id)
{
	// 現在の状態は消灯
	pf_neopixel_info.current[id].r = 0;
	pf_neopixel_info.current[id].g = 0;
	pf_neopixel_info.current[id].b = 0;

	// 目的とする状態は消灯
	pf_neopixel_info.target[id].r = 0;
	pf_neopixel_info.target[id].g = 0;
	pf_neopixel_info.target[id].b = 0;
}

//! @brief		NeoPixel初期化
//! @remarks	プラットフォーム初期化処理から呼び出すこと
void pf_neopixel_init(void)
{
	PF_NEOPIXEL_ID id;

	// オート変数初期化
	id = 0;

	// すべてのIDをループ
	for (id = 0; id < PF_NEOPIXEL_ID_MAX; id++)
	{
		// 1つのIDを初期化
		pf_neopixel_init_id(id);
	}
}

//! @brief		NeoPixel送信(約888kbps)
//! @param		[in] pinmask	NRF_P0->OUTSETまたはNRF_P0->OUTCLRへの出力値
//! @param		[in] buf		送信バッファ
//! @param		[in] bytes		送信バイト数
//! @attention	割り込み禁止区間が非常に長い(約200us)ことに注意する
static void pf_neopixel_888k(u4 pinmask, const u1 *buf, u4 bytes)
{
	u4 enable;
	u4 val;
	u1 data;
	u1 bit;

	// オート変数初期化
	enable = 0;
	val = 0;
	data = 0;
	bit = 0;

	// ms境界まで残り250us未満の場合、次のms開始まで待つ(ラップアラウンドを避ける)
	while (SysTick->VAL < (SystemCoreClock / 4000))
	{
		;
	}

	// 割り込み禁止
	enable = pf_interrupt_global_disable();

	// 現在のSysTickカウンタ
	val = SysTick->VAL;

	// バイトループ
	while (bytes > 0)
	{
		// データフェッチ
		data = *buf;
		bit = 0x80;

		// 8ビットループ
		while (0 != bit)
		{
			// 現在のSysTickカウンタを保存し、H出力
			val = SysTick->VAL;
			NRF_P0->OUTSET = pinmask;

			// 判定
			if (0 != (data & bit))
			{
				// H区間を812.5nsホールド(T1H)
				while ((val - SysTick->VAL) < 52)
				{
					;
				}

				// 現在のSysTickカウンタを保存し、L出力
				val = SysTick->VAL;
				NRF_P0->OUTCLR = pinmask;

				// L区間を312.5nsホールド(T1L)
				while ((val - SysTick->VAL) < 20)
				{
					;
				}
			}
			else
			{
				// H区間を312.5nsホールド(T0H)
				while ((val - SysTick->VAL) < 20)
				{
					;
				}

				// 現在のSysTickカウンタを保存し、L出力
				val = SysTick->VAL;
				NRF_P0->OUTCLR = pinmask;

				// L区間を812.5nsホールド(T0L)
				while ((val - SysTick->VAL) < 52)
				{
					;
				}
			}

			// 次のビットへ
			bit >>= 1;
		}

		// 次のデータへ
		buf++;
		bytes--;
	}

	// 割り込み復元
	pf_interrupt_global_restore(enable);
}

//! @brief		NeoPixel送信
static void pf_neopixel_send(void)
{
	PF_NEOPIXEL_ID id;
	u4 pin;
	u1 *ptr;

	// オート変数初期化
	id = 0;
	pin = 0;
	ptr = NULL;

	// 'L'出力
	pf_gpio_output(PF_GPIO_ID_MAQUEEN_NEOPIXEL, FALSE);

	// すべてのIDをループ
	for (id = 0; id < PF_NEOPIXEL_ID_MAX; id++)
	{
		// 目的とするIDのバッファをポイント
		ptr = &pf_neopixel_info.buf[id][0];

		// 送信データを作成(G)
		*ptr = pf_neopixel_info.target[id].g;
		ptr++;

		// 送信データを作成(R)
		*ptr = pf_neopixel_info.target[id].r;
		ptr++;

		// 送信データを作成(B)
		*ptr = pf_neopixel_info.target[id].b;
		ptr++;
	}

	// GPIOからピンを取得
	pin = pf_gpio_get_pin(PF_GPIO_ID_MAQUEEN_NEOPIXEL);

	// 送信(内部で割り込み禁止されるため処理時間に注意)
	pf_neopixel_888k(1U << pin, &(pf_neopixel_info.buf[0][0]), PF_NEOPIXEL_ID_MAX * PF_NEOPIXEL_SEND_BYTES);
}

//! @brief		NeoPixel定期タスク
//! @remarks	プラットフォーム定期タスク(出力系)処理から呼び出すこと
void pf_neopixel_task(void)
{
	BOOL send;
	PF_NEOPIXEL_ID id;

	// オート変数初期化
	send = FALSE;
	id = 0;

	// すべてのIDをループ
	for (id = 0; id < PF_NEOPIXEL_ID_MAX; id++)
	{
		// R比較
		if (pf_neopixel_info.current[id].r != pf_neopixel_info.target[id].r)
		{
			pf_neopixel_info.current[id].r = pf_neopixel_info.target[id].r;
			send = TRUE;
		}

		// G比較
		if (pf_neopixel_info.current[id].g != pf_neopixel_info.target[id].g)
		{
			pf_neopixel_info.current[id].g = pf_neopixel_info.target[id].g;
			send = TRUE;
		}

		// B比較
		if (pf_neopixel_info.current[id].b != pf_neopixel_info.target[id].b)
		{
			pf_neopixel_info.current[id].b = pf_neopixel_info.target[id].b;
			send = TRUE;
		}
	}

	// 必要に応じて送信
	if (TRUE == send)
	{
		pf_neopixel_send();
	}
}

//! @brief		NeoPixel制御
//! @param		[in] id			NeoPixelのID
//! @param		[in] color		NeoPixelカラー構造体へのポインタ
void pf_neopixel_ctrl(PF_NEOPIXEL_ID id, const PF_NEOPIXEL_COLOR *color)
{
	// パラメータチェック
	if (id < PF_NEOPIXEL_ID_MAX)
	{
		// RGBカラーをセット
		pf_neopixel_info.target[id].r = color->r & PF_NEOPIXEL_COLOR_MASK;
		pf_neopixel_info.target[id].g = color->g & PF_NEOPIXEL_COLOR_MASK;
		pf_neopixel_info.target[id].b = color->b & PF_NEOPIXEL_COLOR_MASK;
	}
}
