//! @file collision.cpp
//! @brief CollisionNX̒`

//--------------------------------------------------------------------------------
// 
// OpenXOPS
// Copyright (c) 2014-2016, OpenXOPS Project / [-_-;](mikan) All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
// * Redistributions of source code must retain the above copyright notice, 
//   this list of conditions and the following disclaimer.
// * Redistributions in binary form must reproduce the above copyright notice, 
//   this list of conditions and the following disclaimer in the documentation 
//   and/or other materials provided with the distribution.
// * Neither the name of the OpenXOPS Project nor the names of its contributors 
//   may be used to endorse or promote products derived from this software 
//   without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL OpenXOPS Project BE LIABLE FOR ANY
// DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
// ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//--------------------------------------------------------------------------------

#include "collision.h"

//! @brief RXgN^
Collision::Collision()
{
	blockdata = NULL;
	cbdata = new Coll_Blockdata[MAX_BLOCKS];
	for(int i=0; i<MAX_BLOCKS; i++){
		cbdata[i].min_x = 0.0f;
		cbdata[i].min_y = 0.0f;
		cbdata[i].min_z = 0.0f;
		cbdata[i].max_x = 0.0f;
		cbdata[i].max_y = 0.0f;
		cbdata[i].max_z = 0.0f;
		cbdata[i].BoardBlock = false;
		cbdata[i].worldgroup = 0;
	}
}

//! @brief fBXgN^
Collision::~Collision()
{
	if( cbdata != NULL ){ delete [] cbdata; }
}

//! @brief ubNf[^荞
//! @param in_blockdata ubNf[^
int Collision::InitCollision(BlockDataInterface* in_blockdata)
{
	int bs;
	struct blockdata data;
	int vID[4];

	if( in_blockdata == NULL ){ return 1; }
	blockdata = in_blockdata;

	bs = blockdata->GetTotaldatas();

	for(int i=0; i<bs; i++){
		cbdata[i].BoardBlock = false;
	}

	for(int i=0; i<bs; i++){
		float cx = 0.0f;
		float cy = 0.0f;
		float cz = 0.0f;
		float x, y, z, rx, ry, r;

		blockdata->Getdata(&data, i);

		//S߂
		for(int j=0; j<8; j++){
			cx += data.x[j];
			cy += data.y[j];
			cz += data.z[j];
		}
		cx /= 8;
		cy /= 8;
		cz /= 8;

		//e_̍WČvZ
		for(int j=0; j<8; j++){
			//S̈ʒu֌WZo
			x = data.x[j] - cx;
			y = data.y[j] - cy;
			z = data.z[j] - cz;
			rx = atan2(z, x);
			ry = atan2(y, sqrt(x*x + z*z));
			r = sqrt(x*x + y*y + z*z);

			//傫
			r += 0.1f;

			//WČvZ
			cbdata[i].x[j] = cx + cos(rx) * cos(ry) * r;
			cbdata[i].y[j] = cy + sin(ry) * r;
			cbdata[i].z[j] = cz + sin(rx) * cos(ry) * r;
		}
	}

	//pʂ̒S_Zo
	for(int i=0; i<bs; i++){
		for(int j=0; j<6; j++){
			blockdataface(j, &vID[0], NULL);

			cbdata[i].polygon_center_x[j] = 0.0f;
			cbdata[i].polygon_center_y[j] = 0.0f;
			cbdata[i].polygon_center_z[j] = 0.0f;

			//4_̒S_Zo
			for(int k=0; k<4; k++){
				cbdata[i].polygon_center_x[j] += cbdata[i].x[ vID[k] ];
				cbdata[i].polygon_center_y[j] += cbdata[i].y[ vID[k] ];
				cbdata[i].polygon_center_z[j] += cbdata[i].z[ vID[k] ];
			}
			cbdata[i].polygon_center_x[j] /= 4;
			cbdata[i].polygon_center_y[j] /= 4;
			cbdata[i].polygon_center_z[j] /= 4;
		}
	}

	//̃ubNo
	for(int i=0; i<bs; i++){
		cbdata[i].BoardBlock = false;

		for(int j=0; j<8; j++){
			for(int k=j+1; k<8; k++){
				//ꃕł_ȂAɂȂĂƔB
				if( (cbdata[i].x[j] == cbdata[i].x[k])&&(cbdata[i].y[j] == cbdata[i].y[k])&&(cbdata[i].z[j] == cbdata[i].z[k]) ){
					cbdata[i].BoardBlock = true;
					j = 8; k = 8;	//break
				}
			}
		}

		//ubN̒SZo
		float mx = 0.0f;
		float my = 0.0f;
		float mz = 0.0f;
		for(int j=0; j<8; j++){
			mx += cbdata[i].x[j];
			my += cbdata[i].y[j];
			mz += cbdata[i].z[j];
		}
		mx /= 8;
		my /= 8;
		mz /= 8;

		//ubN̒S_ɑ΂1ʂł\ʂȂ΁AɂȂĂƔB
		for(int j=0; j<6; j++){
			if( CheckPolygonFront(i, j, mx, my, mz) == true ){
				cbdata[i].BoardBlock = true;
				break;
			}
		}
	}


	//ubNAABB쐬
	for(int i=0; i<bs; i++){
		GetBlockPosMINMAX(i, &cbdata[i].min_x, &cbdata[i].min_y, &cbdata[i].min_z, &cbdata[i].max_x, &cbdata[i].max_y, &cbdata[i].max_z);
	}

	//ubN̋ԕO[vvZ
	for(int i=0; i<bs; i++){
		cbdata[i].worldgroup = GetWorldGroup(cbdata[i].min_x, cbdata[i].min_z);
		if( GetWorldGroup(cbdata[i].max_x, cbdata[i].max_z) != cbdata[i].worldgroup ){
			cbdata[i].worldgroup = 0;
		}
	}

	return 0;
}

//! @brief ubN̍WőlEŏlԂ
//! @param id 肷ubNԍ
//! @param *min_x ŏ XWԂ|C^
//! @param *min_y ŏ YWԂ|C^
//! @param *min_z ŏ ZWԂ|C^
//! @param *max_x ő XWԂ|C^
//! @param *max_y ő YWԂ|C^
//! @param *max_z ő ZWԂ|C^
void Collision::GetBlockPosMINMAX(int id, float *min_x, float *min_y, float *min_z, float *max_x, float *max_y, float *max_z)
{
	*min_x = cbdata[id].x[0];
	*min_y = cbdata[id].y[0];
	*min_z = cbdata[id].z[0];
	*max_x = cbdata[id].x[0];
	*max_y = cbdata[id].y[0];
	*max_z = cbdata[id].z[0];
	for(int i=1; i<8; i++){
		if( *min_x > cbdata[id].x[i] ){ *min_x = cbdata[id].x[i]; }
		if( *min_y > cbdata[id].y[i] ){ *min_y = cbdata[id].y[i]; }
		if( *min_z > cbdata[id].z[i] ){ *min_z = cbdata[id].z[i]; }

		if( *max_x < cbdata[id].x[i] ){ *max_x = cbdata[id].x[i]; }
		if( *max_y < cbdata[id].y[i] ){ *max_y = cbdata[id].y[i]; }
		if( *max_z < cbdata[id].z[i] ){ *max_z = cbdata[id].z[i]; }
	}

	//vZ덷΍̂߁A킸ɑ傫߂ɂB
	*min_x -= 0.1f;
	*min_y -= 0.1f;
	*min_z -= 0.1f;
	*max_x += 0.1f;
	*max_y += 0.1f;
	*max_z += 0.1f;
}

//! @brief ԕ̃O[vZo
//! @param x XW
//! @param z ZW
//! @return O[vԍ
//! @attention uO[vԍv̊TO͕ʓrhLgQ
int Collision::GetWorldGroup(float x, float z)
{
	if( (x > 0)&&(z > 0) ){ return 1; }
	if( (x < 0)&&(z > 0) ){ return 2; }
	if( (x < 0)&&(z < 0) ){ return 3; }
	if( (x > 0)&&(z < 0) ){ return 4; }
	return 0;
}

//! @brief ̍WAubN̖ʂ̕\ׂ
//! @param id 肷ubNԍ
//! @param face 肷ʔԍ
//! @param x XW
//! @param y YW
//! @param z ZW
//! @return \Ftrue@Ffalse
//! @warning ΍Ww肷Kv܂BxNgł͂܂B
//! @attention e蔻̊֐玩IɌĂ΂܂B
bool Collision::CheckPolygonFront(int id, int face, float x, float y, float z)
{
	if( blockdata == NULL ){ return false; }
	if( (id < 0)||(blockdata->GetTotaldatas() <= id) ){ return false; }
	if( (face < 0)||(6 < face) ){ return false; }

	struct blockdata bdata;
	float vx, vy, vz;
	float d;

	blockdata->Getdata(&bdata, id);

	//ʂ̒S_ɁAW̃xNg߂@mȒS_g
	vx = bdata.material[face].center_x - x;
	vy = bdata.material[face].center_y - y;
	vz = bdata.material[face].center_z - z;

	//
	d = bdata.material[face].vx*vx + bdata.material[face].vy*vy + bdata.material[face].vz*vz;

	if( d <= 0.0f ){
		return true;
	}
	return false;
}

//! @brief ̕ɑ΂āAubN̖ʂ̕\ׂ
//! @param id 肷ubNԍ
//! @param face 肷ʔԍ
//! @param rx ̊px
//! @return \Ftrue@Ffalse
bool Collision::CheckPolygonFrontRx(int id, int face, float rx)
{
	if( blockdata == NULL ){ return false; }
	if( (id < 0)||(blockdata->GetTotaldatas() <= id) ){ return false; }
	if( (face < 0)||(6 < face) ){ return false; }

	struct blockdata bdata;
	float vx, vz;
	float d;

	blockdata->Getdata(&bdata, id);

	//xNgZo
	vx = cos(rx);
	vz = sin(rx);

	//
	d = bdata.material[face].vx*vx + bdata.material[face].vz*vz;

	if( d <= 0.0f ){
		return true;
	}
	return false;
}

//! @brief ubN̖ʂƃCiĵ蔻
//! @param blockid 肷ubNԍ
//! @param face 肷ʔԍ
//! @param RayPos_x C̈ʒuin_jw肷 XW
//! @param RayPos_y C̈ʒuin_jw肷 YW
//! @param RayPos_z C̈ʒuin_jw肷 ZW
//! @param RayDir_x C̃xNgw肷 X
//! @param RayDir_y C̃xNgw肷 Y
//! @param RayDir_z C̃xNgw肷 Z
//! @param out_Dist ubNƂ̋󂯎|C^
//! @return ĂFtrue@ĂȂFfalse
//! @warning RayPosin_j RayDirixNgjԈႦȂƁB
//! @attention e蔻̊֐玩IɌĂ΂܂B
bool Collision::CheckIntersectTri(int blockid, int face, float RayPos_x, float RayPos_y, float RayPos_z, float RayDir_x, float RayDir_y, float RayDir_z, float *out_Dist)
{
	struct blockdata data;
	int vID[4];
	float d1, d2;
	float vx1, vy1, vz1;
	float dist;
	float x, y, z;
	float vx2, vy2, vz2;

	blockdata->Getdata(&data, blockid);
	blockdataface(face, vID, NULL);

	//
	d1 = data.material[face].vx*RayDir_x + data.material[face].vy*RayDir_y + data.material[face].vz*RayDir_z;

	if( d1 >= 0.0f ){
		return false;		//ʂƃCsAʂɑ΂ăCt
	}

	//ʂ̒S_ɁAW̃xNg߂@mȒS_g
	vx1 = RayPos_x - data.material[face].center_x;
	vy1 = RayPos_y - data.material[face].center_y;
	vz1 = RayPos_z - data.material[face].center_z;

	//
	d2 = data.material[face].vx*vx1 + data.material[face].vy*vy1 + data.material[face].vz*vz1;		//ʂ܂ł̍ŒZ܂

	//_܂ł̋ƍW
	dist = 1.0f / (d1*-1) * d2;
	x = RayPos_x + RayDir_x * dist;
	y = RayPos_y + RayDir_y * dist;
	z = RayPos_z + RayDir_z * dist;


	//ubNŜAABBɓȂΏO
	if( (x < cbdata[blockid].min_x)||(cbdata[blockid].max_x < x) ){ return false; }
	if( (y < cbdata[blockid].min_y)||(cbdata[blockid].max_y < y) ){ return false; }
	if( (z < cbdata[blockid].min_z)||(cbdata[blockid].max_z < z) ){ return false; }


	//ȉAubNʂ̓Ɍ_邩`FbN
	//@@ʂ`e4ӂƂ̈ʒu֌WZoAʂ̖@ƔrB

	//O
	vx2 = ((data.y[ vID[1] ] - data.y[ vID[0] ]) * (z - data.z[ vID[0] ])) - ((y - data.y[ vID[0] ]) * (data.z[ vID[1] ] - data.z[ vID[0] ]));
	vy2 = ((data.z[ vID[1] ] - data.z[ vID[0] ]) * (x - data.x[ vID[0] ])) - ((z - data.z[ vID[0] ]) * (data.x[ vID[1] ] - data.x[ vID[0] ]));
	vz2 = ((data.x[ vID[1] ] - data.x[ vID[0] ]) * (y - data.y[ vID[0] ])) - ((x - data.x[ vID[0] ]) * (data.y[ vID[1] ] - data.y[ vID[0] ]));

	//
	d1 = data.material[face].vx*vx2 + data.material[face].vy*vy2 + data.material[face].vz*vz2;		//ubNʂ̖@Ƃ̊֌WZo

	if( d1 < 0.0f ){	//OɂΏO
		return false;
	}


	//O
	vx2 = ((data.y[ vID[2] ] - data.y[ vID[1] ]) * (z - data.z[ vID[1] ])) - ((y - data.y[ vID[1] ]) * (data.z[ vID[2] ] - data.z[ vID[1] ]));
	vy2 = ((data.z[ vID[2] ] - data.z[ vID[1] ]) * (x - data.x[ vID[1] ])) - ((z - data.z[ vID[1] ]) * (data.x[ vID[2] ] - data.x[ vID[1] ]));
	vz2 = ((data.x[ vID[2] ] - data.x[ vID[1] ]) * (y - data.y[ vID[1] ])) - ((x - data.x[ vID[1] ]) * (data.y[ vID[2] ] - data.y[ vID[1] ]));

	//
	d1 = data.material[face].vx*vx2 + data.material[face].vy*vy2 + data.material[face].vz*vz2;		//ubNʂ̖@Ƃ̊֌WZo

	if( d1 < 0.0f ){	//OɂΏO
		return false;
	}


	//O
	vx2 = ((data.y[ vID[3] ] - data.y[ vID[2] ]) * (z - data.z[ vID[2] ])) - ((y - data.y[ vID[2] ]) * (data.z[ vID[3] ] - data.z[ vID[2] ]));
	vy2 = ((data.z[ vID[3] ] - data.z[ vID[2] ]) * (x - data.x[ vID[2] ])) - ((z - data.z[ vID[2] ]) * (data.x[ vID[3] ] - data.x[ vID[2] ]));
	vz2 = ((data.x[ vID[3] ] - data.x[ vID[2] ]) * (y - data.y[ vID[2] ])) - ((x - data.x[ vID[2] ]) * (data.y[ vID[3] ] - data.y[ vID[2] ]));

	//
	d1 = data.material[face].vx*vx2 + data.material[face].vy*vy2 + data.material[face].vz*vz2;		//ubNʂ̖@Ƃ̊֌WZo

	if( d1 < 0.0f ){	//OɂΏO
		return false;
	}


	//O
	vx2 = ((data.y[ vID[0] ] - data.y[ vID[3] ]) * (z - data.z[ vID[3] ])) - ((y - data.y[ vID[3] ]) * (data.z[ vID[0] ] - data.z[ vID[3] ]));
	vy2 = ((data.z[ vID[0] ] - data.z[ vID[3] ]) * (x - data.x[ vID[3] ])) - ((z - data.z[ vID[3] ]) * (data.x[ vID[0] ] - data.x[ vID[3] ]));
	vz2 = ((data.x[ vID[0] ] - data.x[ vID[3] ]) * (y - data.y[ vID[3] ])) - ((x - data.x[ vID[3] ]) * (data.y[ vID[0] ] - data.y[ vID[3] ]));

	//
	d1 = data.material[face].vx*vx2 + data.material[face].vy*vy2 + data.material[face].vz*vz2;		//ubNʂ̖@Ƃ̊֌WZo

	if( d1 < 0.0f ){	//OɂΏO
		return false;
	}


	*out_Dist = dist;

	return true;
}

//! @brief ubNɖ܂ĂȂׂ
//! @param blockid 肷ubNԍ
//! @param x XW
//! @param y YW
//! @param z ZW
//! @param worldgroup Ԃ̃O[v𗘗pČvZȗ݂itrueFLEvZȗ@falseFESj
//! @param *planeid \ɂʔԍiNULLj
//! @return ĂFtrue@ĂȂFfalse
//! @warning *planeid Ԃ\ʁi0`5j́A̖ʂYꍇłAŏɌ1ʂ̂ݕԂ܂B
bool Collision::CheckBlockInside(int blockid, float x, float y, float z, bool worldgroup, int *planeid)
{
	if( blockdata == NULL ){ return false; }
	if( (blockid < 0)||(blockdata->GetTotaldatas() <= blockid) ){ return false; }

	//̃ubN͌vZO
	if( cbdata[blockid].BoardBlock == true ){ return false; }


	//̍r
	if( worldgroup == true ){

		//ԕ
		if( cbdata[blockid].worldgroup != 0 ){
			//ϑ_̋Ԃ̃O[v擾
			int worldgroup = GetWorldGroup(x, z);

			if( worldgroup != 0 ){
				//Ԃ̃O[vႦΌvZO
				if( cbdata[blockid].worldgroup != worldgroup ){
					return false;
				}
			}
		}

		//͈͂Ō
		if( (x < cbdata[blockid].min_x)||(cbdata[blockid].max_x < x) ){ return false; }
		if( (y < cbdata[blockid].min_y)||(cbdata[blockid].max_y < y) ){ return false; }
		if( (z < cbdata[blockid].min_z)||(cbdata[blockid].max_z < z) ){ return false; }
	}


	//6ʂ猩đSėʂǂ
	for(int i=0; i<6; i++){
		if( CheckPolygonFront(blockid, i, x, y, z) == true ){
			if( planeid != NULL ){ *planeid = i; }
			return false;	//\ʂȂΏI
		}
	}

	return true;
}

//! @brief SẴubNɖ܂ĂȂׂ
//! @param x XW
//! @param y YW
//! @param z ZW
//! @return ĂFtrue@ĂȂFfalse
bool Collision::CheckALLBlockInside(float x, float y, float z)
{
	if( blockdata == NULL ){ return false; }

	int bs = blockdata->GetTotaldatas();

	for(int i=0; i<bs; i++){
		if( CheckBlockInside(i, x, y, z, true, NULL) == true ){ return true; }
	}

	return false;
}

//! @brief ubNƃCiĵ蔻
//! @param blockid 肷ubNԍ
//! @param RayPos_x C̈ʒuin_jw肷 XW
//! @param RayPos_y C̈ʒuin_jw肷 YW
//! @param RayPos_z C̈ʒuin_jw肷 ZW
//! @param RayDir_x C̃xNgw肷 X
//! @param RayDir_y C̃xNgw肷 Y
//! @param RayDir_z C̃xNgw肷 Z
//! @param face ubN̖ʔԍi0`5j󂯎|C^@iNULLj
//! @param Dist ubNƂ̋󂯎|C^
//! @param maxDist ső勗@i0.0 ŖEj
//! @return ĂFtrue@ĂȂFfalse
//! @warning RayPosin_j RayDirixNgjԈႦȂƁB
//! @warning ső勗w肵ȂƁAptH[}X啝ɒቺ܂B
//! @attention C̎n_痠ɂȂubN̖ʂ͖܂B[ɕό`̃ubN܂B
//! @attention ܂AC̃ubNɓꍇ́AC̎n_ԋ߂Ԃ܂B
bool Collision::CheckBlockIntersectRay(int blockid, float RayPos_x, float RayPos_y, float RayPos_z, float RayDir_x, float RayDir_y, float RayDir_z, int *face, float *Dist, float maxDist)
{
	if( blockdata == NULL ){ return false; }
	if( (blockid < 0)||(blockdata->GetTotaldatas() <= blockid) ){ return false; }
	if( (RayDir_x == 0.0f)&&(RayDir_y == 0.0f)&&(RayDir_z == 0.0f) ){ return false; }

	float pDist;
	float min_pDist = FLT_MAX;
	int min_blockface = -1;
	float rmin_x = 0.0f;
	float rmin_y = 0.0f;
	float rmin_z = 0.0f;
	float rmax_x = 0.0f;
	float rmax_y = 0.0f;
	float rmax_z = 0.0f;
	int worldgroupA = 0;
	int worldgroupB = 0;

	//̃ubN͌vZO
	if( cbdata[blockid].BoardBlock == true ){
		if( face != NULL ){ *face = 0; }
		*Dist = 0.0f;
		return false;
	}

	if( maxDist > 0.0f ){
		//CAABB
		rmin_x = RayPos_x + RayDir_x * maxDist;
		rmin_y = RayPos_y + RayDir_y * maxDist;
		rmin_z = RayPos_z + RayDir_z * maxDist;
		rmax_x = rmin_x;
		rmax_y = rmin_y;
		rmax_z = rmin_z;
		if( rmin_x > RayPos_x ){ rmin_x = RayPos_x; }
		if( rmin_y > RayPos_y ){ rmin_y = RayPos_y; }
		if( rmin_z > RayPos_z ){ rmin_z = RayPos_z; }
		if( rmax_x < RayPos_x ){ rmax_x = RayPos_x; }
		if( rmax_y < RayPos_y ){ rmax_y = RayPos_y; }
		if( rmax_z < RayPos_z ){ rmax_z = RayPos_z; }
	}

	if( maxDist > 0.0f ){
		//n_ƏI_̋ԃO[v擾
		worldgroupA = GetWorldGroup(RayPos_x, RayPos_z);
		worldgroupB = GetWorldGroup(RayPos_x + RayDir_x * maxDist, RayPos_z + RayDir_z * maxDist);
	}

	if( maxDist > 0.0f ){
		if( cbdata[blockid].worldgroup != 0 ){
			//n_ƏI_AԂ̃O[voĂȂ
			if( (worldgroupA != 0)&&(worldgroupA == worldgroupB) ){

				//Ԃ̃O[vႦΌvZO
				if( cbdata[blockid].worldgroup == worldgroupA ){		//worldgroupA == worldgroupB
					//E{bNXmŔ
					if( CollideBoxAABB(cbdata[blockid].min_x, cbdata[blockid].min_y, cbdata[blockid].min_z, cbdata[blockid].max_x, cbdata[blockid].max_y, cbdata[blockid].max_z, rmin_x, rmin_y, rmin_z, rmax_x, rmax_y, rmax_z) == false ){
						//ĂȂ΁ÃubNׂ͒ȂB
						if( face != NULL ){ *face = 0; }
						*Dist = 0.0f;
						return false;
					}
				}
			}
		}
	}

	//E{bNXƃCŔ
	if( CollideAABBRay(cbdata[blockid].min_x, cbdata[blockid].min_y, cbdata[blockid].min_z, cbdata[blockid].max_x, cbdata[blockid].max_y, cbdata[blockid].max_z, RayPos_x, RayPos_y, RayPos_z, RayDir_x, RayDir_y, RayDir_z, NULL, -1.0f) == false ){
		//ĂȂ΁ÃubNׂ͒ȂB
		if( face != NULL ){ *face = 0; }
		*Dist = 0.0f;
		return false;
	}

	//e|SPʂŔ
	for(int i=0; i<6; i++){
		if( CheckPolygonFront(blockid, i, RayPos_x, RayPos_y, RayPos_z) == true ){
			if( CheckIntersectTri(blockid, i, RayPos_x, RayPos_y, RayPos_z, RayDir_x, RayDir_y, RayDir_z, &pDist) == true ){
				if( min_pDist > pDist ){
					min_pDist = pDist;
					min_blockface = i;
				}
			}
		}
	}

	//Ȃ΁A|C^ɓKȐĕԂB
	if( min_blockface == -1 ){
		if( face != NULL ){ *face = 0; }
		*Dist = 0.0f;
		return false;
	}

	//vZʂĕԂ
	if( face != NULL ){ *face = min_blockface; }
	*Dist = min_pDist;
	return true;
}

//! @brief SẴubNƃCiĵ蔻
//! @param RayPos_x C̈ʒuin_jw肷 XW
//! @param RayPos_y C̈ʒuin_jw肷 YW
//! @param RayPos_z C̈ʒuin_jw肷 ZW
//! @param RayDir_x C̃xNgw肷 X
//! @param RayDir_y C̃xNgw肷 Y
//! @param RayDir_z C̃xNgw肷 Z
//! @param id ubNID󂯎|C^@iNULLj
//! @param face ubN̖ʔԍi0`5j󂯎|C^@iNULLj
//! @param Dist ubNƂ̋󂯎|C^
//! @param maxDist ső勗@i0.0 ŖEj
//! @return ĂFtrue@ĂȂFfalse
//! @warning RayPosin_j RayDirixNgjԈႦȂƁB
//! @warning ső勗w肵ȂƁAptH[}X啝ɒቺ܂B
//! @warning g CheckALLBlockIntersectDummyRay()֐ ƗގĂ܂A֐荂xŒᑬłB
//! @attention C̎n_痠ɂȂubN̖ʂ͖܂B[ɕό`̃ubN܂B
//! @attention ܂AC̃ubNɓꍇ́A<b>C̎n_ԋ߂</b>Ԃ܂B
bool Collision::CheckALLBlockIntersectRay(float RayPos_x, float RayPos_y, float RayPos_z, float RayDir_x, float RayDir_y, float RayDir_z, int *id, int *face, float *Dist, float maxDist)
{
	if( blockdata == NULL ){ return false; }
	if( (RayDir_x == 0.0f)&&(RayDir_y == 0.0f)&&(RayDir_z == 0.0f) ){ return false; }

	int bs = blockdata->GetTotaldatas();
	float pDist;
	float min_pDist = FLT_MAX;
	int min_blockid = -1;
	int min_blockface = -1;
	float rmin_x = 0.0f;
	float rmin_y = 0.0f;
	float rmin_z = 0.0f;
	float rmax_x = 0.0f;
	float rmax_y = 0.0f;
	float rmax_z = 0.0f;
	int worldgroupA = 0;
	int worldgroupB = 0;

	if( maxDist > 0.0f ){
		//CAABB
		rmin_x = RayPos_x + RayDir_x * maxDist;
		rmin_y = RayPos_y + RayDir_y * maxDist;
		rmin_z = RayPos_z + RayDir_z * maxDist;
		rmax_x = rmin_x;
		rmax_y = rmin_y;
		rmax_z = rmin_z;
		if( rmin_x > RayPos_x ){ rmin_x = RayPos_x; }
		if( rmin_y > RayPos_y ){ rmin_y = RayPos_y; }
		if( rmin_z > RayPos_z ){ rmin_z = RayPos_z; }
		if( rmax_x < RayPos_x ){ rmax_x = RayPos_x; }
		if( rmax_y < RayPos_y ){ rmax_y = RayPos_y; }
		if( rmax_z < RayPos_z ){ rmax_z = RayPos_z; }
	}

	if( maxDist > 0.0f ){
		//n_ƏI_̋ԃO[v擾
		worldgroupA = GetWorldGroup(RayPos_x, RayPos_z);
		worldgroupB = GetWorldGroup(RayPos_x + RayDir_x * maxDist, RayPos_z + RayDir_z * maxDist);
	}
	
	for(int i=0; i<bs; i++){
		//̃ubN͌vZO
		if( cbdata[i].BoardBlock == true ){ continue; }

		if( maxDist > 0.0f ){
			if( cbdata[i].worldgroup != 0 ){
				//n_ƏI_AԂ̃O[voĂȂ
				if( (worldgroupA != 0)&&(worldgroupA == worldgroupB) ){

					//Ԃ̃O[vႦΌvZO
					if( cbdata[i].worldgroup == worldgroupA ){		//worldgroupA == worldgroupB

						//E{bNXmŔ
						if( CollideBoxAABB(cbdata[i].min_x, cbdata[i].min_y, cbdata[i].min_z, cbdata[i].max_x, cbdata[i].max_y, cbdata[i].max_z, rmin_x, rmin_y, rmin_z, rmax_x, rmax_y, rmax_z) == false ){
							continue;	//ĂȂ΁ÃubNׂ͒ȂB
						}

					}
				}
			}
		}

		//E{bNXƃCŔ
		if( CollideAABBRay(cbdata[i].min_x, cbdata[i].min_y, cbdata[i].min_z, cbdata[i].max_x, cbdata[i].max_y, cbdata[i].max_z, RayPos_x, RayPos_y, RayPos_z, RayDir_x, RayDir_y, RayDir_z, NULL, -1.0f) == false ){
			continue;		//ĂȂ΁ÃubNׂ͒ȂB
		}

		//e|SPʂŔ
		for(int j=0; j<6; j++){
			if( CheckPolygonFront(i, j, RayPos_x, RayPos_y, RayPos_z) == true ){
				if( CheckIntersectTri(i, j, RayPos_x, RayPos_y, RayPos_z, RayDir_x, RayDir_y, RayDir_z, &pDist) == true ){
					if( min_pDist > pDist ){
						min_pDist = pDist;
						min_blockid = i;
						min_blockface = j;
					}
				}
			}
		}
	}

	//ő勗𒴂Ă΁A𖳌ɁB
	if( maxDist >= 0.0f ){
		if( min_pDist > maxDist ){
			min_blockid = -1;
		}
	}

	//Ȃ΁A|C^ɓKȐĕԂB
	if( min_blockid == -1 ){
		if( id != NULL ){ *id = 0; }
		if( face != NULL ){ *face = 0; }
		*Dist = 0.0f;
		return false;
	}

	//vZʂĕԂ
	if( id != NULL ){ *id = min_blockid; }
	if( face != NULL ){ *face = min_blockface; }
	*Dist = min_pDist;
	return true;
}

//! @brief SẴubNƏՓ˂Ă邩
//! @param RayPos_x n_ XW
//! @param RayPos_y n_ YW
//! @param RayPos_z n_ ZW
//! @param RayDir_x xNg X
//! @param RayDir_y xNg Y
//! @param RayDir_z xNg Z
//! @param id ubNID󂯎|C^@iNULLj
//! @param face ubN̖ʔԍi0`5j󂯎|C^@iNULLj
//! @param Dist _~[ϐ̃|C^i 0.0f Ԃj
//! @param maxDist ső勗@i0.0 w sj
//! @return ĂFtrue@ĂȂFfalse
//! @warning n_ xNgԈႦȂƁB
//! @warning g CheckALLBlockIntersectRay()֐ ƗގĂ܂A֐荂ŒᐸxłB
//! @attention C̎n_痠ɂȂubN̖ʂ͖܂B[ɕό`̃ubN܂B
//! @attention ܂AC̃ubNɓꍇ́A<b>ԍŏɔ</b>Ԃ܂B
bool Collision::CheckALLBlockIntersectDummyRay(float RayPos_x, float RayPos_y, float RayPos_z, float RayDir_x, float RayDir_y, float RayDir_z, int *id, int *face, float *Dist, float maxDist)
{
	if( blockdata == NULL ){ return false; }
	if( maxDist <= 0.0f ){ return false; }

	float RayPos2_x = RayPos_x + RayDir_x * maxDist/2;
	float RayPos2_y = RayPos_y + RayDir_y * maxDist/2;
	float RayPos2_z = RayPos_z + RayDir_z * maxDist/2;
	float RayPos3_x = RayPos_x + RayDir_x * maxDist;
	float RayPos3_y = RayPos_y + RayDir_y * maxDist;
	float RayPos3_z = RayPos_z + RayDir_z * maxDist;
	
	if( (id == NULL)&&(face == NULL) ){
		for(int i=0; i<MAX_BLOCKS; i++){
			//I_
			if( CheckBlockInside(i, RayPos3_x, RayPos3_y, RayPos3_z, true, NULL) == true ){
				*Dist = 0.0f;
				return true;
			}

			//Ԏ_
			if( CheckBlockInside(i, RayPos2_x, RayPos2_y, RayPos2_z, true, NULL) == true ){
				*Dist = 0.0f;
				return true;
			}
		}

		*Dist = 0.0f;
		return false;
	}

	for(int i=0; i<MAX_BLOCKS; i++){
		int surface;

		//Jnn_
		CheckBlockInside(i, RayPos_x, RayPos_y, RayPos_z, false, &surface);

		//I_
		if( CheckBlockInside(i, RayPos3_x, RayPos3_y, RayPos3_z, true, &surface) == true ){
			if( id != NULL ){ *id = i; }
			if( face != NULL ){ *face = surface; }
			*Dist = 0.0f;
			return true;
		}

		//Ԏ_
		if( CheckBlockInside(i, RayPos2_x, RayPos2_y, RayPos2_z, true, &surface) == true ){
			if( id != NULL ){ *id = i; }
			if( face != NULL ){ *face = surface; }
			*Dist = 0.0f;
			return true;
		}
	}

	if( id != NULL ){ *id = 0; }
	if( face != NULL ){ *face = 0; }
	*Dist = 0.0f;
	return false;
}

//! @brief ubNɉĈړxNg߂
void Collision::ScratchVector(int id, int face, float in_vx, float in_vy, float in_vz, float *out_vx, float *out_vy, float *out_vz)
{
	if( blockdata == NULL ){ return; }
	if( (id < 0)||(blockdata->GetTotaldatas() <= id) ){ return; }
	if( (face < 0)||(6 < face) ){ return; }

	struct blockdata bdata;
	blockdata->Getdata(&bdata, id);

	//
	float Dot = in_vx * bdata.material[face].vx + in_vy * bdata.material[face].vy + in_vz * bdata.material[face].vz;

	*out_vx = in_vx - Dot * bdata.material[face].vx;
	*out_vy = in_vy - Dot * bdata.material[face].vy;
	*out_vz = in_vz - Dot * bdata.material[face].vz;
}

//! @brief ubNɔ˂xNg߂
void Collision::ReflectVector(int id, int face, float in_vx, float in_vy, float in_vz, float *out_vx, float *out_vy, float *out_vz)
{
	if( blockdata == NULL ){ return; }
	if( (id < 0)||(blockdata->GetTotaldatas() <= id) ){ return; }
	if( (face < 0)||(6 < face) ){ return; }

	struct blockdata bdata;
	blockdata->Getdata(&bdata, id);

	//
	float Dot = in_vx * bdata.material[face].vx + in_vy * bdata.material[face].vy + in_vz * bdata.material[face].vz;

	*out_vx = in_vx - 2.0f * Dot * bdata.material[face].vx;
	*out_vy = in_vy - 2.0f * Dot * bdata.material[face].vy;
	*out_vz = in_vz - 2.0f * Dot * bdata.material[face].vz;
}


//! @brief AABBɂ邠蔻
//! @param box1_min_x A̍ŏ XW
//! @param box1_min_y A̍ŏ YW
//! @param box1_min_z A̍ŏ ZW
//! @param box1_max_x A̍ő XW
//! @param box1_max_y A̍ő YW
//! @param box1_max_z A̍ő ZW
//! @param box2_min_x B̍ŏ XW
//! @param box2_min_y B̍ŏ YW
//! @param box2_min_z B̍ŏ ZW
//! @param box2_max_x B̍ő XW
//! @param box2_max_y B̍ő YW
//! @param box2_max_z B̍ő ZW
//! @return ĂFtrue@ĂȂFfalse
//! @attention G[ꍇuĂȂFfalsevԂ܂B
bool CollideBoxAABB(float box1_min_x, float box1_min_y, float box1_min_z, float box1_max_x, float box1_max_y, float box1_max_z, float box2_min_x, float box2_min_y, float box2_min_z, float box2_max_x, float box2_max_y, float box2_max_z)
{
	//G[΍
	if( box1_min_x > box1_max_x ){ return false; }
	if( box1_min_y > box1_max_y ){ return false; }
	if( box1_min_z > box1_max_z ){ return false; }
	if( box2_min_x > box2_max_x ){ return false; }
	if( box2_min_y > box2_max_y ){ return false; }
	if( box2_min_z > box2_max_z ){ return false; }

	if(
		(box1_min_x < box2_max_x)&&(box1_max_x > box2_min_x)&&
		(box1_min_y < box2_max_y)&&(box1_max_y > box2_min_y)&&
		(box1_min_z < box2_max_z)&&(box1_max_z > box2_min_z)
	){
		return true;
	}

	return false;
}

//! @brief ~m̓蔻
//! @param c1_x ~1 ӂxW
//! @param c1_y ~1 ӂyW
//! @param c1_z ~1 ӂzW
//! @param c1_r ~1 ̔a
//! @param c1_h ~1 ̍
//! @param c2_x ~2 ӂxW
//! @param c2_y ~2 ӂyW
//! @param c2_z ~2 ӂzW
//! @param c2_r ~2 ̔a
//! @param c2_h ~2 ̍
//! @param *angle ~1݂ڐGpx󂯎|C^iNULLj
//! @param *length ~1݂ڐG󂯎|C^iNULLj
//! @return ĂFtrue@ĂȂFfalse
bool CollideCylinder(float c1_x, float c1_y, float c1_z, float c1_r, float c1_h, float c2_x, float c2_y, float c2_z, float c2_r, float c2_h, float *angle, float *length)
{
	//ɂyȍŔ
	if( (c1_y < c2_y + c2_h)&&(c1_y + c1_h > c2_y) ){

		//Ŕ
		float x = c1_x - c2_x;
		float z = c1_z - c2_z;
		float caser = x*x + z*z;
		float minr = (c1_r+c2_r) * (c1_r+c2_r);
		if( caser < minr ){
			if( angle != NULL ){ *angle = atan2(z, x); }
			if( length != NULL ){ *length = sqrt(minr) - sqrt(caser); }
			return true;
		}
	}

	return false;
}

//! @brief ̂ƃCiĵ蔻
//! @param s_x ̂xW
//! @param s_y ̂yW
//! @param s_z ̂zW
//! @param s_r ̂̔a
//! @param RayPos_x C̈ʒuin_jw肷 XW
//! @param RayPos_y C̈ʒuin_jw肷 YW
//! @param RayPos_z C̈ʒuin_jw肷 ZW
//! @param RayDir_x C̃xNgw肷 X
//! @param RayDir_y C̃xNgw肷 Y
//! @param RayDir_z C̃xNgw肷 Z
//! @param Dist ̂Ƃ̋󂯎|C^
//! @param maxDist ső勗@i0.0 ŖEj
//! @return ĂFtrue@ĂȂFfalse
//! @warning RayPosin_j RayDirixNgjԈႦȂƁB
//! @warning ső勗w肵ȂƁAptH[}X啝ɒቺ܂B
//! @todo C̎n_̂̊Oł̂Ƌt̏ꍇAłȂH
bool CollideSphereRay(float s_x, float s_y, float s_z, float s_r, float RayPos_x, float RayPos_y, float RayPos_z, float RayDir_x, float RayDir_y, float RayDir_z, float *Dist, float maxDist)
{
	if( maxDist > 0.0f ){
		float pmin_x, pmin_y, pmin_z, pmax_x, pmax_y, pmax_z;

		//CAABB
		pmin_x = RayPos_x + RayDir_x * maxDist;
		pmin_y = RayPos_y + RayDir_y * maxDist;
		pmin_z = RayPos_z + RayDir_z * maxDist;
		pmax_x = pmin_x;
		pmax_y = pmin_y;
		pmax_z = pmin_z;
		if( pmin_x > RayPos_x ){ pmin_x = RayPos_x; }
		if( pmin_y > RayPos_y ){ pmin_y = RayPos_y; }
		if( pmin_z > RayPos_z ){ pmin_z = RayPos_z; }
		if( pmax_x < RayPos_x ){ pmax_x = RayPos_x; }
		if( pmax_y < RayPos_y ){ pmax_y = RayPos_y; }
		if( pmax_z < RayPos_z ){ pmax_z = RayPos_z; }

		//E{bNXmŔ
		if( CollideBoxAABB(s_x - s_r, s_y - s_r, s_z - s_r, s_x + s_r, s_y + s_r, s_z + s_r, pmin_x, pmin_y, pmin_z, pmax_x, pmax_y, pmax_z) == false ){
			return false;
		}
	}

	float x, y, z, d;
	float MinDist, RayDist, RDist;

	//_ƃCn_̋
	x = s_x - RayPos_x;
	y = s_y - RayPos_y;
	z = s_z - RayPos_z;
	d = sqrt(x*x + y*y + z*z);

	//Cn_a߂i߂荞łj
	if( d < s_r ){
		*Dist = 0.0f;
		return true;
	}

	//_î̒SjƃC̍ŒZ߂
	MinDist = DistancePosRay(s_x, s_y, s_z, RayPos_x, RayPos_y, RayPos_z, RayDir_x, RayDir_y, RayDir_z);

	if( MinDist <= s_r ){
		RayDist = sqrt(d*d - MinDist*MinDist);		//iCn_j_ɍł߂Â
		RDist = sqrt(s_r*s_r - MinDist*MinDist);	//i_aj_ɍł߂Â

		*Dist = RayDist - RDist;	//C_ŒZ - aŒZ = C_甼a܂ł̍ŒZ

		//ső勗΁AĂȂƂ
		if( (maxDist > 0.0f)&&(maxDist < *Dist) ){
			return false;
		}
		return true;
	}

	return false;
}

//! @brief AABBƃCiĵ蔻
//! @param box_min_x ̂̍ŏ XW
//! @param box_min_y ̂̍ŏ YW
//! @param box_min_z ̂̍ŏ ZW
//! @param box_max_x ̂̍ő XW
//! @param box_max_y ̂̍ő YW
//! @param box_max_z ̂̍ő ZW
//! @param RayPos_x C̈ʒuin_jw肷 XW
//! @param RayPos_y C̈ʒuin_jw肷 YW
//! @param RayPos_z C̈ʒuin_jw肷 ZW
//! @param RayDir_x C̃xNgw肷 X
//! @param RayDir_y C̃xNgw肷 Y
//! @param RayDir_z C̃xNgw肷 Z
//! @param Dist AABBƂ̋󂯎|C^
//! @param maxDist ső勗@i0.0 ŖEj
//! @return ĂFtrue@ĂȂFfalse
//! @warning RayPosin_j RayDirixNgjԈႦȂƁB
//! @warning ső勗w肵ȂƁAptH[}X啝ɒቺ܂B
bool CollideAABBRay(float box_min_x, float box_min_y, float box_min_z, float box_max_x, float box_max_y, float box_max_z, float RayPos_x, float RayPos_y, float RayPos_z, float RayDir_x, float RayDir_y, float RayDir_z, float *Dist, float maxDist)
{
	if( box_min_x > box_max_x ){ return false; }
	if( box_min_y > box_max_y ){ return false; }
	if( box_min_z > box_max_z ){ return false; }

	//C̎n_AABB̓ɓĂ΁AɓĂ
	if( (box_min_x <= RayPos_x)&&(RayPos_x <= box_max_x)&&(box_min_y <= RayPos_y)&&(RayPos_y <= box_max_y)&&(box_min_z <= RayPos_z)&&(RayPos_z <= box_max_z) ){
		if( Dist != NULL ){ *Dist = 0.0f; }
		return true;
	}

	if( maxDist > 0.0f ){
		float pmin_x, pmin_y, pmin_z, pmax_x, pmax_y, pmax_z;

		//CAABB
		pmin_x = RayPos_x + RayDir_x * maxDist;
		pmin_y = RayPos_y + RayDir_y * maxDist;
		pmin_z = RayPos_z + RayDir_z * maxDist;
		pmax_x = pmin_x;
		pmax_y = pmin_y;
		pmax_z = pmin_z;
		if( pmin_x > RayPos_x ){ pmin_x = RayPos_x; }
		if( pmin_y > RayPos_y ){ pmin_y = RayPos_y; }
		if( pmin_z > RayPos_z ){ pmin_z = RayPos_z; }
		if( pmax_x < RayPos_x ){ pmax_x = RayPos_x; }
		if( pmax_y < RayPos_y ){ pmax_y = RayPos_y; }
		if( pmax_z < RayPos_z ){ pmax_z = RayPos_z; }

		//E{bNXmŔ
		if( CollideBoxAABB(box_min_x, box_min_y, box_min_z, box_max_x, box_max_y, box_max_z, pmin_x, pmin_y, pmin_z, pmax_x, pmax_y, pmax_z) == false ){
			return false;
		}
	}

	//xNg𐳋K
	float r = sqrt(RayDir_x*RayDir_x + RayDir_y*RayDir_y +  RayDir_z*RayDir_z);
	if( r > 1.0f ){
		RayDir_x /= r;
		RayDir_y /= r;
		RayDir_z /= r;
	}

	//eWzɊi[
	float box_min[] = {box_min_x, box_min_y, box_min_z};
	float box_max[] = {box_max_x, box_max_y, box_max_z};
	float RayPos[] = {RayPos_x, RayPos_y, RayPos_z};
	float RayDir[] = {RayDir_x, RayDir_y, RayDir_z};
	float Ray_min[3];
	float Ray_max[3];
	float Ray_tmin, Ray_tmax;

	//XEYEZ3̏
	for(int axis=0; axis<3; axis++){
		if( ( (RayPos[axis] < box_min[axis])||(box_max[axis] < RayPos[axis]) )&&(RayDir[axis] == 0.0f) ){
			//ɓĂȂ̂ xNg̕ 0 ȂAAABB̊O
			return false;
		}
		else{
			//ɓĂȂAxNgȂ`
			float t1, t2;

			//AABBxNg˂_܂ł̋擾
			t1 = (box_min[axis] - RayPos[axis]) / RayDir[axis];
			t2 = (box_max[axis] - RayPos[axis]) / RayDir[axis];

			//_܂ł̋ŏEőŋtȂAւB
			if( t1 > t2 ){
				float temp = t1; t1 = t2; t2 = temp;
			}

			//̏ƂċL^
			Ray_min[axis] = t1;
			Ray_max[axis] = t2;

			if( (box_min[axis] <= RayPos[axis])&&(RayPos[axis] <= box_max[axis]) ){
				//ɓĂ΁Â܂܎̏ƂċL^
				Ray_min[axis] = 0.0f;
			}

			//}CiXixNgtjȂAABBƂ Ȃ
			if( Ray_min[axis] < 0.0f ){ return false; }
			if( Ray_max[axis] < 0.0f ){ return false; }
		}
	}

	//eŁAłeŏfƍł߂eő勗fZo
	Ray_tmin = Ray_min[0];
	Ray_tmax = Ray_max[0];
	if( Ray_tmin < Ray_min[1] ){ Ray_tmin = Ray_min[1]; }
	if( Ray_tmax > Ray_max[1] ){ Ray_tmax = Ray_max[1]; }
	if( Ray_tmin < Ray_min[2] ){ Ray_tmin = Ray_min[2]; }
	if( Ray_tmax > Ray_max[2] ){ Ray_tmax = Ray_max[2]; }

	//ŏƍő勗̊֌W΁`
	if( (Ray_tmax - Ray_tmin) > 0 ){
		//ső勗艓΁A薳B
		if( maxDist > 0.0f ){
			if( Ray_tmin > maxDist ){ return false; }
		}

		//Ԃ
		if( Dist != NULL ){ *Dist = Ray_tmin; }
		return true;
	}

	return false;
}

//! @brief ~ƃCiĵ蔻
//! @param c_x ~ ӂxW
//! @param c_y ~ ӂyW
//! @param c_z ~ ӂzW
//! @param c_r ~ ̔a
//! @param c_h ~ ̍
//! @param RayPos_x C̈ʒuin_jw肷 XW
//! @param RayPos_y C̈ʒuin_jw肷 YW
//! @param RayPos_z C̈ʒuin_jw肷 ZW
//! @param RayDir_x C̃xNgw肷 X
//! @param RayDir_y C̃xNgw肷 Y
//! @param RayDir_z C̃xNgw肷 Z
//! @param Dist ~Ƃ̋󂯎|C^
//! @param maxDist ső勗@i0.0f𒴂lj
//! @return ĂFtrue@ĂȂFfalse
//! @warning RayPosin_j RayDirixNgjԈႦȂƁB
//! @attention Y֐ɗ~łB͕ς܂B
bool CollideCylinderRay(float c_x, float c_y, float c_z, float c_r, float c_h, float RayPos_x, float RayPos_y, float RayPos_z, float RayDir_x, float RayDir_y, float RayDir_z, float *Dist, float maxDist)
{
	float x, z, d;
	float cMinDist, cRayDist, cRDist;
	float Ray_min[2];
	float Ray_max[2];
	float Ray_tmin, Ray_tmax;


	// X-Zʂŉ~Ƃď

	//_ƃCn_̋
	x = c_x - RayPos_x;
	z = c_z - RayPos_z;
	d = sqrt(x*x + z*z);

	//_i~̒SjƃC̍ŒZ߂
	cMinDist = DistancePosRay(c_x, 0.0f, c_z, RayPos_x, 0.0f, RayPos_z, RayDir_x, 0.0f, RayDir_z);

	//ŒZa藣Ă鎞_œȂ
	if( cMinDist > c_r ){
		return false;
	}

	cRayDist = sqrt(d*d - cMinDist*cMinDist);		//iCn_j_ɍł߂Â
	cRDist = sqrt(c_r*c_r - cMinDist*cMinDist);	//i_aj_ɍł߂Â

	if( d < c_r ){
		Ray_min[0] = 0;		//n_~̒Ȃ狗[
	}
	else{
		Ray_min[0] = cRayDist - cRDist;	//C_ŒZ - aŒZ = C_甼a܂ł̍ŒZ
	}

	//_ƃCI_̋
	x = c_x - RayPos_x + RayDir_x*maxDist;
	z = c_z - RayPos_z + RayDir_x*maxDist;
	d = sqrt(x*x + z*z);

	if( d < c_r ){
		Ray_max[0] = maxDist;		//I_~̒Ȃő勗
	}
	else{
		Ray_max[0] = cRayDist + cRDist;
	}


	//ŶAABBƓl̏

	if( ( (RayPos_y < c_y)||((c_y+c_h) < RayPos_y) )&&(RayDir_y == 0.0f) ){
		//ɓĂȂ̂ xNg̕ 0 ȂAɉ~̊O
		return false;
	}

	//ɓĂȂAxNgȂ`
	float t1, t2;

	//~xNg˂_܂ł̋擾
	t1 = (c_y - RayPos_y) / RayDir_y;
	t2 = ((c_y+c_h) - RayPos_y) / RayDir_y;

	//_܂ł̋ŏEőŋtȂAւB
	if( t1 > t2 ){
		float temp = t1; t1 = t2; t2 = temp;
	}

	//̏ƂċL^
	Ray_min[1] = t1;
	Ray_max[1] = t2;

	if( (c_y <= RayPos_y)&&(RayPos_y <= (c_y+c_h)) ){
		//ɓĂ΁Â܂܎̏ƂċL^
		Ray_min[1] = 0.0f;
	}

	//}CiXixNgtjȂ~Ƃ Ȃ
	if( Ray_min[1] < 0.0f ){ return false; }
	if( Ray_max[1] < 0.0f ){ return false; }


	//vZ킹

	//eŁAłeŏfƍł߂eő勗fZo
	Ray_tmin = Ray_min[0];
	Ray_tmax = Ray_max[0];
	if( Ray_tmin < Ray_min[1] ){ Ray_tmin = Ray_min[1]; }
	if( Ray_tmax > Ray_max[1] ){ Ray_tmax = Ray_max[1]; }

	//ŏƍő勗̊֌W΁`
	if( (Ray_tmax - Ray_tmin) > 0 ){
		//ső勗艓΁A薳B
		if( maxDist > 0.0f ){
			if( Ray_tmin > maxDist ){ return false; }
		}

		//Ԃ
		if( Dist != NULL ){ *Dist = Ray_tmin; }
		return true;
	}

	return false;
}

//! @brief _ƃC̍ŒZ߂
//! @param Pos_x _XW
//! @param Pos_y _YW
//! @param Pos_z _ZW
//! @param RayPos_x C̈ʒuin_jw肷 XW
//! @param RayPos_y C̈ʒuin_jw肷 YW
//! @param RayPos_z C̈ʒuin_jw肷 ZW
//! @param RayDir_x C̃xNgw肷 X
//! @param RayDir_y C̃xNgw肷 Y
//! @param RayDir_z C̃xNgw肷 Z
//! @return ŒZ
//! @warning RayPosin_j RayDirixNgjԈႦȂƁB
//! @attention C͍̕l܂AC͍̒l܂B
float DistancePosRay(float Pos_x, float Pos_y, float Pos_z, float RayPos_x, float RayPos_y, float RayPos_z, float RayDir_x, float RayDir_y, float RayDir_z)
{
	float x1, y1, z1;
	float x2, y2, z2;
	float x3, y3, z3;
	float Dot;

	x1 = Pos_x - RayPos_x;
	y1 = Pos_y - RayPos_y;
	z1 = Pos_z - RayPos_z;
	x2 = RayDir_x;
	y2 = RayDir_y;
	z2 = RayDir_z;

	//
	Dot = x1 * x2 + y1 * y2 + z1 * z2;

	//C̃xNgtȂ
	if( Dot < 0.0f ){
		return sqrt(x1*x1 + y1*y1 + z1*z1);
	}

	//O
	x3 = y1 * z2 - z1 * y2;
	y3 = z1 * x2 - x1 * z2;
	z3 = x1 * y2 - y1 * x2;

	return sqrt(x3*x3 + y3*y3 + z3*z3) / sqrt(RayDir_x*RayDir_x + RayDir_y*RayDir_y + RayDir_z*RayDir_z);
}

//! @brief Ɛ̓蔻i2Dj
//! @param A1x A̎n_ XW
//! @param A1y A̎n_ YW
//! @param A2x ȀI_ XW
//! @param A2y ȀI_ YW
//! @param B1x B̎n_ XW
//! @param B1y B̎n_ YW
//! @param B2x B̏I_ XW
//! @param B2y B̏I_ YW
//! @param out_x _ XW 󂯎|C^iNULLj
//! @param out_y _ YW 󂯎|C^iNULLj
//! @return Ftrue@ȂFfalse
bool Collide2DLine(int A1x, int A1y, int A2x, int A2y, int B1x, int B1y, int B2x, int B2y, int *out_x, int *out_y)
{
	//̃xNg߂
	int Avx = A2x - A1x;
	int Avy = A2y - A1y;
	int Bvx = B2x - B1x;
	int Bvy = B2y - B1y;

	float v1_v2 = (float)(Avx * Bvy - Avy * Bvx);	//O
	if( v1_v2 == 0.0f ){
		return false;	//s
	}

	float vx = (float)(B1x - A1x);
	float vy = (float)(B1y - A1y);
	float v_v1 = vx * Avy - vy * Avx;	//O
	float v_v2 = vx * Bvy - vy * Bvx;	//O
	float t1 = v_v2 / v1_v2;
	float t2 = v_v1 / v1_v2;

	if( (t1 <= 0)||(1 <= t1)||(t2 <= 0)||(1 <= t2) ){
		return false;	//ĂȂ
	}

	if( out_x != NULL ){ *out_x = (int)(A1x + Avx * t1); }
	if( out_y != NULL ){ *out_y = (int)(A1y + Avy * t1); }

	return true;
}

//! @brief lp`Ɏ܂߂i2Dj
//! @param line_x1 ̎n_ XW
//! @param line_y1 ̎n_ YW
//! @param line_x2 ̏I_ XW
//! @param line_y2 ̏I_ YW
//! @param box_x1 lp`̍ XW
//! @param box_y1 lp`̍ YW
//! @param box_x2 lp`̉E XW
//! @param box_y2 lp`̉E YW
//! @param out_line_x1 lp`Ɏ܂ ̎n_ XW 󂯎|C^
//! @param out_line_y1 lp`Ɏ܂ ̎n_ YW 󂯎|C^
//! @param out_line_x2 lp`Ɏ܂ ̏I_ XW 󂯎|C^
//! @param out_line_y2 lp`Ɏ܂ ̏I_ YW 󂯎|C^
//! @return Li`悷jFtrue@i`悵ȂjFfalse
//! @warning ͕Kubox_x1 < box_x2vubox_x1 < box_x2vɂ邱
//! @note ȈՃ[_[̃}bv\p
bool Get2DLineInBox(int line_x1, int line_y1, int line_x2, int line_y2, int box_x1, int box_y1, int box_x2, int box_y2, int *out_line_x1, int *out_line_y1, int *out_line_x2, int *out_line_y2)
{
	//lp`w肪ُ
	if( (box_x1 >= box_x2)||(box_x1 >= box_x2) ){ return false; }

	//㉺E̋ԂɂȂAIO
	if( (line_x1 < box_x1)&&(line_x2 < box_x1) ){ return false; }
	if( (line_y1 < box_y1)&&(line_y2 < box_y1) ){ return false; }
	if( (box_x2 < line_x1)&&(box_x2 < line_x2) ){ return false; }
	if( (box_y2 < line_y1)&&(box_y2 < line_y2) ){ return false; }

	//Ɏlp`Ɏ܂
	if( (box_x1 <= line_x1)&&(line_x1 <= box_x2)&&(box_y1 <= line_y1)&&(line_y1 <= box_y2) ){
		if( (box_x1 <= line_x2)&&(line_x2 <= box_x2)&&(box_y1 <= line_y2)&&(line_y2 <= box_y2) ){
			*out_line_x1 = line_x1;
			*out_line_y1 = line_y1;
			*out_line_x2 = line_x2;
			*out_line_y2 = line_y2;
			return true;
		}
	}

	int x, y;

	//
	if( Collide2DLine(box_x1, box_y1, box_x2, box_y1, line_x1, line_y1, line_x2, line_y2, &x, &y) == true ){
		//n_lp`̓ȂI_AႦ΁iI_jn_B
		if( (box_x1 <= line_x1)&&(line_x1 <= box_x2)&&(box_y1 <= line_y1)&&(line_y1 <= box_y2) ){
			line_x2 = x;
			line_y2 = y;
		}
		else{
			line_x1 = x;
			line_y1 = y;
		}
	}
	//E
	if( Collide2DLine(box_x2, box_y1, box_x2, box_y2, line_x1, line_y1, line_x2, line_y2, &x, &y) == true ){
		//n_lp`̓ȂI_AႦ΁iI_jn_B
		if( (box_x1 <= line_x1)&&(line_x1 <= box_x2)&&(box_y1 <= line_y1)&&(line_y1 <= box_y2) ){
			line_x2 = x;
			line_y2 = y;
		}
		else{
			line_x1 = x;
			line_y1 = y;
		}
	}
	//
	if( Collide2DLine(box_x2, box_y2, box_x1, box_y2, line_x1, line_y1, line_x2, line_y2, &x, &y) == true ){
		//n_lp`̓ȂI_AႦ΁iI_jn_B
		if( (box_x1 <= line_x1)&&(line_x1 <= box_x2)&&(box_y1 <= line_y1)&&(line_y1 <= box_y2) ){
			line_x2 = x;
			line_y2 = y;
		}
		else{
			line_x1 = x;
			line_y1 = y;
		}
	}
	//
	if( Collide2DLine(box_x1, box_y2, box_x1, box_y1, line_x1, line_y1, line_x2, line_y2, &x, &y) == true ){
		//n_lp`̓ȂI_AႦ΁iI_jn_B
		if( (box_x1 <= line_x1)&&(line_x1 <= box_x2)&&(box_y1 <= line_y1)&&(line_y1 <= box_y2) ){
			line_x2 = x;
			line_y2 = y;
		}
		else{
			line_x1 = x;
			line_y1 = y;
		}
	}

	//߂Ďlp`Ɏ܂邩mF
	if( (box_x1 <= line_x1)&&(line_x1 <= box_x2)&&(box_y1 <= line_y1)&&(line_y1 <= box_y2) ){
		if( (box_x1 <= line_x2)&&(line_x2 <= box_x2)&&(box_y1 <= line_y2)&&(line_y2 <= box_y2) ){
			*out_line_x1 = line_x1;
			*out_line_y1 = line_y1;
			*out_line_x2 = line_x2;
			*out_line_y2 = line_y2;
			return true;
		}
	}

	return false;
}

//! @brief ϑ_Ώۓ_ւ EpxZo
//! @param pos_x ϑ_XW
//! @param pos_y ϑ_YW
//! @param pos_z ϑ_ZW
//! @param rx ϑ_̐px
//! @param ry ϑ_̐px
//! @param target_x Ώۓ_XW
//! @param target_y Ώۓ_YW
//! @param target_z Ώۓ_ZW
//! @param checkdist 苗i0.0fȉŔ薳j
//! @param out_rx Ώۓ_ւ̐pxi΁`-΁j󂯎|C^iNULLj
//! @param out_ry Ώۓ_ւ̐px󂯎|C^iNULLj
//! @param out_dist2 Ώۓ_ւ̋<b>̓</b>󂯎|C^iNULLj
//! @return Ftrue sFfalse
//! @warning out_dist2͋<b></b>łBKvɉĉ߂ sqrt()֐ ȂǂpĂB
//! @attention  checkdist ɗLȋ^ꍇ́Aϑ_Ώۓ_ւ̋s܂Bw肳ꂽ藣ĂꍇApxvZ false Ԃ܂B
//! @attention tɁA checkdist 0.0fȉ^ꍇAɂ锻s܂B֐͏ true Ԃ܂B
bool CheckTargetAngle(float pos_x, float pos_y, float pos_z, float rx, float ry, float target_x, float target_y, float target_z, float checkdist, float *out_rx, float *out_ry, float *out_dist2)
{
	float x, y, z;
	float dist2 = 0.0f;
	float angleX, angleY;

	x = target_x - pos_x;
	y = target_y - pos_y;
	z = target_z - pos_z;

	if( (checkdist > 0.0f)||(out_dist2 != NULL) ){
		dist2 = (x*x + y*y + z*z);
	}

	if( checkdist > 0.0f ){
		if( dist2 > checkdist * checkdist ){
			return false;
		}
	}

	if( out_rx != NULL ){
		angleX = atan2(z, x) - rx;
		for(; angleX > (float)M_PI; angleX -= (float)M_PI*2){}
		for(; angleX < (float)M_PI*-1; angleX += (float)M_PI*2){}

		*out_rx = angleX;
	}
	if( out_ry != NULL ){
		angleY = atan2(y, sqrt(x*x + z*z))  - ry;

		*out_ry = angleY;
	}
	if( out_dist2 != NULL ){ *out_dist2 = dist2; }

	return true;
}