//! @file collision.cpp
//! @brief CollisionNX̒`

//--------------------------------------------------------------------------------
// 
// OpenXOPS
// Copyright (c) 2014-2015, OpenXOPS Project / [-_-;](mikan) All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
// * Redistributions of source code must retain the above copyright notice, 
//   this list of conditions and the following disclaimer.
// * Redistributions in binary form must reproduce the above copyright notice, 
//   this list of conditions and the following disclaimer in the documentation 
//   and/or other materials provided with the distribution.
// * Neither the name of the OpenXOPS Project nor the@names of its contributors 
//   may be used to endorse or promote products derived from this software 
//   without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL OpenXOPS Project BE LIABLE FOR ANY
// DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
// ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//--------------------------------------------------------------------------------

#include "collision.h"

//! RXgN^
Collision::Collision()
{
	blockdata = NULL;
	bmin_x = new float[MAX_BLOCKS];
	bmin_y = new float[MAX_BLOCKS];
	bmin_z = new float[MAX_BLOCKS];
	bmax_x = new float[MAX_BLOCKS];
	bmax_y = new float[MAX_BLOCKS];
	bmax_z = new float[MAX_BLOCKS];
	BoardBlock = new bool[MAX_BLOCKS];
	bdata_worldgroup = new int[MAX_BLOCKS];
	for(int i=0; i<MAX_BLOCKS; i++){
		bmin_x[i] = 0.0f;
		bmin_y[i] = 0.0f;
		bmin_z[i] = 0.0f;
		bmax_x[i] = 0.0f;
		bmax_y[i] = 0.0f;
		bmax_z[i] = 0.0f;
		BoardBlock[i] = false;
		bdata_worldgroup[i] = 0;
	}
}

//! fBXgN^
Collision::~Collision()
{
	if( bmin_x != NULL ){ delete [] bmin_x; }
	if( bmin_y != NULL ){ delete [] bmin_y; }
	if( bmin_z != NULL ){ delete [] bmin_z; }
	if( bmax_x != NULL ){ delete [] bmax_x; }
	if( bmax_y != NULL ){ delete [] bmax_y; }
	if( bmax_z != NULL ){ delete [] bmax_z; }
	if( BoardBlock != NULL ){ delete [] BoardBlock; }
	if( bdata_worldgroup != NULL ){ delete [] bdata_worldgroup; }
}

//! ubNf[^荞
//! @param in_blockdata ubNf[^
int Collision::InitCollision(BlockDataInterface* in_blockdata)
{
	int bs;
	struct blockdata data;
	int vID[4];
	float g0, g1, costheta;

	if( in_blockdata == NULL ){ return 1; }
	blockdata = in_blockdata;

	bs = blockdata->GetTotaldatas();

	for(int i=0; i<bs; i++){
		BoardBlock[i] = false;
	}

	//pʍ쐬
	for(int i=0; i<bs; i++){
		blockdata->Getdata(&data, i);
		for(int j=0; j<6; j++){
			blockdataface(j, &vID[0], NULL);

			D3DXPlaneFromPoints(&bdata_plane[i][j][0],
				&D3DXVECTOR3( data.x[ vID[1] ], data.y[ vID[1] ], data.z[ vID[1] ] ),
				&D3DXVECTOR3( data.x[ vID[2] ], data.y[ vID[2] ], data.z[ vID[2] ] ),
				&D3DXVECTOR3( data.x[ vID[0] ], data.y[ vID[0] ], data.z[ vID[0] ] ) );
			D3DXPlaneFromPoints(&bdata_plane[i][j][1],
				&D3DXVECTOR3( data.x[ vID[0] ], data.y[ vID[0] ], data.z[ vID[0] ] ),
				&D3DXVECTOR3( data.x[ vID[2] ], data.y[ vID[2] ], data.z[ vID[2] ] ),
				&D3DXVECTOR3( data.x[ vID[3] ], data.y[ vID[3] ], data.z[ vID[3] ] ) );

			//2̎Op`@̂Ȃp߂
			g0 = sqrt(bdata_plane[i][j][0].a * bdata_plane[i][j][0].a + bdata_plane[i][j][0].b * bdata_plane[i][j][0].b + bdata_plane[i][j][0].c * bdata_plane[i][j][0].c);
			g1 = sqrt(bdata_plane[i][j][1].a * bdata_plane[i][j][1].a + bdata_plane[i][j][1].b * bdata_plane[i][j][1].b + bdata_plane[i][j][1].c * bdata_plane[i][j][1].c);
			costheta = (bdata_plane[i][j][0].a * bdata_plane[i][j][1].a + bdata_plane[i][j][0].b * bdata_plane[i][j][1].b + bdata_plane[i][j][0].c * bdata_plane[i][j][1].c) / (g0 * g1);

			//1̖ʂŖ@90xȏႤiُjȂ`
			if( acos(costheta) > (float)M_PI/2 ){
				//ႤOp`ō
				D3DXPlaneFromPoints(&bdata_plane[i][j][0],
					&D3DXVECTOR3( data.x[ vID[2] ], data.y[ vID[2] ], data.z[ vID[2] ] ),
					&D3DXVECTOR3( data.x[ vID[3] ], data.y[ vID[3] ], data.z[ vID[3] ] ),
					&D3DXVECTOR3( data.x[ vID[1] ], data.y[ vID[1] ], data.z[ vID[1] ] ) );
				D3DXPlaneFromPoints(&bdata_plane[i][j][1],
					&D3DXVECTOR3( data.x[ vID[1] ], data.y[ vID[1] ], data.z[ vID[1] ] ),
					&D3DXVECTOR3( data.x[ vID[3] ], data.y[ vID[3] ], data.z[ vID[3] ] ),
					&D3DXVECTOR3( data.x[ vID[0] ], data.y[ vID[0] ], data.z[ vID[0] ] ) );
			}
		}
	}

	//̃ubNo
	for(int i=0; i<bs; i++){
		blockdata->Getdata(&data, i);

		BoardBlock[i] = false;

		for(int j=0; j<8; j++){
			for(int k=j+1; k<8; k++){
				//ꃕł_ȂAɂȂĂƔB
				if( (data.x[j] == data.x[k])&&(data.y[j] == data.y[k])&&(data.z[j] == data.z[k]) ){
					BoardBlock[i] = true;
				}
			}
		}
	}


	//ubNAABB쐬
	for(int i=0; i<bs; i++){
		blockdata->Getdata(&data, i);
		GetBlockPosMINMAX(data, &bmin_x[i], &bmin_y[i], &bmin_z[i], &bmax_x[i], &bmax_y[i], &bmax_z[i]);
	}

	//ubN̋ԕO[vvZ
	for(int i=0; i<bs; i++){
		///*
		blockdata->Getdata(&data, i);

		bdata_worldgroup[i] = GetWorldGroup(bmin_x[i], bmin_z[i]);
		if( GetWorldGroup(bmax_x[i], bmax_z[i]) != bdata_worldgroup[i] ){
			bdata_worldgroup[i] = 0;
		}
		//*/
		//bdata_worldgroup[i] = 0;
	}

	return 0;
}

//! ubN̍WőlEŏlԂ
//! @param data ubNf[^̍\
//! @param *min_x ŏ XWԂ|C^
//! @param *min_y ŏ YWԂ|C^
//! @param *min_z ŏ ZWԂ|C^
//! @param *max_x ő XWԂ|C^
//! @param *max_y ő YWԂ|C^
//! @param *max_z ő ZWԂ|C^
void Collision::GetBlockPosMINMAX(struct blockdata data, float *min_x, float *min_y, float *min_z, float *max_x, float *max_y, float *max_z)
{
	*min_x = data.x[0];
	*min_y = data.y[0];
	*min_z = data.z[0];
	*max_x = data.x[0];
	*max_y = data.y[0];
	*max_z = data.z[0];
	for(int i=1; i<8; i++){
		if( *min_x > data.x[i] ){ *min_x = data.x[i]; }
		if( *min_y > data.y[i] ){ *min_y = data.y[i]; }
		if( *min_z > data.z[i] ){ *min_z = data.z[i]; }

		if( *max_x < data.x[i] ){ *max_x = data.x[i]; }
		if( *max_y < data.y[i] ){ *max_y = data.y[i]; }
		if( *max_z < data.z[i] ){ *max_z = data.z[i]; }
	}
}

//! ԕ̃O[vZo
//! @param x XW
//! @param z ZW
//! @return O[vԍ
//! @attention uO[vԍv̊TO͕ʓrhLgQ
int Collision::GetWorldGroup(float x, float z)
{
	if( (x > 0)&&(z > 0) ){ return 1; }
	if( (x < 0)&&(z > 0) ){ return 2; }
	if( (x < 0)&&(z < 0) ){ return 3; }
	if( (x > 0)&&(z < 0) ){ return 4; }
	return 0;
}

//! ubNɖ܂ĂȂׂ
//! @param blockid 肷ubNԍ
//! @param x XW
//! @param y XW
//! @param z XW
//! @param worldgroup Ԃ̃O[v𗘗pČvZȗ݂itrueFLEvZȗ@falseFESj
//! @param *planeid \ɂʔԍiNULLj
//! @return ĂFtrue@ĂȂFfalse
//! @warning *planeid Ԃ\ʁi0`5j́A̖ʂYꍇłAŏɌ1ʂ̂ݕԂ܂B
bool Collision::CheckBlockInside(int blockid, float x, float y, float z, bool worldgroup, int *planeid)
{
	if( blockdata == NULL ){ return false; }
	if( (blockid < 0)||(blockdata->GetTotaldatas() <= blockid) ){ return false; }

	//̃ubN͌vZO
	if( BoardBlock[blockid] == true ){ return false; }


	//̍r
	if( worldgroup == true ){

		//ԕ
		if( bdata_worldgroup[blockid] != 0 ){
			//ϑ_̋Ԃ̃O[v擾
			int worldgroup = GetWorldGroup(x, z);

			if( worldgroup != 0 ){
				//Ԃ̃O[vႦΌvZO
				if( bdata_worldgroup[blockid] != worldgroup ){
					return false;
				}
			}
		}

		//͈͂Ō
		if( (x < bmin_x[blockid])||(bmax_x[blockid] < x) ){ return false; }
		if( (y < bmin_y[blockid])||(bmax_y[blockid] < y) ){ return false; }
		if( (z < bmin_z[blockid])||(bmax_z[blockid] < z) ){ return false; }
	}


	//6ʂ猩đSėʂǂ
	for(int i=0; i<6; i++){
		if( (D3DXPlaneDotCoord(&bdata_plane[blockid][i][0], &D3DXVECTOR3( x, y, z )) > 0)||(D3DXPlaneDotCoord(&bdata_plane[blockid][i][1], &D3DXVECTOR3( x, y, z )) > 0) ){
			if( planeid != NULL ){ *planeid = i; }
			return false;	//\ʂȂΏI
		}
	}

	return true;
}

//! SẴubNɖ܂ĂȂׂ
//! @param x XW
//! @param y XW
//! @param z XW
//! @return ĂFtrue@ĂȂFfalse
bool Collision::CheckALLBlockInside(float x, float y, float z)
{
	if( blockdata == NULL ){ return false; }

	int bs = blockdata->GetTotaldatas();

	for(int i=0; i<bs; i++){
		if( CheckBlockInside(i, x, y, z, true, NULL) == true ){ return true; }
	}

	return false;
}

//! SẴubNƃCiĵ蔻
//! @param blockid 肷ubNԍ
//! @param RayPos_x C̈ʒuin_jw肷 XW
//! @param RayPos_y C̈ʒuin_jw肷 YW
//! @param RayPos_z C̈ʒuin_jw肷 ZW
//! @param RayDir_x C̃xNgw肷 X
//! @param RayDir_y C̃xNgw肷 Y
//! @param RayDir_z C̃xNgw肷 Z
//! @param face ubN̖ʔԍi0`5j󂯎|C^@iNULLj
//! @param Dist ubNƂ̋󂯎|C^
//! @param maxDist ső勗@i0.0 ŖEj
//! @return ĂFtrue@ĂȂFfalse
//! @warning RayPosin_j RayDirixNgjԈႦȂƁB
//! @warning ső勗w肵ȂƁAptH[}X啝ɒቺ܂B
//! @attention C̎n_痠ɂȂubN̖ʂ͖܂B[ɕό`̃ubN܂B
//! @attention ܂AC̃ubNɓꍇ́AC̎n_ԋ߂Ԃ܂B
bool Collision::CheckBlockIntersectRay(int blockid, float RayPos_x, float RayPos_y, float RayPos_z, float RayDir_x, float RayDir_y, float RayDir_z, int *face, float *Dist, float maxDist)
{
	if( blockdata == NULL ){ return false; }

	int bs = blockdata->GetTotaldatas();
	struct blockdata data;
	int vID[4];
	float pDist;
	float min_pDist = FLT_MAX;
	int min_blockface = -1;
	float rmin_x, rmin_y, rmin_z, rmax_x, rmax_y, rmax_z;
	int worldgroupA, worldgroupB;

	//̃ubN͌vZO
	if( BoardBlock[blockid] == true ){
		if( face != NULL ){ *face = 0; }
		*Dist = 0.0f;
		return false;
	}

	if( maxDist > 0.0f ){
		//CAABB
		rmin_x = RayPos_x + RayDir_x * maxDist;
		rmin_y = RayPos_y + RayDir_y * maxDist;
		rmin_z = RayPos_z + RayDir_z * maxDist;
		rmax_x = rmin_x;
		rmax_y = rmin_y;
		rmax_z = rmin_z;
		if( rmin_x > RayPos_x ){ rmin_x = RayPos_x; }
		if( rmin_y > RayPos_y ){ rmin_y = RayPos_y; }
		if( rmin_z > RayPos_z ){ rmin_z = RayPos_z; }
		if( rmax_x < RayPos_x ){ rmax_x = RayPos_x; }
		if( rmax_y < RayPos_y ){ rmax_y = RayPos_y; }
		if( rmax_z < RayPos_z ){ rmax_z = RayPos_z; }
	}

	D3DXVECTOR3 pRayPos(RayPos_x, RayPos_y, RayPos_z);
	D3DXVECTOR3 pRayDir(RayDir_x, RayDir_y, RayDir_z);

	if( maxDist > 0.0f ){
		//n_ƏI_̋ԃO[v擾
		worldgroupA = GetWorldGroup(RayPos_x, RayPos_z);
		worldgroupB = GetWorldGroup(RayPos_x + RayDir_x * maxDist, RayPos_z + RayDir_z * maxDist);
	}

	blockdata->Getdata(&data, blockid);

	if( maxDist > 0.0f ){
		if( bdata_worldgroup[blockid] != 0 ){
			//n_ƏI_AԂ̃O[voĂȂ
			if( (worldgroupA != 0)&&(worldgroupA == worldgroupB) ){

				//Ԃ̃O[vႦΌvZO
				if( bdata_worldgroup[blockid] == worldgroupA ){		//worldgroupA == worldgroupB
					//E{bNXmŔ
					if( CollideBoxAABB(bmin_x[blockid], bmin_y[blockid], bmin_z[blockid], bmax_x[blockid], bmax_y[blockid], bmax_z[blockid], rmin_x, rmin_y, rmin_z, rmax_x, rmax_y, rmax_z) == false ){
						//ĂȂ΁ÃubNׂ͒ȂB
						if( face != NULL ){ *face = 0; }
						*Dist = 0.0f;
						return false;
					}
				}
			}
		}
	}

	//E{bNXƃCŔ
	if( CollideAABBRay(bmin_x[blockid], bmin_y[blockid], bmin_z[blockid], bmax_x[blockid], bmax_y[blockid], bmax_z[blockid], RayPos_x, RayPos_y, RayPos_z, RayDir_x, RayDir_y, RayDir_z, NULL, -1.0f) == false ){
		//ĂȂ΁ÃubNׂ͒ȂB
		if( face != NULL ){ *face = 0; }
		*Dist = 0.0f;
		return false;
	}

	//e|SPʂŔ
	for(int i=0; i<6; i++){
		blockdataface(i, vID, NULL);
		if( (D3DXPlaneDotCoord(&bdata_plane[blockid][i][0], &pRayPos) >= 0)||(D3DXPlaneDotCoord(&bdata_plane[blockid][i][1], &pRayPos) >= 0) ){
			if( (D3DXIntersectTri(
				&D3DXVECTOR3( data.x[vID[0]], data.y[vID[0]], data.z[vID[0]]),
				&D3DXVECTOR3( data.x[vID[1]], data.y[vID[1]], data.z[vID[1]]),
				&D3DXVECTOR3( data.x[vID[2]], data.y[vID[2]], data.z[vID[2]]), 
				&pRayPos, &pRayDir, NULL,  NULL, &pDist) == TRUE)||
				(D3DXIntersectTri(
				&D3DXVECTOR3( data.x[vID[2]], data.y[vID[2]], data.z[vID[2]]),
				&D3DXVECTOR3( data.x[vID[3]], data.y[vID[3]], data.z[vID[3]]),
				&D3DXVECTOR3( data.x[vID[0]], data.y[vID[0]], data.z[vID[0]]), 
				&pRayPos, &pRayDir, NULL,  NULL, &pDist) == TRUE)
			){
				if( min_pDist > pDist ){
					min_pDist = pDist;
					min_blockface = i;
				}
			}
		}
	}

	//Ȃ΁A|C^ɓKȐĕԂB
	if( min_blockface == -1 ){
		if( face != NULL ){ *face = 0; }
		*Dist = 0.0f;
		return false;
	}

	//vZʂĕԂ
	if( face != NULL ){ *face = min_blockface; }
	*Dist = min_pDist;
	return true;
}

//! SẴubNƃCiĵ蔻
//! @param RayPos_x C̈ʒuin_jw肷 XW
//! @param RayPos_y C̈ʒuin_jw肷 YW
//! @param RayPos_z C̈ʒuin_jw肷 ZW
//! @param RayDir_x C̃xNgw肷 X
//! @param RayDir_y C̃xNgw肷 Y
//! @param RayDir_z C̃xNgw肷 Z
//! @param id ubNID󂯎|C^@iNULLj
//! @param face ubN̖ʔԍi0`5j󂯎|C^@iNULLj
//! @param Dist ubNƂ̋󂯎|C^
//! @param maxDist ső勗@i0.0 ŖEj
//! @return ĂFtrue@ĂȂFfalse
//! @warning RayPosin_j RayDirixNgjԈႦȂƁB
//! @warning ső勗w肵ȂƁAptH[}X啝ɒቺ܂B
//! @warning g CheckALLBlockIntersectDummyRay()֐ ƗގĂ܂A֐荂xŒᑬłB
//! @attention C̎n_痠ɂȂubN̖ʂ͖܂B[ɕό`̃ubN܂B
//! @attention ܂AC̃ubNɓꍇ́A<b>C̎n_ԋ߂</b>Ԃ܂B
bool Collision::CheckALLBlockIntersectRay(float RayPos_x, float RayPos_y, float RayPos_z, float RayDir_x, float RayDir_y, float RayDir_z, int *id, int *face, float *Dist, float maxDist)
{
	if( blockdata == NULL ){ return false; }

	int bs = blockdata->GetTotaldatas();
	struct blockdata data;
	int vID[4];
	float pDist;
	float min_pDist = FLT_MAX;
	int min_blockid = -1;
	int min_blockface = -1;
	float rmin_x, rmin_y, rmin_z, rmax_x, rmax_y, rmax_z;
	int worldgroupA, worldgroupB;

	if( maxDist > 0.0f ){
		//CAABB
		rmin_x = RayPos_x + RayDir_x * maxDist;
		rmin_y = RayPos_y + RayDir_y * maxDist;
		rmin_z = RayPos_z + RayDir_z * maxDist;
		rmax_x = rmin_x;
		rmax_y = rmin_y;
		rmax_z = rmin_z;
		if( rmin_x > RayPos_x ){ rmin_x = RayPos_x; }
		if( rmin_y > RayPos_y ){ rmin_y = RayPos_y; }
		if( rmin_z > RayPos_z ){ rmin_z = RayPos_z; }
		if( rmax_x < RayPos_x ){ rmax_x = RayPos_x; }
		if( rmax_y < RayPos_y ){ rmax_y = RayPos_y; }
		if( rmax_z < RayPos_z ){ rmax_z = RayPos_z; }
	}

	D3DXVECTOR3 pRayPos(RayPos_x, RayPos_y, RayPos_z);
	D3DXVECTOR3 pRayDir(RayDir_x, RayDir_y, RayDir_z);

	if( maxDist > 0.0f ){
		//n_ƏI_̋ԃO[v擾
		worldgroupA = GetWorldGroup(RayPos_x, RayPos_z);
		worldgroupB = GetWorldGroup(RayPos_x + RayDir_x * maxDist, RayPos_z + RayDir_z * maxDist);
	}
	
	for(int i=0; i<bs; i++){
		//̃ubN͌vZO
		if( BoardBlock[i] == true ){ continue; }

		blockdata->Getdata(&data, i);

		if( maxDist > 0.0f ){
			if( bdata_worldgroup[i] != 0 ){
				//n_ƏI_AԂ̃O[voĂȂ
				if( (worldgroupA != 0)&&(worldgroupA == worldgroupB) ){

					//Ԃ̃O[vႦΌvZO
					if( bdata_worldgroup[i] == worldgroupA ){		//worldgroupA == worldgroupB

						//E{bNXmŔ
						if( CollideBoxAABB(bmin_x[i], bmin_y[i], bmin_z[i], bmax_x[i], bmax_y[i], bmax_z[i], rmin_x, rmin_y, rmin_z, rmax_x, rmax_y, rmax_z) == false ){
							continue;	//ĂȂ΁ÃubNׂ͒ȂB
						}

					}
				}
			}
		}

		//E{bNXƃCŔ
		if( CollideAABBRay(bmin_x[i], bmin_y[i], bmin_z[i], bmax_x[i], bmax_y[i], bmax_z[i], RayPos_x, RayPos_y, RayPos_z, RayDir_x, RayDir_y, RayDir_z, NULL, -1.0f) == false ){
			continue;		//ĂȂ΁ÃubNׂ͒ȂB
		}

		//e|SPʂŔ
		for(int j=0; j<6; j++){
			blockdataface(j, vID, NULL);
			if( (D3DXPlaneDotCoord(&bdata_plane[i][j][0], &pRayPos) >= 0)||(D3DXPlaneDotCoord(&bdata_plane[i][j][1], &pRayPos) >= 0) ){
				if( (D3DXIntersectTri(
					&D3DXVECTOR3( data.x[vID[0]], data.y[vID[0]], data.z[vID[0]]),
					&D3DXVECTOR3( data.x[vID[1]], data.y[vID[1]], data.z[vID[1]]),
					&D3DXVECTOR3( data.x[vID[2]], data.y[vID[2]], data.z[vID[2]]), 
					&pRayPos, &pRayDir, NULL,  NULL, &pDist) == TRUE)||
					(D3DXIntersectTri(
					&D3DXVECTOR3( data.x[vID[2]], data.y[vID[2]], data.z[vID[2]]),
					&D3DXVECTOR3( data.x[vID[3]], data.y[vID[3]], data.z[vID[3]]),
					&D3DXVECTOR3( data.x[vID[0]], data.y[vID[0]], data.z[vID[0]]), 
					&pRayPos, &pRayDir, NULL,  NULL, &pDist) == TRUE)
				){
					if( min_pDist > pDist ){
						min_pDist = pDist;
						min_blockid = i;
						min_blockface = j;
					}
				}
			}
		}
	}

	//ő勗𒴂Ă΁A𖳌ɁB
	if( maxDist >= 0.0f ){
		if( min_pDist > maxDist ){
			min_blockid = -1;
		}
	}

	//Ȃ΁A|C^ɓKȐĕԂB
	if( min_blockid == -1 ){
		if( id != NULL ){ *id = 0; }
		if( face != NULL ){ *face = 0; }
		*Dist = 0.0f;
		return false;
	}

	//vZʂĕԂ
	if( id != NULL ){ *id = min_blockid; }
	if( face != NULL ){ *face = min_blockface; }
	*Dist = min_pDist;
	return true;
}

//! SẴubNƏՓ˂Ă邩
//! @param RayPos_x n_ XW
//! @param RayPos_y n_ YW
//! @param RayPos_z n_ ZW
//! @param RayDir_x xNg X
//! @param RayDir_y xNg Y
//! @param RayDir_z xNg Z
//! @param id ubNID󂯎|C^@iNULLj
//! @param face ubN̖ʔԍi0`5j󂯎|C^@iNULLj
//! @param Dist _~[ϐ̃|C^i 0.0f Ԃj
//! @param maxDist ső勗@i0.0 w sj
//! @return ĂFtrue@ĂȂFfalse
//! @warning n_ xNgԈႦȂƁB
//! @warning g CheckALLBlockIntersectRay()֐ ƗގĂ܂A֐荂ŒᐸxłB
//! @attention C̎n_痠ɂȂubN̖ʂ͖܂B[ɕό`̃ubN܂B
//! @attention ܂AC̃ubNɓꍇ́A<b>ԍŏɔ</b>Ԃ܂B
bool Collision::CheckALLBlockIntersectDummyRay(float RayPos_x, float RayPos_y, float RayPos_z, float RayDir_x, float RayDir_y, float RayDir_z, int *id, int *face, float *Dist, float maxDist)
{
	if( blockdata == NULL ){ return false; }
	if( maxDist <= 0.0f ){ return false; }

	float RayPos2_x = RayPos_x + RayDir_x * maxDist/2;
	float RayPos2_y = RayPos_y + RayDir_y * maxDist/2;
	float RayPos2_z = RayPos_z + RayDir_z * maxDist/2;
	float RayPos3_x = RayPos_x + RayDir_x * maxDist;
	float RayPos3_y = RayPos_y + RayDir_y * maxDist;
	float RayPos3_z = RayPos_z + RayDir_z * maxDist;
	
	if( (id == NULL)&&(face == NULL) ){
		for(int i=0; i<MAX_BLOCKS; i++){
			//I_
			if( CheckBlockInside(i, RayPos3_x, RayPos3_y, RayPos3_z, true, NULL) == true ){
				*Dist = 0.0f;
				return true;
			}

			//Ԏ_
			if( CheckBlockInside(i, RayPos2_x, RayPos2_y, RayPos2_z, true, NULL) == true ){
				*Dist = 0.0f;
				return true;
			}
		}

		*Dist = 0.0f;
		return false;
	}

	for(int i=0; i<MAX_BLOCKS; i++){
		int surface;

		//Jnn_
		CheckBlockInside(i, RayPos_x, RayPos_y, RayPos_z, false, &surface);

		//I_
		if( CheckBlockInside(i, RayPos3_x, RayPos3_y, RayPos3_z, true, &surface) == true ){
			if( id != NULL ){ *id = i; }
			if( face != NULL ){ *face = surface; }
			*Dist = 0.0f;
			return true;
		}

		//Ԏ_
		if( CheckBlockInside(i, RayPos2_x, RayPos2_y, RayPos2_z, true, &surface) == true ){
			if( id != NULL ){ *id = i; }
			if( face != NULL ){ *face = surface; }
			*Dist = 0.0f;
			return true;
		}
	}

	if( id != NULL ){ *id = 0; }
	if( face != NULL ){ *face = 0; }
	*Dist = 0.0f;
	return false;
}

//! ubNɉĈړxNg߂
void Collision::ScratchVector(BlockDataInterface* in_blockdata, int id, int face, float in_vx, float in_vy, float in_vz, float *out_vx, float *out_vy, float *out_vz)
{
	struct blockdata bdata;
	in_blockdata->Getdata(&bdata, id);

	D3DXVECTOR3 out;
	D3DXVECTOR3 front(in_vx, in_vy, in_vz);
	D3DXVECTOR3 normal(bdata.material[face].vx, bdata.material[face].vy, bdata.material[face].vz);

	//D3DXVec3Normalize(&out, &(front - D3DXVec3Dot(&front, &normal) * normal));
	out = (front - D3DXVec3Dot(&front, &normal) * normal);

	*out_vx = out.x;
	*out_vy = out.y;
	*out_vz = out.z;
}

//! ubNɔ˂xNg߂
void Collision::ReflectVector(BlockDataInterface* in_blockdata, int id, int face, float in_vx, float in_vy, float in_vz, float *out_vx, float *out_vy, float *out_vz)
{
	struct blockdata bdata;
	in_blockdata->Getdata(&bdata, id);

	D3DXVECTOR3 out;
	D3DXVECTOR3 front(in_vx, in_vy, in_vz);
	D3DXVECTOR3 normal(bdata.material[face].vx, bdata.material[face].vy, bdata.material[face].vz);

	//D3DXVec3Normalize(&out, &(front - 2.0f * D3DXVec3Dot(&front, &normal) * normal));
	out = (front - 2.0f * D3DXVec3Dot(&front, &normal) * normal);

	*out_vx = out.x;
	*out_vy = out.y;
	*out_vz = out.z;
}


//! AABBɂ邠蔻
//! @param box1_min_x A̍ŏ XW
//! @param box1_min_y A̍ŏ YW
//! @param box1_min_z A̍ŏ ZW
//! @param box1_max_x A̍ő XW
//! @param box1_max_y A̍ő YW
//! @param box1_max_z A̍ő ZW
//! @param box2_min_x B̍ŏ XW
//! @param box2_min_y B̍ŏ YW
//! @param box2_min_z B̍ŏ ZW
//! @param box2_max_x B̍ő XW
//! @param box2_max_y B̍ő YW
//! @param box2_max_z B̍ő ZW
//! @return ĂFtrue@ĂȂFfalse
//! @attention G[ꍇuĂȂFfalsevԂ܂B
bool CollideBoxAABB(float box1_min_x, float box1_min_y, float box1_min_z, float box1_max_x, float box1_max_y, float box1_max_z, float box2_min_x, float box2_min_y, float box2_min_z, float box2_max_x, float box2_max_y, float box2_max_z)
{
	//G[΍
	if( box1_min_x > box1_max_x ){ return false; }
	if( box1_min_y > box1_max_y ){ return false; }
	if( box1_min_z > box1_max_z ){ return false; }
	if( box2_min_x > box2_max_x ){ return false; }
	if( box2_min_y > box2_max_y ){ return false; }
	if( box2_min_z > box2_max_z ){ return false; }

	if(
		(box1_min_x < box2_max_x)&&(box1_max_x > box2_min_x)&&
		(box1_min_y < box2_max_y)&&(box1_max_y > box2_min_y)&&
		(box1_min_z < box2_max_z)&&(box1_max_z > box2_min_z)
	){
		return true;
	}

	return false;
}

//! ~m̓蔻
//! @param c1_x ~1 ӂxW
//! @param c1_y ~1 ӂyW
//! @param c1_z ~1 ӂzW
//! @param c1_r ~1 ̔a
//! @param c1_h ~1 ̍
//! @param c2_x ~2 ӂxW
//! @param c2_y ~2 ӂyW
//! @param c2_z ~2 ӂzW
//! @param c2_r ~2 ̔a
//! @param c2_h ~2 ̍
//! @param *angle ~1݂ڐGpx󂯎|C^iNULLj
//! @param *length ~1݂ڐG󂯎|C^iNULLj
//! @return ĂFtrue@ĂȂFfalse
bool CollideCylinder(float c1_x, float c1_y, float c1_z, float c1_r, float c1_h, float c2_x, float c2_y, float c2_z, float c2_r, float c2_h, float *angle, float *length)
{
	//ɂyȍŔ
	if( (c1_y < c2_y + c2_h)&&(c1_y + c1_h > c2_y) ){

		//Ŕ
		float x = c1_x - c2_x;
		float z = c1_z - c2_z;
		float caser = x*x + z*z;
		float minr = (c1_r+c2_r) * (c1_r+c2_r);
		if( caser < minr ){
			if( angle != NULL ){ *angle = atan2(z, x); }
			if( length != NULL ){ *length = sqrt(minr) - sqrt(caser); }
			return true;
		}
	}

	return false;
}

//! ̂ƃCiĵ蔻
//! @param s_x ̂xW
//! @param s_y ̂yW
//! @param s_z ̂zW
//! @param s_r ̂̔a
//! @param RayPos_x C̈ʒuin_jw肷 XW
//! @param RayPos_y C̈ʒuin_jw肷 YW
//! @param RayPos_z C̈ʒuin_jw肷 ZW
//! @param RayDir_x C̃xNgw肷 X
//! @param RayDir_y C̃xNgw肷 Y
//! @param RayDir_z C̃xNgw肷 Z
//! @param Dist ̂Ƃ̋󂯎|C^
//! @param maxDist ső勗@i0.0 ŖEj
//! @return ĂFtrue@ĂȂFfalse
//! @warning RayPosin_j RayDirixNgjԈႦȂƁB
//! @warning ső勗w肵ȂƁAptH[}X啝ɒቺ܂B
bool CollideSphereRay(float s_x, float s_y, float s_z, float s_r, float RayPos_x, float RayPos_y, float RayPos_z, float RayDir_x, float RayDir_y, float RayDir_z, float *Dist, float maxDist)
{
	if( maxDist > 0.0f ){
		float pmin_x, pmin_y, pmin_z, pmax_x, pmax_y, pmax_z;

		//CAABB
		pmin_x = RayPos_x + RayDir_x * maxDist;
		pmin_y = RayPos_y + RayDir_y * maxDist;
		pmin_z = RayPos_z + RayDir_z * maxDist;
		pmax_x = pmin_x;
		pmax_y = pmin_y;
		pmax_z = pmin_z;
		if( pmin_x > RayPos_x ){ pmin_x = RayPos_x; }
		if( pmin_y > RayPos_y ){ pmin_y = RayPos_y; }
		if( pmin_z > RayPos_z ){ pmin_z = RayPos_z; }
		if( pmax_x < RayPos_x ){ pmax_x = RayPos_x; }
		if( pmax_y < RayPos_y ){ pmax_y = RayPos_y; }
		if( pmax_z < RayPos_z ){ pmax_z = RayPos_z; }

		//E{bNXmŔ
		if( CollideBoxAABB(s_x - s_r, s_y - s_r, s_z - s_r, s_x + s_r, s_y + s_r, s_z + s_r, pmin_x, pmin_y, pmin_z, pmax_x, pmax_y, pmax_z) == FALSE ){
			return false;
		}
	}

	D3DXVECTOR3 pCenter(s_x, s_y, s_z);
	D3DXVECTOR3 pRayPos(RayPos_x, RayPos_y, RayPos_z);
	D3DXVECTOR3 pRayDir(RayDir_x, RayDir_y, RayDir_z);

	//̂ƃC̔
	if( D3DXSphereBoundProbe(&pCenter, s_r, &pRayPos, &pRayDir) == TRUE ){
		if( Dist != NULL ){
			if( maxDist < 0.0f ){ maxDist = s_r * 2; }

			//_ƃCn_̋
			float x, y, z, d;
			x = s_x - RayPos_x;
			y = s_y - RayPos_y;
			z = s_z - RayPos_z;
			d = sqrt(x*x + y*y + z*z);

			//Cn_a߂i߂荞łj
			if( d < s_r ){
				*Dist = 0.0f;
				return true;
			}

			//_î̒SjƃC̍ŒZ߂
			float MinDist, RayDist, RDist;
				//_ƃC̍ŒZ
			MinDist = DistancePosRay(s_x, s_y, s_z, RayPos_x, RayPos_y, RayPos_z, RayDir_x, RayDir_y, RayDir_z, maxDist);
			RayDist = sqrt(d*d - MinDist*MinDist);		//iCn_j_ɍł߂Â
			RDist = sqrt(s_r*s_r - MinDist*MinDist);	//i_aj_ɍł߂Â

			*Dist = RayDist - RDist;	//C_ŒZ - aŒZ = C_甼a܂ł̍ŒZ
		}
		return true;
	}

	return false;
}

//! AABBƃCiĵ蔻
//! @param box_min_x ̂̍ŏ XW
//! @param box_min_y ̂̍ŏ YW
//! @param box_min_z ̂̍ŏ ZW
//! @param box_max_x ̂̍ő XW
//! @param box_max_y ̂̍ő YW
//! @param box_max_z ̂̍ő ZW
//! @param RayPos_x C̈ʒuin_jw肷 XW
//! @param RayPos_y C̈ʒuin_jw肷 YW
//! @param RayPos_z C̈ʒuin_jw肷 ZW
//! @param RayDir_x C̃xNgw肷 X
//! @param RayDir_y C̃xNgw肷 Y
//! @param RayDir_z C̃xNgw肷 Z
//! @param Dist AABBƂ̋󂯎|C^
//! @param maxDist ső勗@i0.0 ŖEj
//! @return ĂFtrue@ĂȂFfalse
//! @warning RayPosin_j RayDirixNgjԈႦȂƁB
//! @warning ső勗w肵ȂƁAptH[}X啝ɒቺ܂B
//! @todo ƂċߎlԂ̂ŁAmȒl߂B
bool CollideAABBRay(float box_min_x, float box_min_y, float box_min_z, float box_max_x, float box_max_y, float box_max_z, float RayPos_x, float RayPos_y, float RayPos_z, float RayDir_x, float RayDir_y, float RayDir_z, float *Dist, float maxDist)
{
	if( box_min_x > box_max_x ){ return false; }
	if( box_min_y > box_max_y ){ return false; }
	if( box_min_z > box_max_z ){ return false; }

	if( maxDist > 0.0f ){
		float pmin_x, pmin_y, pmin_z, pmax_x, pmax_y, pmax_z;

		//CAABB
		pmin_x = RayPos_x + RayDir_x * maxDist;
		pmin_y = RayPos_y + RayDir_y * maxDist;
		pmin_z = RayPos_z + RayDir_z * maxDist;
		pmax_x = pmin_x;
		pmax_y = pmin_y;
		pmax_z = pmin_z;
		if( pmin_x > RayPos_x ){ pmin_x = RayPos_x; }
		if( pmin_y > RayPos_y ){ pmin_y = RayPos_y; }
		if( pmin_z > RayPos_z ){ pmin_z = RayPos_z; }
		if( pmax_x < RayPos_x ){ pmax_x = RayPos_x; }
		if( pmax_y < RayPos_y ){ pmax_y = RayPos_y; }
		if( pmax_z < RayPos_z ){ pmax_z = RayPos_z; }

		//E{bNXmŔ
		if( CollideBoxAABB(box_min_x, box_min_y, box_min_z, box_max_x, box_max_y, box_max_z, pmin_x, pmin_y, pmin_z, pmax_x, pmax_y, pmax_z) == FALSE ){
			return false;
		}
	}

	D3DXVECTOR3 pMin(box_min_x, box_min_y, box_min_z);
	D3DXVECTOR3 pMax(box_max_x, box_max_y, box_max_z);
	D3DXVECTOR3 pRayPos(RayPos_x, RayPos_y, RayPos_z);
	D3DXVECTOR3 pRayDir(RayDir_x, RayDir_y, RayDir_z);

	//AABBƃCij̔
	if( D3DXBoxBoundProbe(&pMin, &pMax, &pRayPos, &pRayDir) == TRUE ){
		if( Dist != NULL ){
			float x, y, z;
			float d, mind;

			//AABB̊e_狗ZoāAԋ߂uvƂB
			//@@ƌvZ@

			x = box_min_x - RayPos_x;
			y = box_min_y - RayPos_y;
			z = box_min_z - RayPos_z;
			d = sqrt(x*x + y*y + z*z);
			mind = d;

			x = box_max_x - RayPos_x;
			y = box_min_y - RayPos_y;
			z = box_min_z - RayPos_z;
			d = sqrt(x*x + y*y + z*z);
			if( d < mind ){ mind = d; }

			x = box_min_x - RayPos_x;
			y = box_max_y - RayPos_y;
			z = box_min_z - RayPos_z;
			d = sqrt(x*x + y*y + z*z);
			if( d < mind ){ mind = d; }

			x = box_max_x - RayPos_x;
			y = box_max_y - RayPos_y;
			z = box_min_z - RayPos_z;
			d = sqrt(x*x + y*y + z*z);
			if( d < mind ){ mind = d; }

			x = box_min_x - RayPos_x;
			y = box_min_y - RayPos_y;
			z = box_max_z - RayPos_z;
			d = sqrt(x*x + y*y + z*z);
			if( d < mind ){ mind = d; }

			x = box_max_x - RayPos_x;
			y = box_min_y - RayPos_y;
			z = box_max_z - RayPos_z;
			d = sqrt(x*x + y*y + z*z);
			if( d < mind ){ mind = d; }

			x = box_min_x - RayPos_x;
			y = box_max_y - RayPos_y;
			z = box_max_z - RayPos_z;
			d = sqrt(x*x + y*y + z*z);
			if( d < mind ){ mind = d; }

			x = box_max_x - RayPos_x;
			y = box_max_y - RayPos_y;
			z = box_max_z - RayPos_z;
			d = sqrt(x*x + y*y + z*z);
			if( d < mind ){ mind = d; }

			*Dist = mind;
		}
		return true;
	}

	return false;
}

//! _ƃC̍ŒZ߂
//! @param Pos_x _XW
//! @param Pos_y _YW
//! @param Pos_z _ZW
//! @param RayPos_x C̈ʒuin_jw肷 XW
//! @param RayPos_y C̈ʒuin_jw肷 YW
//! @param RayPos_z C̈ʒuin_jw肷 ZW
//! @param RayDir_x C̃xNgw肷 X
//! @param RayDir_y C̃xNgw肷 Y
//! @param RayDir_z C̃xNgw肷 Z
//! @param maxDist ső勗
//! @return ŒZ
//! @warning RayPosin_j RayDirixNgjԈႦȂƁB
float DistancePosRay(float Pos_x, float Pos_y, float Pos_z, float RayPos_x, float RayPos_y, float RayPos_z, float RayDir_x, float RayDir_y, float RayDir_z, float maxDist)
{
	float x1, y1, z1;
	float x2, y2, z2;
	D3DXVECTOR3 out;

	x1 = Pos_x - RayPos_x;
	y1 = Pos_y - RayPos_y;
	z1 = Pos_z - RayPos_z;
	x2 = RayDir_x * maxDist;
	y2 = RayDir_y * maxDist;
	z2 = RayDir_z * maxDist;

	D3DXVec3Cross(&out, &D3DXVECTOR3(x1, y1, z1), &D3DXVECTOR3(x2, y2, z2));

	return sqrt(out.x*out.x + out.y*out.y + out.z*out.z) / maxDist;
}

//! ϑ_Ώۓ_ւ EpxZo
//! @param pos_x ϑ_XW
//! @param pos_y ϑ_YW
//! @param pos_z ϑ_ZW
//! @param rx ϑ_̐px
//! @param ry ϑ_̐px
//! @param target_x Ώۓ_XW
//! @param target_y Ώۓ_YW
//! @param target_z Ώۓ_ZW
//! @param checkdist 苗i0.0fȉŔ薳j
//! @param out_rx Ώۓ_ւ̐pxi΁`-΁j󂯎|C^iNULLj
//! @param out_ry Ώۓ_ւ̐px󂯎|C^iNULLj
//! @param out_dist2 Ώۓ_ւ̋<b>̓</b>󂯎|C^iNULLj
//! @return Ftrue sFfalse
//! @warning out_dist2͋<b></b>łBKvɉĉ߂ sqrt()֐ ȂǂpĂB
//! @attention  checkdist ɗLȋ^ꍇ́Aϑ_Ώۓ_ւ̋s܂Bw肳ꂽ藣ĂꍇApxvZ false Ԃ܂B
//! @attention tɁA checkdist 0.0fȉ^ꍇAɂ锻s܂B֐͏ true Ԃ܂B
bool CheckTargetAngle(float pos_x, float pos_y, float pos_z, float rx, float ry, float target_x, float target_y, float target_z, float checkdist, float *out_rx, float *out_ry, float *out_dist2)
{
	float x, y, z;
	float dist2 = 0.0f;
	float angleX, angleY;

	x = target_x - pos_x;
	y = target_y - pos_y;
	z = target_z - pos_z;

	if( (checkdist > 0.0f)||(out_dist2 != NULL) ){
		dist2 = (x*x + y*y + z*z);
	}

	if( checkdist > 0.0f ){
		if( dist2 > checkdist * checkdist ){
			return false;
		}
	}

	if( out_rx != NULL ){
		angleX = atan2(z, x) - rx;
		for(; angleX > (float)M_PI; angleX -= (float)M_PI*2){}
		for(; angleX < (float)M_PI*-1; angleX += (float)M_PI*2){}

		*out_rx = angleX;
	}
	if( out_ry != NULL ){
		angleY = atan2(y, sqrt(x*x + z*z))  - ry;

		*out_ry = angleY;
	}
	if( out_dist2 != NULL ){ *out_dist2 = dist2; }

	return true;
}