/**************************************************
OpengateM - MAC address authentication system 
 module for Authentication of User

this file includes the code for main control and pop3/ftp auth.
other authentication protocols are coded in other files.

Copyright (C) 1999 Opengate Project Team
Written by Yoshiaki Watanabe
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Email: watanaby@is.saga-u.ac.jp
**************************************************/

#include "opengatemmng.h"

void onAuthReplyAlarm(int signo);
int authFtp(char *userid, char *passwd);
int AuthFtp(char *userid, char *passwd);
int authPop3(char *userid, char *passwd);
int AuthPop3(char *userid, char *passwd);
int authPam(char *userid, char *passwd);
int AuthPam(char *userid, char *passwd);
int authRadius(char *userid, char *passwd);
int AuthRadius(char *userid, char *passwd);
int authPop3s(char *userid, char *passwd);
int AuthPop3s(char *userid, char *passwd);
int authFtpse(char *userid, char *passwd);
int AuthFtpse(char *userid, char *passwd);
int authFtpsi(char *userid, char *passwd);
int AuthFtpsi(char *userid, char *passwd);
int authLdap(char *userid, char *passwd);
int AuthLdap(char *userid, char *passwd);

/****************************************
get userid from anywhere (cookie/env/postdata)
if not get, send back auth page.
language indicates the one for web description

return value: TRUE(1)=sccess/FALSE(0)=fail 
requestStr: (input) string sended with http-POST
userId,extraId: (output) entered as 'userId@extraId' 
language: (input) html description language(ja/en/..) 
userType: (input) auth as ADMINUSER(1) or NORMALUSER(0) 
cgiName: (input) the cgi to which the auth page sends data
mailDefault: (output) initial value of user mail address
redirectedUrl: (input) url before redirection (return to this after processing)
****************************************/
int getUserId(char* requestStr, char* userId, char* extraId, char* language, int userType, char* cgiName, char* mailDefault, char* redirectedUrl){
  int authResult=DENY;
  char useridfull[USERMAXLN]; /* userid@extraid */
  char cookie[SIDMAXLN];
  char password[USERMAXLN];
  char* docName;

  /* default mail address is null */
  mailDefault[0]='\0';

  /***** try to get userid from db corresponding to http cookie *****/
  switch(userType){
  case NORMALUSER:
    if(GetHttpCookie(cookie, GetConfValue("AuthUserCookie"))){
      if(IsCookieFoundInWorkDb(cookie, userId, extraId, NORMALUSER)){ 
	ConcatUserId(useridfull, userId, extraId);
	GetMailDefaultFromWorkDb(cookie, mailDefault);
	authResult=ACCEPT;
      }
    }
    break;
  case ADMINUSER:
    if(GetHttpCookie(cookie, GetConfValue("AuthAdminCookie"))){
      if(IsCookieFoundInWorkDb(cookie, userId, extraId, ADMINUSER)){ 
	ConcatUserId(useridfull, userId, extraId);
	authResult=ACCEPT;
      }
    }
    break;
  }

  /***** try to get uid from environment variables(shibboleth/httpbasic) *****/
  if(authResult==DENY){

    /* search shibboleth / httpbasic auth settings in conf */
    ResetAuthServerPointer();
    while(SelectNextAuthServer()){

      /* if server setting is not matched to the required usertype, skip it */
      if( (strcmp(GetConfValue("AuthServer/UserType"), "admin")==0)
	&& (userType!=ADMINUSER) ) continue;
      if( (strcmp(GetConfValue("AuthServer/UserType"), "admin")!=0)
	&& (userType==ADMINUSER) ) continue;
       
      /* if the server setting is not shibboleth and not httpbasic, skip it */
      if(strcmp(GetConfValue("AuthServer/Protocol"), "shibboleth")!=0
	 && strcmp(GetConfValue("AuthServer/Protocol"), "httpbasic")!=0) continue;
	
      /* if reached to this line(=shibboleth/httpbasic), get userid from env var */
      if(GetUserIdFromEnv(useridfull)){

	/* split user@extra to user and extra. then search conf extra set */
	SplitId(useridfull, userId, extraId);
	SetupConfExtra(userId, extraId);

	/* if the user is found in accept user list, accept user, else deny */
	/* the user list (defined in conf file) restricts users (if accept all, remove the list) */
	if(IsUserIdFoundInAcceptUsersList(userId)){
	  authResult=ACCEPT;
	  MakeMailDefault(userId, extraId, mailDefault);
	}
	else{
	  SetMessage(NoInfoInDb); 
	  PutDenyToClient(language);
	  err_msg("DENY: user %s", useridfull);
	  authResult=DENY;
	  return FALSE;
	}
      }
    }
  }

  /***** try to get userid from request string *****/
  if(authResult==DENY){

    /* if not get, in request string, send back auth page */
    if(!GetUserIdFromPostData(requestStr, useridfull, password)){

      /* split user@extra to user and extra, then search conf extra set */
      SplitId(useridfull, userId, extraId);
      SetupConfExtra(userId, extraId);
      
      /* select document content */
      if(userType==ADMINUSER) docName=GetConfValue("AuthAdminDoc");
      else docName=GetConfValue("AuthDoc");

      /* put page and exit */
      PutAuthRequestPageToClient(language, cgiName, docName, redirectedUrl);
      return FALSE;
    }

    /* reaching this line means that userid is found in request string */
    /* split user@extra to user and extra. then search conf extra set */
    SplitId(useridfull, userId, extraId);
    SetupConfExtra(userId, extraId);

    /* check user by authenticate servers */
    ResetAuthServerPointer();
    while(SelectNextAuthServer()){
      
      /* if check normal user and admin auth server, skip the server */
      /* if check admin user and not admin auth server, skip the server */
      if(userType==NORMALUSER 
	 && strcmp(GetConfValue("AuthServer/UserType"), "admin")==0) continue;
      if(userType==ADMINUSER
	 && strcmp(GetConfValue("AuthServer/UserType"), "admin")!=0) continue;

      /* authenticate the user with auth server. if deny, goto next server */
      if((authResult=AuthenticateUser(userId, password))==DENY) continue;

      /* if userid is not found in the user list in conf, goto next server */
      /* the user list (defined in conf file) restricts users (if accept all, remove the list) */
       if(!IsUserIdFoundInAcceptUsersList(userId)){
	authResult=DENY;
	continue;
      }

      /* if accepted, set mail default(used for warning mail) and exit loop */
      if(authResult==ACCEPT){
	MakeMailDefault(userId, extraId, mailDefault);
	break;
      }
    }

    /* if all check is failed, put error */
    if(authResult==DENY){
      SetMessage(NoInfoInDb);
      if(userType==ADMINUSER) SetMessage(RequestAdminAuth); 
      PutDenyToClient(language);
      err_msg("DENY: user %s", useridfull);
    }
  }
  if(authResult==ACCEPT) return TRUE;
  else return FALSE;
}

/**************************************
 if accept users are listed in conf file,
  find the userid in the list
 if no list is indicated, return true
**************************************/
int isUserIdFoundInAcceptUsersList(char* userId){
  char usersList[BUFFMAXLN];
  char userIdPattern[WORDMAXLN];

  /* get accept users list. if not exist, return true(accept all users) */
  strlcpy(usersList,GetConfValue("AuthServer/AcceptUsers"),BUFFMAXLN);
  if(isNull(usersList)) return TRUE;

  /* if userid is found in the usersList, return true */
  /* example of usersList is [user1 user2 user3 user4] */

  /* regular expression matched to "(^| )userid( |$)" */
  /* it means that [(head or space) userid-string (space or tail)] */
 /* last-arg 0 of RegExMatch means ignore-case */
  strlcpy(userIdPattern, "(^| )", WORDMAXLN);
  strlcat(userIdPattern, userId, WORDMAXLN);
  strlcat(userIdPattern, "( |$)", WORDMAXLN);
  return RegExMatch(usersList, userIdPattern, 0);
}


/*******************************************************/
/* Authenticate user by accessing to ftp server        */
/*  userid : user to auth (short form(omit @extraid))  */
/*  password : password for the user                   */
/*******************************************************/
int authenticateUser(char *userid, char *passwd)
{
  char* proto;
  int authResult=DENY;
  int timeout;

  /* get timeout value of authserver reply from conf */
  timeout=atoi(GetConfValue("AuthServer/Timeout"));

  /* set auth server reply timeout */
  if(timeout>0){
    AddAlarm("AuthReplyAlarm", timeout, TRUE, onAuthReplyAlarm); 
    EnableAlarm();
  }

  /* get authentication protocol from conf */
  proto=GetConfValue("AuthServer/Protocol");

  /* authenticate by ftp access */
  if(strcmp(proto, "ftp")==0){
    authResult=AuthFtp(userid, passwd);
  }

  /* authenticate by pop3 access */
  else if(strcmp(proto, "pop3")==0){
    authResult=AuthPop3(userid, passwd);
  }

  /* authenticate by pam */
  else if(strcmp(proto, "pam")==0){
    authResult=AuthPam(userid, passwd);
  }

  /* authenticate by radius */
  else if(strcmp(proto, "radius")==0){
    authResult=AuthRadius(userid, passwd);
  }

  /* authenticate by pop3s */
  else if(strcmp(proto, "pop3s")==0){
    authResult=AuthPop3s(userid, passwd);
  }

  /* authenticate by ldap */
  else if(strcmp(proto, "ldap")==0){
    authResult=AuthLdap(userid, passwd);
  }

  /* authenticate by ftps-explicit */
  else if(strcmp(proto, "ftpse")==0){
    authResult=AuthFtpse(userid, passwd);
  }

  /* authenticate by ftps-implicit */
  else if(strcmp(proto, "ftpsi")==0){
    authResult=AuthFtpsi(userid, passwd);
  }

  /* authenticate by shibboleth */
  /* in this case, auth request cannot reach to this line */
  else if(strcmp(proto, "shibboleth")==0){
    authResult=DENY;
  }

  /* authenticate by httpbasic */
  /* in this case, auth request cannot reach to this line */
  else if(strcmp(proto, "httpbasic")==0){
    authResult=DENY;
  }
  
  /* deny all users */
  else if(strcmp(proto, "deny")==0){
    authResult=DENY;
  }

  /* accept all users */
  else if(strcmp(proto, "accept")==0){
    authResult=ACCEPT;
  }

  /* show splash page and auth is passed */
  else if(strcmp(proto, "splash")==0){
    authResult=ACCEPT;
  }

  /* unknown request */
  else{
    err_msg("ERR at %s#%d: Unknown protocol:%s",__FILE__,__LINE__,proto);
    authResult=DENY;
  }

  /* stop the auth reply timeout */
  if(timeout>0) RemoveAlarm("AuthReplyAlarm");
  return authResult;
}

/*****************************/
/* split userid and extraId */
/*****************************/
void splitId(char* useridfull, char* userId, char* extraId)
{
  char useridSave[USERMAXLN];

  char* markPnt=NULL;

  strlcpy(useridSave, useridfull, USERMAXLN);

  /* separate extraId from userid (UserIdSeparator is '@') */
  markPnt=strchr(useridSave, *GetConfValue("UserIdSeparator"));
  if(markPnt==NULL){  
    /* separator mark is not found. no extraId */
    strlcpy(extraId,"",USERMAXLN);
    strlcpy(userId,useridSave,USERMAXLN);
  }else{
    /* separator is found. pick up extraId */
    *markPnt='\0';
    strlcpy(extraId,markPnt+1,USERMAXLN);
    strlcpy(userId,useridSave,USERMAXLN);
  }
} 

/*************************
concatinate userid and extraid
*************************/
char* concatUserId(char* useridfull, char* userId, char* extraId){

  /* set full userid */
  strlcpy(useridfull, userId,USERMAXLN);
  if(!isNull(extraId)){
    strlcat(useridfull, GetConfValue("UserIdSeparator"), USERMAXLN);
    strlcat(useridfull, extraId, USERMAXLN);
  }
  return useridfull;
}

/***********************/
/* Authenticate by FTP */
/***********************/
int authFtp(char *userid, char *passwd)
{
  int		sockfd, n;
  char		recvline[BUFFMAXLN];
  int           authResult;
  char* serverAddr; /* auth server address */
  char* port;      /* auth server port */

  /* get auth server address */
  serverAddr=GetConfValue("AuthServer/Address");

  if(isNull(serverAddr)){
    err_msg("ERR at %s#%d: Missing address for FTP server in config",
	    __FILE__,__LINE__);
    return DENY;
  }

  /* get auth server port */
  port=GetConfValue("AuthServer/Port");

  /* FTP server connect (if port is not defined in conf, default is used) */
  if(isNull(port)){
    sockfd = Tcp_connect(serverAddr, "ftp");
  }else{
    sockfd = Tcp_connect(serverAddr, port);
  }
  if(sockfd<0){
    err_msg("ERR at %s#%d: Ftp server is not normal 0",__FILE__,__LINE__);
    return DENY;
  }

  /* get [220 <host> FTP server ..] */
  if((n = readln(sockfd, recvline, BUFFMAXLN)) < 0) {
    err_msg("ERR at %s#%d: Ftp server is not normal 1",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }
  if(strstr(recvline,"220")!=recvline){
    err_msg("ERR at %s#%d: Ftp server is not normal 2",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }

  /* put [user <userid>] */
  Writefmt(sockfd, "user %s\r\n", userid);

  /* get [331 Password required ..] (greeting lines might exist before 331 request) */
  if((n = readln(sockfd, recvline, BUFFMAXLN)) < 0) {
    err_msg("ERR at %s#%d: Ftp server is not normal 3",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }

  /* if multi-line greeting [220 ...] exist, skip them. */
  while(strstr(recvline,"220")==recvline){
    if((n = readln(sockfd, recvline, BUFFMAXLN)) < 0) {
      err_msg("ERR at %s#%d: Ftp server is not normal 3",__FILE__,__LINE__);
      Close(sockfd);
      return DENY;
    }
  }

  /* check [331 Password required ..] */
  if(strstr(recvline,"331")!=recvline){
    err_msg("ERR at %s#%d: Ftp server is not normal 4",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }

  /* put [pass <password>] */
  Writefmt(sockfd, "pass %s\r\n", passwd);

  /* get [230 User <userid> logged in] (the string means correct userid/passwd) */
  if((n = readln(sockfd, recvline, BUFFMAXLN)) < 0) {
    err_msg("ERR at %s#%d: Ftp server is not normal 5",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }
  if(strstr(recvline,"230")==recvline){
    authResult=ACCEPT;
  }else{
    authResult=DENY;
  }
  
  /* put [quit] */
  Writefmt(sockfd,"quit\r\n");

  Close(sockfd);
  return authResult;
}

/************************/
/* Authenticate by POP3 */
/************************/
int authPop3(char *userid, char *passwd)
{
  int		sockfd, n;
  char		recvline[BUFFMAXLN];
  int           authResult;
  char*  serverAddr;
  char* port;

  /* get auth server address */
  serverAddr=GetConfValue("AuthServer/Address");

  if(isNull(serverAddr)){
    err_msg("ERR at %s#%d: Missing address for POP3 server in config",
	    __FILE__,__LINE__);
    return DENY;
  }

  /* get auth server port */
  port=GetConfValue("AuthServer/Port");

  /* POP3 server connect */
  if(isNull(port)){
    sockfd = Tcp_connect(serverAddr, "pop3");
  }else{
    sockfd = Tcp_connect(serverAddr, port);
  }
  if(sockfd<0){
    err_msg("ERR at %s#%d: Pop3 server is not normal 0",__FILE__,__LINE__);
    return DENY;
  }

  /* get [+OK POP3 <host> <ver> server ready]*/
  if((n = readln(sockfd, recvline, BUFFMAXLN)) < 0) {
    err_msg("ERR at %s#%d: Pop3 server is not normal 1",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }
  if(strstr(recvline,"+OK")!=recvline){
    err_msg("ERR at %s#%d: Pop3 server is not normal 2",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }

  /* put [user <userid>] */
  Writefmt(sockfd, "user %s\r\n", userid);

  /* get [+OK User name accepted, password please] */
  if((n = readln(sockfd, recvline, BUFFMAXLN)) < 0) {
    err_msg("ERR at %s#%d: Pop3 server is not normal 3",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }
  if(strstr(recvline,"+OK")!=recvline){
    err_msg("ERR at %s#%d: Pop3 server is not normal 4",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }

  /* put [pass <password>] */
  Writefmt(sockfd, "pass %s\r\n", passwd);

  /* get [+OK Mailbox open, <count> messages] */
  if((n = readln(sockfd, recvline, BUFFMAXLN)) < 0) {
    err_msg("ERR at %s#%d: Pop3 server is not normal 5",__FILE__,__LINE__);
    Close(sockfd);
    return DENY;
  }
  if(strstr(recvline,"+OK")==recvline){
    authResult=ACCEPT;
  }else{
    authResult=DENY;
  }
  
  /* put [quit] */
  Writefmt(sockfd,"quit\r\n");

  Close(sockfd);
  return authResult;
}

/***************************************/
/* called at auth reply timeout        */
/* on timeout, read wait is cleared    */
/***************************************/
void onAuthReplyAlarm(int signo)
{
}


/***********************************
check duplication of shibboleth / httpbasic auth server
shibboleth/httpbasic cannot be used for both usertype (admin and normal)
***********************************/
int isShibOrBasicAuthDuplicated(void){
  int nShib=0;
  int nBasic=0;

  /* scan config file to count shibboleth/httpbasic auth server setting */
  ResetAuthServerPointer();
  while(SelectNextAuthServer()){
    if(strcmp(GetConfValue("AuthServer/Protocol"), "shibboleth")==0) nShib++;
    if(strcmp(GetConfValue("AuthServer/Protocol"), "httpbasic")==0) nBasic++;
  }
  ResetAuthServerPointer();

  /* if duplicated, put error */
  if(nShib>1 || nBasic>1){
    err_msg("ERR at %s#%d: Shibboleth or HttpBasic auth cannot duplicate",__FILE__,__LINE__);
    return TRUE;
  }else{
    return FALSE;
  }
}

/**************************************
setup default mail address
made from conf value at MailAttribute or MailDomain
**************************************/
int makeMailDefault(char* userId, char* extraId, char* mailDefault){

  char* pMailAttribute=NULL;
  char* pMailDomain=NULL;
  char* pMailAddr=NULL;

  /* set default null */
  mailDefault[0]='\0';

  /* get conf values for mail domain */
  pMailAttribute=GetConfValue("AuthServer/MailAttribute");
  pMailDomain=GetConfValue("AuthServer/MailDomain");
  if(isNull(pMailAttribute) && isNull(pMailDomain)){
      err_msg("ERR at %s#%d: Can't get mail attibute/domain setting from conf file.",__FILE__,__LINE__);
    return FALSE;
  }

  /* if set MailAttribute, mail from the env variable set in MailAttribute */
  /* (Shibboleth puts the user mail address into an env variable) */
  if(!isNull(pMailAttribute)){
    pMailAddr=getenvEx(pMailAttribute,TRUE,FALSE); /* var list is permitted */
    if(!isNull(pMailAddr)){
      strlcpy(mailDefault, pMailAddr, BUFFMAXLN);
      return TRUE;
    }else{
      err_msg("ERR at %s#%d: Can't get mail attibute from environment variable.",__FILE__,__LINE__);
      return FALSE;
    }
  }

  /* if set MailDomain, mail from userid+@+MailDomain */
  if(!isNull(pMailDomain)){
    strlcpy(mailDefault, userId, BUFFMAXLN);
    strlcat(mailDefault, "@", BUFFMAXLN);
    strlcat(mailDefault, pMailDomain, BUFFMAXLN);
    return TRUE;
  }else{
      err_msg("ERR at %s#%d: Can't get mail domain string from conf file.",__FILE__,__LINE__);
    return FALSE;
  }

  err_msg("ERR at %s#%d: Can't set mail. Check AuthServer/MailAttribute or AuthServer/MailDomain in conf.",__FILE__,__LINE__);
  return FALSE;  
}

/**********************************
is AuthServer setting for normal user 'splash'
**********************************/
int isSplashOnlyManagement(void){
  int splash=FALSE;

  /* search AuthServer setting having Protocol=splash and UserType!=admin */
  ResetAuthServerPointer();
  while(SelectNextAuthServer()){
    if((strcmp(GetConfValue("AuthServer/Protocol"), "splash")==0)&&
       (strcmp(GetConfValue("AuthServer/UserType"), "admin")!=0) ){
      splash=TRUE;
      break;
    }
  }
  ResetAuthServerPointer();
  return splash;
}
  
/**********************************/
/**********************************/
int GetUserId(char* requestStr, char* userId, char* extraId, char* language, int userType, char* cgiName, char* mailDefault, char* redirectedUrl){
  int ret;
  if(debug>1) err_msg("DEBUG:=>getUserId(..,%s,%d)",language,userType);
  ret=getUserId(requestStr, userId, extraId, language, userType, cgiName, mailDefault, redirectedUrl);
  if(debug>1) err_msg("DEBUG:(%d)<=getUserId(%s,%s,%s)",ret,userId,extraId,mailDefault);
  return ret;
}

int AuthFtp(char *userid, char *passwd)
{
  int ret;

  if(debug>1) err_msg("DEBUG:=>authFtp(%s,passwd)",userid);
  ret=authFtp(userid,passwd);
  if(debug>1) err_msg("DEBUG:(%d)<=authFtp( )",ret);

  return ret;
}


int AuthPop3(char *userid, char *passwd)
{
  int ret;

  if(debug>1) err_msg("DEBUG:=>authPop3(%s,passwd)",userid);
  ret=authPop3(userid,passwd);
  if(debug>1) err_msg("DEBUG:(%d)<=authPop3( )",ret);

  return ret;
}

int AuthenticateUser(char *userid, char *password)
{
  int ret;

  if(debug>1) err_msg("DEBUG:=>authenticateUser(%s,passwd)",userid);
  ret=authenticateUser(userid,password);
  if(debug>1) err_msg("DEBUG:(%d)<=authenticateUser( )",ret);

  return ret;
}

void SplitId(char* userid, char* useridshort, char* extraId)
{
  if(debug>1) err_msg("DEBUG:=>splitId(%s,,)",userid);
  splitId(userid,useridshort,extraId);
  if(debug>1) err_msg("DEBUG:<=splitId(,%s,%s)",useridshort,extraId);
}

char* ConcatUserId(char*  useridfull, char* userId, char* extraId)
{
  char* ret;
  if(debug>1) err_msg("DEBUG:=>concatUserId(,%s,%s)",userId,extraId);
  ret=concatUserId(useridfull, userId, extraId);
  if(debug>1) err_msg("DEBUG:<=concatUserId(%s,,)",useridfull);
  return ret;
}

int IsUserIdFoundInAcceptUsersList(char* userId){
  int ret;

  if(debug>1) err_msg("DEBUG:=>isUserIdFoundInAcceptUsersList(%s)",userId);
  ret=isUserIdFoundInAcceptUsersList(userId);
  if(debug>1) err_msg("DEBUG:(%d)<=isUserIdFoundInAcceptUsersList( )",ret);

  return ret;
}

int IsShibOrBasicAuthDuplicated(void){
  int ret;

  if(debug>1) err_msg("DEBUG:=>isShibOrBasicAuthDuplicated( )");
  ret=isShibOrBasicAuthDuplicated();
  if(debug>1) err_msg("DEBUG:(%d)<=isShibOrBasicAuthDuplicated( )",ret);

  return ret;
}

int MakeMailDefault(char* userId, char* extraId, char* mailDefault){
  int ret;

  if(debug>1) err_msg("DEBUG:=>makeMailDefault(%s,%s,)",userId, extraId);
  ret=makeMailDefault(userId, extraId,mailDefault);
  if(debug>1) err_msg("DEBUG:(%d)<=makeMailDefault(,,%s)",ret,mailDefault);

  return ret;
}

int IsSplashOnlyManagement(void){
  int ret;

  if(debug>1) err_msg("DEBUG:=>isSplashOnlyManagement()");
  ret=isSplashOnlyManagement();
  if(debug>1) err_msg("DEBUG:(%d)<=isSplashOnlyManagement",ret);

  return ret;
}
