/*
 * The MIT License (MIT)
 * 
 * Copyright (c) 2014 Yuki SAKAI
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 ******************************************************************************/

/*******************************************************************************
 * Include ********************************************************************/
#if !defined(REMPIC_H__)
#  include <rempic.h>
#endif
#if !defined(_STDIO_H_) && !defined(_STDIO_H)
#  include <stdio.h>
#endif
#if !defined(_STRING_H_) && !defined(_STRING_H)
#  include <string.h>
#endif
#if !defined(_UNISTD_H_) && !defined(_UNISTD_H)
#  include <unistd.h>
#endif
#if !defined(_ERRNO_H_) && !defined(_ERRNO_H)
#  include <errno.h>
#endif
#if !defined(_SYS_TYPES_H_) && !defined(_SYS_TYPES_H)
#  include <sys/types.h>
#endif
#if !defined(_SYS_STAT_H_) && !defined(_SYS_STAT_H)
#  include <sys/stat.h>
#endif
#if !defined(_SYS_FILE_H_) && !defined(_SYS_FILE_H)
#  include <sys/file.h>
#endif
#if !defined(_FCNTL_H_) && !defined(_FCNTL_H)
#  include <fcntl.h>
#endif


/*******************************************************************************
 * read file ******************************************************************/
static int read_file (unsigned char *path, picInfo *pinfo)
{
  int fd;

  if ((fd=open ((char *)path, O_RDONLY))==-1) {return 0;}
  else if (flock (fd, LOCK_SH)==-1)           {close (fd); return 0;}

  if      (FLGCHK(pinfo->ft, GEXTJPG))  {jpeg_chk (fd, pinfo);}
  else if (FLGCHK(pinfo->ft, GEXTJPEG)) {jpeg_chk (fd, pinfo);}
  else if (FLGCHK(pinfo->ft, GEXTPNG))  {png_chk (fd, pinfo); }
  else if (FLGCHK(pinfo->ft, GEXTGIF))  {gif_chk (fd, pinfo); }

  flock (fd, LOCK_UN);
  close (fd);

  return 1;
}

/*******************************************************************************
 * file extension chk *********************************************************/
static unsigned int file_extchk (unsigned char *path)
{
  unsigned char *pext;
  unsigned char ext[10];
  (void) memset ((char *)ext, 0, 10);

  for (pext = path + strlen ((char *)path) - 1; pext!=path; pext--) {
    if (*pext=='.') {break;}
  }
  if      (*pext!='.')                  {return GEXTERROR;}
  else if (strlen ((char *)(pext+1))>9) {return GEXTERROR;}
  strncpy ((char *)ext, (char *)(pext+1), 10);
  return ext_chk (ext);
}

/*******************************************************************************
 * file size check ************************************************************/
static int fsize_chk (off_t size)
{
  if (FLGCHK(flags, FLGRMUPPERSIZE) && FLGCHK(flags, FLGRMLOWERSIZE)) {
    if (UpperSize<=LowerSize) {
      if (UpperSize>=size)      {return RMCHKREMOVE;}
      else if (LowerSize<=size) {return RMCHKREMOVE;}
    } else {
      if (UpperSize>=size && LowerSize<=size) {return RMCHKREMOVE;}
    }
  } else if (FLGCHK(flags, FLGRMUPPERSIZE)) {
    if (UpperSize>=size) {return RMCHKREMOVE;}
  } else if (FLGCHK(flags, FLGRMLOWERSIZE)) {
    if (LowerSize<=size) {return RMCHKREMOVE;}
  }

  return RMCHKINIT;
}

/*******************************************************************************
 * file X check ***************************************************************/
static int fx_chk (unsigned long img_x)
{
  if (FLGCHK(flags, FLGRMUPPERX) && FLGCHK(flags, FLGRMLOWERX)) {
    if (UpperX<=LowerX) {
      if      (UpperX>=img_x) {return RMCHKREMOVE;}
      else if (LowerX<=img_x) {return RMCHKREMOVE;}
    } else {
      if (UpperX>=img_x && LowerX<=img_x) {return RMCHKREMOVE;}
    }
  } else if (FLGCHK(flags, FLGRMUPPERX)) {
    if (UpperX>=img_x) {return RMCHKREMOVE;}
  } else if (FLGCHK(flags, FLGRMLOWERX)) {
    if (LowerX<=img_x) {return RMCHKREMOVE;}
  }

  return RMCHKINIT;
}

/*******************************************************************************
 * file Y check ***************************************************************/
static int fy_chk (unsigned long img_y)
{
  if (FLGCHK(flags, FLGRMUPPERY) && FLGCHK(flags, FLGRMLOWERY)) {
    if (UpperY<=LowerY) {
      if      (UpperY>=img_y) {return RMCHKREMOVE;}
      else if (LowerY<=img_y) {return RMCHKREMOVE;}
    } else {
      if (UpperY>=img_y && LowerY<=img_y) {return RMCHKREMOVE;}
    }
  } else if (FLGCHK(flags, FLGRMUPPERY)) {
    if (UpperY>=img_y) {return RMCHKREMOVE;}
  } else if (FLGCHK(flags, FLGRMLOWERY)) {
    if (LowerY<=img_y) {return RMCHKREMOVE;}
  }

  return RMCHKINIT;
}

/*******************************************************************************
 * Remove check ***************************************************************/
int remove_chk (spname finfo)
{
  /*****************************************************************************
   * if determine remove process(chk!=0), jump remove process ******************
   ****************************************************************************/
  unsigned int ft;
  int          chk;
  picInfo      pinfo;

  chk = RMCHKINIT;
  ft  = GEXTERROR;
  (void) memset ((char *)&pinfo, 0, sizeof (picInfo));
  pinfo.fsize = finfo.sb.st_size;
  /*****************************************************************************
   * Analyse filepath name extension ******************************************/
  if ((pinfo.ft=file_extchk (finfo.path))==GEXTERROR) {chk = RMCHKSAVE; goto rm_proc;}
  else if ((pinfo.ft&TargetFileType)==0)              {chk = RMCHKSAVE; goto rm_proc;}
  /*****************************************************************************
   * File if check ************************************************************/
  if ((chk=fsize_chk (pinfo.fsize))!=RMCHKINIT)        {goto rm_proc;} /* file size  */
  else if (!read_file (finfo.path, &pinfo))            {return 0;}     /* read file i*/
  else if (pinfo.fm) {
    if (pinfo.fm==2)                         {chk = RMCHKSAVE;  }
    else if (FLGCHK(flags, FLGRMPICBADFILE)) {chk = RMCHKREMOVE;}
    else                                     {chk = RMCHKSAVE;  }
    goto rm_proc;
  }
  else if (FLGCHK(flags, FLGRMPICBADFILE) && pinfo.fm) {chk = RMCHKSAVE; goto rm_proc;}
  else if ((chk=fx_chk (pinfo.img_x))!=RMCHKINIT)      {goto rm_proc;} /* file size  */
  else if ((chk=fy_chk (pinfo.img_y))!=RMCHKINIT)      {goto rm_proc;} /* file size  */
  
  chk = RMCHKSAVE;
  /*****************************************************************************
   * Remove Process ***********************************************************/
 rm_proc:
  if (FLGCHK(flags, FLGDEBUGMODE)) {
    /* not remove */
    if (chk==RMCHKREMOVE)    {fprintf (debugout, "remove: %s\n"  , finfo.path);}
    else if (chk==RMCHKSAVE) {fprintf (debugout, "save: %s\n"    , finfo.path);}
    else                     {fprintf (debugout, "no check: %s\n", finfo.path);}
  } else {
    if (chk==RMCHKREMOVE && 1) {
      errno = 0;
      if (unlink ((char *)finfo.path)<0) {
	switch (errno) {
	case ENOTDIR     : fprintf (patherr, "%s: path is include not directory and file name\n", finfo.path); break;
	case ENAMETOOLONG: fprintf (patherr, "%s: path name is long\n"                          , finfo.path); break;
	case ENOENT      : fprintf (patherr, "%s: no such file\n"                               , finfo.path); break;
	case EACCES      : fprintf (patherr, "%s: Can't access\n"                               , finfo.path); break;
	case ELOOP       : fprintf (patherr, "%s: Many symbolic link\n"                         , finfo.path); break;
	case EPERM       : fprintf (patherr, "%s: Don't permit remove\n"                        , finfo.path); break;
	case EBUSY       : fprintf (patherr, "%s: Can't remove for file is busy\n"              , finfo.path); break;
	case EIO         : fprintf (patherr, "%s: I/O error\n"                                  , finfo.path); break;
	case EROFS       : fprintf (patherr, "%s: file is read only\n"                          , finfo.path); break;
	case EFAULT      : fprintf (patherr, "%s: Pointer error\n"                              , finfo.path); break;
	}
	fprintf (stdout, "can't rm: %s\n"  , finfo.path);
      } else {fprintf (stdout, "remove: %s\n"  , finfo.path);}
    } else {fprintf (stdout, "save  : %s\n"  , finfo.path);}
  }
  return 1;
}




