package hayashi.kiban;
import hayashi.osm.api.Command;

import javax.xml.parsers.*;
import org.w3c.dom.*;
import org.xml.sax.*;
import java.io.*;
import java.util.Iterator;
import java.util.TreeMap;

/**
 * ====基盤地図情報のダウンロード====
 *
 * まず最初に、基盤地図情報をダウンロードします。
 *
 * 国土地理院の基盤地図情報ダウンロードページから、ログイン画面へ進みログインして、「基盤地図情報 縮尺レベル25000 JPGIS形式」を選択し、対象の地域のファイルをダウンロードします。(「JPGIS(GML)形式」ではありません)
 * http://fgd.gsi.go.jp/download/
 *
 * 対象地域と項目の選択方法は、ダウンロードページの指示に従ってください。
 *
 * 「基盤地図情報項目」では「行政区画の境界線及び代表点」を選択してください。
 * 		「FG-JPS-39-05-Z001.zip」というような「FG-JPS-xx-05-Z001.zip」という形式のZIPファイルがダウンロードされます。
 *
 * ダウンロードした圧縮ファイルを解凍しておきます。
 * 解凍すると「FG-JPS-44209-CommBdry-20091109-0001.xml」というような「FG-JPS-xxxx-CommBdry-xxxxxxxx-xxxx.xml」という形式のXMLファイルが出来上がります。
 * @author yuu
 *
 */
public class ConvAdmPt25000 {
	File inputFile;
	String admCodeStr = "";
	int iCounter = 0;
	boolean link = true;
	TreeMap<String, AdmPt> admMap = null;

	/**
	 *  メイン
	 *
	 * <GI>
	 * <dataset>
	 *
	 * <AdmPt id="K6_1" uuid="fgoid:10-00150-11-1002-1530">
		 <fid>fgoid:10-00150-11-1002-1530</fid>
		 <lfSpanFr>
		  <jps:position>
		   <jps:date8601>2008-03-31</jps:date8601>
		  </jps:position>
		 </lfSpanFr>
		 <lfSpanTo>
		  <jps:position>
		   <jps:date8601>2999-12-31</jps:date8601>
		  </jps:position>
		 </lfSpanTo>

		<devDate>
		  <jps:position>
		    <jps:date8601>2011-01-31</jps:date8601>
		  </jps:position>
		</devDate>

		<orgGILvl>25000</orgGILvl>
		<vis>表示</vis>

		<pos id="K6_2" uuid="fgoid:10-00150-11-1002-1530-g">
		  <jps:CRS uuidref="fguuid:jgd2000.bl"/>
		  <jps:position>
		    <jps:coordinate>37.055674 139.338021</jps:coordinate>
		  </jps:position>
		</pos>

		<type>その他</type>
		<name>福島県南会津郡檜枝岐村</name>
		<admCode>07364</admCode>

		</AdmPt>

		</dataset>

		</GI>

	 */
	public static void main(String[] argv)
	{
		boolean link = true;
		for (String arg : argv) {
			if (arg.equals("-nolink")) {
				link = false;
			}
		}

		File dir = new File(".");
		File[] files = dir.listFiles();
		int counter = 0;
		for (File iFile : files) {
			if (checkFile(iFile)) {
				counter++;
				ConvAdmPt25000 t = new ConvAdmPt25000(iFile, link);
				t.runner();
			}
		}
		System.out.println("["+ counter +"]つのファイルを処理しました。");
	}

	public ConvAdmPt25000 (File iFile, boolean link) {
		this.inputFile = iFile;
		this.link = link;
	}

	public void runner() {
		DocumentBuilderFactory factory;
		DocumentBuilder        builder;
		Node root;

		this.iCounter = 0;
		String iStr = this.inputFile.getName();
		String[] nameAry = iStr.split("-");
		String fileName = "";
		for (int i=0; i < nameAry.length; i++) {
			if (i != 0) {
				fileName = fileName + "-";
			}
			fileName = fileName + nameAry[i];
			if (i == 2) {
				this.admCodeStr = nameAry[2];
			}
		}

		File outputFile = new File(iStr.substring(0, iStr.length() - 4) +".gpx");
		BufferedWriter w = null;

		File htmlFile = new File(iStr.substring(0, iStr.length() - 4) +".html");
		BufferedWriter hw = null;

		try {
			w = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(outputFile), "UTF-8"));
			w.write("<?xml version=\"1.0\" encoding=\"UTF-8\" ?>");
			w.newLine();
			w.write("<gpx xmlns=\"http://www.topografix.com/GPX/1/1\" version=\"1.1\" creator=\"osmtracker-android\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.topografix.com/GPX/1/1 http://www.topografix.com/GPX/1/1/gpx.xsd \">");
			w.newLine();

			System.out.println("=== "+ admCodeStr +": "+ Command.getCodeName(Integer.parseInt(admCodeStr)) +" ===");
			System.out.println("{| class=\"wikitable sortable\"");

			System.out.println("|- ");
			System.out.println("! style=\"width:70px\" | admCode");
			System.out.println("! | name");
			System.out.println("! style=\"width:70px\" | type");
			System.out.println("! | 備考/最終更新日");


			hw = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(htmlFile), "UTF-8"));
			hw.write("<!DOCTYPE html>");
			hw.newLine();
			hw.write("<html><head><meta charset=\"utf-8\" /></head>");
			hw.newLine();
			hw.write("<body><table border='1'>");
			hw.newLine();
			hw.write("<tr>");
			hw.write("<td>admCode</td>");
			hw.write("<td>name</td>");
			hw.write("<td></td>");
			hw.write("</tr>");
			hw.newLine();

			factory = DocumentBuilderFactory.newInstance();
			builder = factory.newDocumentBuilder();
			factory.setIgnoringElementContentWhitespace(true);
			factory.setIgnoringComments(true);
			factory.setValidating(true);
			root    = builder.parse(inputFile.getName());

			//---------------------------------------
			admMap = new TreeMap<String, AdmPt>();
			showNodes(root, "");
			//---------------------------------------

			Iterator<String> it = admMap.keySet().iterator();
	        while (it.hasNext()) {
	            String code = (String) it.next();
	            AdmPt adm = admMap.get(code);

				System.out.println("|- ");
				System.out.println("| " + adm.code);
				System.out.println("| " + adm.name);
				System.out.println("| " + adm.type);
				System.out.println("| ");

				hw.write("<tr>");
				hw.write("<td>"+ adm.code +"</td>");
				hw.write("<td>"+ adm.name +"</td>");
				hw.write("<td>"+ adm.type +"</td>");
				hw.write("</tr>");
				hw.newLine();

				w.write("<wpt lat=\""+ adm.latStr +"\" lon=\""+ adm.lonStr +"\">");
				w.newLine();
				w.write(" <time>"+ adm.dateStr +"T00:00:00Z</time>");
				w.newLine();
				w.write(" <name><![CDATA["+ adm.name +"]]></name>");
				w.newLine();
				w.write("</wpt>");
				w.newLine();
	        }

			w.write("</gpx>");
			w.newLine();

			hw.write("</table></body></html>");
			hw.newLine();

			System.out.println("|}");

		} catch (ParserConfigurationException e0) {
			System.out.println(e0.getMessage());
		} catch (SAXException e1){
			System.out.println(e1.getMessage());
		} catch (IOException e2) {
			System.out.println(e2.getMessage());
		}
		finally {
			if (w != null) {
				try {
					w.close();
					hw.close();
				}
				catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
	}

	public void showNodes(Node node, String space) throws IOException {
		NodeList nodes = node.getChildNodes();
		for (int i=0; i<nodes.getLength(); i++) {
			Node node2 = nodes.item(i);
			if (node2.getNodeName().equals("AdmPt")) {
				showAdmPt(node2);
			}
			else {
				showNodes(node2, space + "    ");
			}
		}
	}

	public void showAdmPt(Node node) throws IOException {
		String positionStr = "";
		String typeStr = "";
		String nameStr = "";
		String latStr = "";
		String lonStr = "";
		String dateStr = "";
		String codeStr = "";

		NodeList nodes = node.getChildNodes();
		for (int i=0; i < nodes.getLength(); i++) {
			Node node2 = nodes.item(i);
			if (node2.getNodeName().equals("admCode")) {
				codeStr = node2.getTextContent();
			}
			else if (node2.getNodeName().equals("pos")) {
				NodeList nodes3 = node2.getChildNodes();
				for (int j=0; j < nodes3.getLength(); j++) {
					Node node3 = nodes3.item(j);
					if (node3.getNodeName().equals("jps:position")) {
						NodeList nodes4 = node3.getChildNodes();
						for (int k=0; k < nodes4.getLength(); k++) {
							Node node4 = nodes4.item(k);
							if (node4.getNodeName().equals("jps:coordinate")) {
								positionStr = node4.getTextContent();
								String[] str4Ary = positionStr.split(" ");
								latStr = str4Ary[0];
								lonStr = str4Ary[1];
							}
						}
					}
				}
			}
			else if (node2.getNodeName().equals("devDate")) {
				NodeList nodes3 = node2.getChildNodes();
				for (int j=0; j < nodes3.getLength(); j++) {
					Node node3 = nodes3.item(j);
					if (node3.getNodeName().equals("jps:position")) {
						NodeList nodes4 = node3.getChildNodes();
						for (int k=0; k < nodes4.getLength(); k++) {
							Node node4 = nodes4.item(k);
							if (node4.getNodeName().equals("jps:date8601")) {
								dateStr = node4.getTextContent();
							}
						}
					}
				}
			}
			else if (node2.getNodeName().equals("type")) {
				typeStr = node2.getTextContent();

			}
			else if (node2.getNodeName().equals("name")) {
				nameStr = node2.getTextContent();
			}
		}

		if (admMap.get(codeStr) == null) {
			AdmPt adm = new AdmPt();
			adm.code = codeStr;
			adm.name = nameStr;
			adm.type = typeStr;
			adm.latStr = latStr;
			adm.lonStr = lonStr;
			adm.dateStr = dateStr;
			admMap.put(codeStr, adm);
		}
	}

	class AdmPt {
		public String code;
		public String name;
		public String type;
		public String latStr;
		public String lonStr;
		public String dateStr;
	}

	/**
	 *
	 * 例　[FG-JPS-07-AdmPt25000-20110131-0001.xml]
	 *
	 * @param f
	 * @return
	 */
	static boolean checkFile(File f) {
		String name = f.getName();
		if (!name.startsWith("FG-JPS-")) {
			return false;
		}
		if (!name.toUpperCase().endsWith(".XML")) {
			return false;
		}
		if (name.toUpperCase().indexOf("-ADMPT25000-") < 0) {
			return false;
		}
		return true;
	}


}