/************************************************************
Copyright 2005 Masahiko SAWAI All Rights Reserved. 
************************************************************/

import sdl4gcj.SDLSystem;
import sdl4gcj.SDLConstants;
import sdl4gcj.SDLException;
import sdl4gcj.video.*;
import sdl4gcj.event.*;
import sdl4gcj.input.Keyboard;
import sdl4gcj.mixer.*;

import java.util.StringTokenizer;

class Component implements Rect
{
	static final Color DEFAULT_BACKGROUND = new Color(0x33, 0x33, 0x33);
	static final Color DEFAULT_FOREGROUND = Color.WHITE;
	
	private int x = 0;
	private int y = 0;
	private int w = 100;
	private int h = 100;
	private Color background = DEFAULT_BACKGROUND;
	private Color foreground = DEFAULT_FOREGROUND;

	public Color getBackground() { return background; }
	public void setBackground(Color background) { this.background = background; }

	public Color getForeground() { return foreground; }
	public void setForeground(Color foreground) { this.foreground = foreground; }

	public int getW() { return w; }
	public void setW(int w) { this.w = w; }

	public int getH() { return h; }
	public void setH(int h) { this.h = h; }

	public int getX() { return x; }
	public void setX(int x) { this.x = x; }

	public int getY() { return y; }
	public void setY(int y) { this.y = y; }

	public void setLocation(int x, int y)
	{
		this.setX(x);
		this.setY(y);
	}

	public void setSize(int w, int h)
	{
		this.setW(w);
		this.setH(h);
	}

	public void paint(Surface surface)
	{
		surface.setClipRect(this.getX(), this.getY(), 
			this.getW(), this.getH());
		int bgPixel = surface.mapRGB(this.getBackground());
		surface.fillRect(this.getX(), this.getY(), 
			this.getW(), this.getH(), bgPixel);
	}
}

class Label extends Component
{
	public Label(ImageFont font, String text)
	{
		this.setText(text);
		this.setFont(font);
	}

	public Label(ImageFont font)
	{
		this(font, "Label");
	}

	public CharSequence getText() { return text; }
	public void setText(CharSequence text) { this.text = text; }

	public ImageFont getFont() { return font; }
	public void setFont(ImageFont font) { this.font = font; }

	public int getVerticalAlignment() { return verticalAlignment; }
	public void setVerticalAlignment(int verticalAlignment) { this.verticalAlignment = verticalAlignment; }

	public int getHorizontalAlignment() { return horizontalAlignment; }
	public void setHorizontalAlignment(int horizontalAlignment) { this.horizontalAlignment = horizontalAlignment; }

	public void paint(Surface surface)
	{
		super.paint(surface);

		CharSequence text = this.getText();
		ImageFont font = this.getFont();

		int x = this.getX();
		switch (this.getHorizontalAlignment())
		{
			case CENTER :
				x += (this.getW() - font.getStringWidth(text)) / 2;
				break;
			case RIGHT :
				x += (this.getW() - font.getStringWidth(text));
				break;
		}

		int y = this.getY();
		switch (this.getVerticalAlignment())
		{
			case CENTER :
				y += (this.getH() - font.getHeight()) / 2;
				break;
			case BOTTOM :
				y += (this.getH() - font.getHeight());
				break;
		}

		font.draw(surface, text, x, y);
	}

	private ImageFont font;
	private CharSequence text;
	private int verticalAlignment = CENTER;
	private int horizontalAlignment = LEFT;

	public static final int CENTER = 0;
	public static final int LEFT = 1;
	public static final int RIGHT = 2;
	public static final int TOP = 3;
	public static final int BOTTOM = 4;
}

class PlayingStatusLabel extends Label
{
	private Mixer mixer; 

	PlayingStatusLabel(ImageFont font, Mixer mixer)
	{
		super(font);
		this.mixer = mixer;
	}

	public CharSequence getText()
	{
		String text = "??";

		if (this.mixer.playingMusic())
		{
			if (this.mixer.pausedMusic())
				text = "Status : Paused";
			else
				text = "Status : Playing";
		}
		else
		{
			text = "Status : Stoped";
		}

		return text;
	}
}

class MusicVolumeLabel extends Label
{
	private Mixer mixer; 
	private StringBuffer buffer = new StringBuffer();

	MusicVolumeLabel(ImageFont font, Mixer mixer)
	{
		super(font);
		this.mixer = mixer;
	}

	public CharSequence getText()
	{
		buffer.delete(0, buffer.length());
		buffer.append("Volume : ");
		buffer.append(mixer.getVolumeMusic());

		return buffer;
	}
}

class MusicPlayer implements MixerConstants
{
	private final Screen screen;
	private final Mixer mixer;
	private final Music music;

	private final ImageFont font = new ImageFont("../test/images/font-16x32.bmp", 16, 32);
	private final int bgPixel;
	private final StringBuffer buffer = new StringBuffer();

	private Label titleLabel;
	private Label statusLabel;
	private Label musicVolumeLabel;
	private Label playLabel;
	private Label stopLable;
	private Label quitLable;

	MusicPlayer(Screen screen, Mixer mixer, Music music)
	{
		this.screen = screen;
		this.mixer = mixer;
		this.music = music;
		this.bgPixel = this.screen.mapRGB(Color.BLACK);

		int x = 10;
		int y = 10;
		int w = this.screen.getW() - 20;
		int h = this.font.getHeight()+10;

		this.titleLabel = new Label(font, "Music Player");
		this.titleLabel.setLocation(x, y);
		this.titleLabel.setSize(w, h);
		this.titleLabel.setHorizontalAlignment(Label.CENTER);

		y += h + 10;
		this.statusLabel = new PlayingStatusLabel(font, this.mixer);
		this.statusLabel.setLocation(x, y);
		this.statusLabel.setSize(w, h);

		y += h + 10;
		this.musicVolumeLabel = new MusicVolumeLabel(font, this.mixer);
		this.musicVolumeLabel.setLocation(x, y);
		this.musicVolumeLabel.setSize(w, h);

		y += h + 10;
		this.playLabel = new Label(font, "P or SPACE : PAUSE or PLAY");
		this.playLabel.setLocation(x, y);
		this.playLabel.setSize(w, h);

		y += h;
		this.stopLable = new Label(font, "S : STOP");
		this.stopLable.setLocation(x, y);
		this.stopLable.setSize(w, h);

		y += h;
		this.quitLable = new Label(font, "Q : QUIT");
		this.quitLable.setLocation(x, y);
		this.quitLable.setSize(w, h);
	}

	public void paint()
	{
		this.screen.fillRect(this.bgPixel);
		this.titleLabel.paint(this.screen);
		this.statusLabel.paint(this.screen);
		this.musicVolumeLabel.paint(this.screen);
		this.playLabel.paint(this.screen);
		this.stopLable.paint(this.screen);
		this.quitLable.paint(this.screen);
	}

	public void playOrPause()
	{
		if (mixer.playingMusic())
		{
			if (mixer.pausedMusic())
			{
				mixer.resumeMusic();
			}
			else
			{
				mixer.pauseMusic();
			}
		}
		else
		{
			mixer.playMusic(this.music);
		}
	}

	public void stop()
	{
		if (mixer.playingMusic())
		{
			mixer.haltMusic();
		}
	}

	public void volumeUp()
	{
		int volume = mixer.getVolumeMusic();
		volume++;
		if ((volume <= 128) && (volume >= 0))
			mixer.setVolumeMusic(volume);
	}

	public void volumeDown()
	{
		int volume = mixer.getVolumeMusic();
		volume--;
		if ((volume <= 128) && (volume >= 0))
			mixer.setVolumeMusic(volume);
	}

	public void rewind()
	{
		if (mixer.playingMusic())
		{
			switch (music.getMusicType())
			{
			case MUS_MOD :
			case MUS_OGG :
			case MUS_MP3 :
			case MUS_MID :
				mixer.rewindMusic();
				break;
			default:
				boolean paused = mixer.pausedMusic();
				mixer.haltMusic();
				mixer.playMusic(music);
				if (paused) mixer.pauseMusic();
				break;
			}
		}
	}

	public void mainLoop()
	{
		this.paint();
		this.screen.flip();
		int keyUpWait = 5;
		int keyDownWait = 5;

		Keyboard.getKeyboard().enableKeyRepeat(500, 10);
		EventManager event = new EventManager();

		boolean done = false;
		while (!done)
		{
			while (event.pollEvent() > 0)
			{
				switch (event.type)
				{
					case SDLEvent.SDL_KEYDOWN :
						switch (event.keydown.sym)
						{
							case Keyboard.SDLK_SPACE:
							case Keyboard.SDLK_p:
								this.playOrPause();
								break;
							case Keyboard.SDLK_s:
								this.stop();
								break;
							case Keyboard.SDLK_LEFT:
								this.rewind();
								break;
							case Keyboard.SDLK_UP:
								this.volumeUp();
								break;
							case Keyboard.SDLK_DOWN:
								this.volumeDown();
								break;
							case Keyboard.SDLK_ESCAPE:
							case Keyboard.SDLK_q:
								done = true;
								break;
							default:
								break;
						}
						break;
					case SDLEvent.SDL_QUIT:
						done = true;
						break;
					default:
						break;
				}
			}

			this.paint();
			this.screen.flip();

			SDLSystem.delay(30);
		}
	}

}

public class PlayMusic implements  SDLConstants, MixerConstants
{
	static final String DEFAULT_MUSIC_FILE = "sounds/music.wav";

	public static void main(String[] args)
	{
		if (SDLSystem.init(SDL_INIT_VIDEO) != 0) System.exit(0);

		try
		{
			Screen screen = Screen.setVideoMode(
				640, 480, 8, SDL_SWSURFACE|SDL_ANYFORMAT);

			Mixer mixer = Mixer.openAudio(
				MIX_DEFAULT_FREQUENCY, MIX_DEFAULT_FORMAT,
				MIX_DEFAULT_CHANNELS, 1024);

			String musicFile = DEFAULT_MUSIC_FILE;
			if (args.length > 0) musicFile = args[0];
			Music music = new Music(musicFile);

			MusicPlayer player = new MusicPlayer(screen, mixer, music);
			player.mainLoop();

			music.freeMusic();
			mixer.closeAudio();
		}
		catch (SDLException e)
		{
			e.printStackTrace();
		}
		finally
		{
			SDLSystem.quit();
		}

		return ;
	}
}

