import sdl4gcj.SDLSystem;
import sdl4gcj.SDLConstants;
import sdl4gcj.video.*;
import sdl4gcj.event.*;
import sdl4gcj.cdrom.*;
import sdl4gcj.input.Keyboard;

class CDPlayer implements SDLConstants
{
	private final CD cd;
	private final Screen screen;
	private final ImageFont font = new ImageFont();
	private final int bgPixel;
	private final StringBuffer buffer = new StringBuffer();

	private CDTrack[] tracks = null;

	CDPlayer(CD cd)
	{
		this.cd = cd;
		this.screen = Screen.getVideoSurface();
		this.bgPixel = this.screen.mapRGB(Color.BLACK);
		if(!this.cd.isOpened()) this.cd.open();
	}

	private String getStatusMessage()
	{
		String message = "??";
		switch (this.cd.getStatus())
		{
			case CD_TRAYEMPTY :
				message = "CD_TRAYEMPTY";
				break;
			case CD_STOPPED :
				message = "CD_STOPPED";
				break;
			case CD_PLAYING :
				message = "CD_PLAYING";
				break;
			case CD_PAUSED :
				message = "CD_PAUSED";
				break;
			case CD_ERROR :
				message = "CD_ERROR";
				break;
			default:
				break;
		}
		return message;
	}

	private void drawStatus()
	{
		this.screen.fillRect(0, 0, screen.getW(), font.getHeight(), this.bgPixel);
		this.font.draw(this.screen, this.getStatusMessage(), 10, 0);
		this.screen.updateRect(0, 0, screen.getW(), font.getHeight());
	}

	private void drawTrackNum()
	{
		int x = 10;
		int y = this.font.getHeight();

		this.screen.fillRect(0, y, screen.getW(), font.getHeight(), this.bgPixel);
		buffer.delete(0, buffer.length());
		buffer.append("Current Track : ");
		buffer.append(this.cd.getCurrentTrack());
		buffer.append(" / ");
		buffer.append(this.cd.getNumTracks());

		this.font.draw(this.screen, buffer, x, y);
		this.screen.updateRect(0, y, screen.getW(), font.getHeight());
	}

	private void drawTime()
	{
		int x = 10;
		int y = this.font.getHeight()*2;
		int frame;

		this.screen.fillRect(0, y, screen.getW(), font.getHeight(), this.bgPixel);
		frame = this.cd.getCurrentFrame();
		buffer.delete(0, buffer.length());
		buffer.append("Time : ");
		buffer.append(CD.framesToMinute(frame));
		buffer.append(":");
		buffer.append(CD.framesToSecond(frame));

		buffer.append(" / ");

		if (this.tracks != null)
			frame = this.tracks[this.cd.getCurrentTrack()].length;
		else
			frame = 0;
		buffer.append(CD.framesToMinute(frame));
		buffer.append(":");
		buffer.append(CD.framesToSecond(frame));

		this.font.draw(this.screen, buffer, x, y);
		this.screen.updateRect(0, y, screen.getW(), font.getHeight());
	}

	private void drawUsage()
	{
		int x = 10;
		int y = font.getHeight()*4;
		this.screen.fillRect(0, y, screen.getW(), screen.getH()-y, this.screen.mapRGB(Color.BLACK));
		this.font.draw(this.screen, "P or SPACE : PAUSE or PLAY", x, y);
		y += font.getHeight();
		this.font.draw(this.screen, "S : STOP", x, y);
		y += font.getHeight();
		this.font.draw(this.screen, "E : EJECT", x, y);
		y += font.getHeight();
		this.font.draw(this.screen, "Q : QUIT", x, y);
	}

	public void mainLoop()
	{
		this.drawStatus();
		this.drawTrackNum();
		this.drawTime();
		this.drawUsage();
		this.screen.flip();

		EventManager event = new EventManager();
		int status = cd.getStatus();
		if (status != CD_TRAYEMPTY)
			this.tracks = this.cd.getTracks();

		boolean done = false;
		while (!done)
		{
			while (event.pollEvent() > 0)
			{
				switch (event.type)
				{
					case SDLEvent.SDL_KEYDOWN :
						switch (event.keydown.sym)
						{
							case Keyboard.SDLK_SPACE:
							case Keyboard.SDLK_p:
								if (cd.getStatus() == CD_STOPPED)
									cd.playTracks();
								else if (cd.getStatus() == CD_PLAYING)
									cd.pause();
								else if (cd.getStatus() == CD_PAUSED)
									cd.resume();
								break;
							case Keyboard.SDLK_s:
								if (cd.getStatus() == CD_PLAYING ||
									cd.getStatus() == CD_PAUSED) cd.stop();
								break;
							case Keyboard.SDLK_e:
								cd.eject();
								break;
							case Keyboard.SDLK_ESCAPE:
							case Keyboard.SDLK_q:
								done = true;
								break;
							default:
								break;
						}
						break;
					case SDLEvent.SDL_QUIT:
						done = true;
						break;
					default:
						break;
				}
			}

			if (status == CD_PLAYING)
			{
				this.drawTrackNum();
				this.drawTime();
			}

			if (cd.getStatus() != status)
			{
				this.drawStatus();
				status = cd.getStatus();
				switch (status)
				{
					case CD_STOPPED :
						this.tracks = this.cd.getTracks();
						break;
					case CD_TRAYEMPTY :
						this.tracks = null;
						break;
					default:
						break;
				}
			}

			SDLSystem.delay(30);
		}
	}

}

public class CDPlayerTest implements SDLConstants
{
	public static void main(String[] args)
	{
		if (SDLSystem.init(SDL_INIT_VIDEO|SDL_INIT_CDROM) != 0)
		{
			System.out.println("Init() fail");
			System.exit(0);
		}

		int index = -1;
		if (args.length > 0)
		{
			try
			{
				index = Integer.parseInt(args[0]);
			}
			catch (NumberFormatException e) { }
		}

		if (index < 0)
		{
			CD[] drives = CD.getAllCD();
			for (int i = 0;i < drives.length;i++)
			{
				drives[i].open();
				System.out.println("drives[" + i + "]");
				System.out.println("\tIndex             : " + drives[i].getIndex());
				System.out.println("\tName              : " + drives[i].getName());
				if (drives[i].inDrive())
				{
					System.out.println("\tNumber of Tracks   : " + drives[i].getNumTracks());
				}
				System.out.println();
				drives[i].close();
			}
			return ;
		}

		CD cd = CD.getCD(index);
		cd.open();
		try
		{
			Screen screen = Screen.setVideoMode(640, 480, 32, Surface.SDL_SWSURFACE|Surface.SDL_DOUBLEBUF);
			CDPlayer player = new CDPlayer(cd);
			player.mainLoop();
		}
		catch (Exception e)
		{
			e.printStackTrace();
		}
		finally
		{
			cd.stop();
			cd.close();
			SDLSystem.quit();
		}
	}
}
