package org.seasar.timer;

import org.seasar.util.ELinkedList;
import org.seasar.util.SeasarRuntimeException;

public class TimeoutManager implements Runnable {

    private static TimeoutManager _instance = new TimeoutManager();

    private Thread _thread;
    private ELinkedList _timeoutTaskList = new ELinkedList();

    private TimeoutManager() {
        start();
    }

    public static TimeoutManager getInstance() {
        return _instance;
    }

    public void start() {
        if (_thread != null) {
            throw new SeasarRuntimeException("ECHR0018", new Object[]{"_thread"});
        }
        _thread = new Thread(this);
        _thread.setDaemon(true);
        _thread.start();
    }

    public void stop() {
        _thread.interrupt();
        _thread = null;
    }

    public synchronized void clear() {
        _timeoutTaskList.clear();
    }

    public void run() {
        while (true) {
            try {
                synchronized (_timeoutTaskList) {
                    while (_timeoutTaskList.isEmpty()) {
                        _timeoutTaskList.wait();
                    }
                    for (ELinkedList.Entry e = _timeoutTaskList.getFirstEntry(); e != null; e = e.getNext()) {
                        TimeoutTask task = (TimeoutTask) e.getElement();
                        if (task.isCanceled()) {
                            e.remove();
                            continue;
                        }
                        if (task.isStopped()) {
                            continue;
                        }
                        if (task.isExpired()) {
                            task.expired();
                            if (task.isPermanent()) {
                                task.restart();
                            } else {
                                e.remove();
                            }
                        }
                    }
                }
                Thread.sleep(1000);
            } catch (InterruptedException ignore) {
            }
        }
    }


    public TimeoutTask addTimeoutTarget(final TimeoutTarget timeoutTarget,
            int timeout, boolean permanent) {

        TimeoutTask task = new TimeoutTask(timeoutTarget, timeout, permanent);
        synchronized (_timeoutTaskList) {
            _timeoutTaskList.addLast(task);
            _timeoutTaskList.notifyAll();
        }
        return task;
    }

    public int getTimeoutTaskCount() {
        return _timeoutTaskList.size();
    }
}