package org.seasar.expr;

import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;

import org.seasar.util.Assertion;

public final class ExprCoreTokenizer {
	
    public static final int TT_EOF = -1;
    public static final int TT_INTEGER = -11;
	public static final int TT_LONG = -12;
	public static final int TT_DOUBLE = -13;
    public static final int TT_WORD = -3;
    
    private static final int TT_NOTHING = -4;
    private static final int NEED_CHAR = Integer.MAX_VALUE;
    private static final int QUOTE = '\'';
	
	public int ttype = TT_NOTHING;
	public String sval;
	public Integer ival;
	public Long lval;
	public Double dval;
	
    private Reader reader = null;

    private char[] buf = new char[20];
    private int peekc = NEED_CHAR;
    private byte peekct = 0;
    
    private byte[] ctype = new byte[256];
    private static final byte CT_WHITESPACE = 1;
    private static final byte CT_DIGIT = 2;
    private static final byte CT_ALPHA = 4;

    public ExprCoreTokenizer(String s) {
    	Assertion.assertNotNull("s", s);
		reader = new StringReader(s);
        wordChars('a', 'z');
        wordChars('A', 'Z');
        wordChars('0', '9');
        wordChars('@', '@');
        wordChars('|', '|');
        wordChars('_', '_');
        wordChars('?', '?');
        wordChars('>', '>');
        wordChars('=', '=');
        wordChars('!', '!');
        wordChars('<', '<');
        wordChars(':', ':');
        wordChars('"', '"');
        wordChars('~', '~');
        wordChars('*', '*');
        ordinaryChar('(');
        ordinaryChar(')');
        ordinaryChar('+');
        ordinaryChar('-');
        ordinaryChar('/');
        ordinaryChar('%');
        ordinaryChar(',');
        ordinaryChar('[');
        ordinaryChar(']');
        ordinaryChar('{');
        ordinaryChar('}');
        whitespaceChars(0, ' ');
        parseNumbers();
    }

    public synchronized int nextToken() throws IOException {
		initVal();
		if (processEOF()) {
			return ttype;
		}
		if (processWhitespace()) {
			return ttype;
		}
		if (processDigit()) {
			return ttype;
		}
		if (processWord()) {
			return ttype;
		}
		if (processQuote()) {
			return ttype;
		}
		if (processOrdinary()) {
			return ttype;
		}
		return ttype = peekc;
    }

	private int read() throws IOException {
	    return reader.read();
    }

	private void resetSyntax() {
		for (int i = ctype.length; --i >= 0; ) {
	    	ctype[i] = 0;
		}
    }

    private void wordChars(int low, int hi) {
		if (low < 0) {
	    	low = 0;
		}
		if (hi >= ctype.length) {
	    	hi = ctype.length - 1;
		}
		while (low <= hi) {
	    	ctype[low++] |= CT_ALPHA;
		}
    }

    private void whitespaceChars(int low, int hi) {
		if (low < 0) {
	    	low = 0;
		}
		if (hi >= ctype.length) {
	    	hi = ctype.length - 1;
		}
		while (low <= hi) {
	    	ctype[low++] = CT_WHITESPACE;
		}
    }

    private void ordinaryChars(int low, int hi) {
		if (low < 0) {
	    	low = 0;
		}
		if (hi >= ctype.length) {
	    	hi = ctype.length - 1;
		}
		while (low <= hi) {
	    	ctype[low++] = 0;
		}
    }

    private void ordinaryChar(int ch) {
        if (ch >= 0 && ch < ctype.length) {
  	    	ctype[ch] = 0;
        }
    }

    private void parseNumbers() {
		for (int i = '0'; i <= '9'; i++) {
	    	ctype[i] |= CT_DIGIT;
		}
		ctype['.'] |= CT_DIGIT;
		ctype['-'] |= CT_DIGIT;
    }
    
	private void initVal() {
		sval = null;
		ival = null;
		lval = null;
		dval = null;
	}
	
	private boolean processEOF() throws IOException {
		if (peekc < 0) {
	    	ttype = TT_EOF;
	    	return true;
		}
		if (peekc == NEED_CHAR) {
	    	peekc = read();
	    	if (peekc < 0) {
				ttype = TT_EOF;
				return true;
	    	}
		}
		return false;
	}
	        
    private boolean processWhitespace() throws IOException {
		peekct = peekc < 256 ? ctype[peekc] : CT_ALPHA;
		while ((peekct & CT_WHITESPACE) != 0) {
	    	if (peekc == '\r') {
				peekc = read();
				if (peekc == '\n') {
		    		peekc = read();
				}
	    	} else {
				peekc = read();
	    	}
	    	if (peekc < 0) {
				ttype = TT_EOF;
				return true;
	    	}
	    	peekct = peekc < 256 ? ctype[peekc] : CT_ALPHA;
		}
		return false;
    }

	private boolean processDigit() throws IOException {
		if ((peekct & CT_DIGIT) != 0) {
	    	boolean neg = false;
	    	if (peekc == '-') {
				peekc = read();
				if (peekc != '.' && (peekc < '0' || peekc > '9')) {
		    		ttype = '-';
		    		return true;
				}
				neg = true;
	    	}
	    	long v = 0;
	    	int decexp = 0;
	    	int seendot = 0;
	    	while (true) {
				if (peekc == '.' && seendot == 0) {
		    		seendot = 1;
				} else if ('0' <= peekc && peekc <= '9') {
		    		v = v * 10 + (peekc - '0');
		    		decexp += seendot;
				} else {
		    		break;
				}
				peekc = read();
	    	}
	    	long v2 = neg ? -v : v;
	    	if (decexp != 0) {
	    		dval = new Double((double) v2 / Math.pow(10, decexp));
	    		ttype = TT_DOUBLE;
	    		return true;
	    	}
	    	if (v2 > Integer.MAX_VALUE || v2 < Integer.MIN_VALUE) {
	    		lval = new Long(v2);
	    		ttype = TT_LONG;
	    		return true;
	    	}
	    	int v3 = (int) v;
	    	v3 = neg ? -v3 : v3;
	    	ival = new Integer(v3);
	    	ttype = TT_INTEGER;
	    	return true;
		}
		return false;
	}
	
	private boolean processWord() throws IOException {
		if ((peekct & CT_ALPHA) != 0) {
	    	int i = 0;
	    	do {
				if (i >= buf.length) {
		    		char nb[] = new char[buf.length * 2];
		    		System.arraycopy(buf, 0, nb, 0, buf.length);
		    		buf = nb;
				}
				buf[i++] = (char) peekc;
				peekc = read();
				peekct = peekc < 0 ? CT_WHITESPACE : (peekc < 256 ? ctype[peekc] : CT_ALPHA);
	    	} while ((peekct & (CT_ALPHA | CT_DIGIT)) != 0);
	    	sval = String.copyValueOf(buf, 0, i);
	    	ttype = TT_WORD;
	    	return true;
		}
		return false;
	}
	
	private boolean processQuote() throws IOException {
		if (peekc == QUOTE) {
	    	ttype = QUOTE;
	    	int i = 0;
	    	int d = read();
	    	int c = d;
	    	while (d >= 0) {
	    		if (d == QUOTE) {
					int d2 = read();
					if (d2 == QUOTE) {
						c = QUOTE;
					} else {
						d = d2;
						break;
					}
	    		} else {
		    		c = d;
				}
				if (i >= buf.length) {
		    		char nb[] = new char[buf.length * 2];
		    		System.arraycopy(buf, 0, nb, 0, buf.length);
		    		buf = nb;
				}
				buf[i++] = (char) c;
				d = read();
	    	}
	    	peekc = d;
	    	sval = String.copyValueOf(buf, 0, i);
	    	return true;
		}
		return false;
    }
    
    private boolean processOrdinary() throws IOException {
		if (peekct == 0) {
	    	ttype = peekc;
	    	peekc = read();
	    	peekct = peekc < 0 ? CT_WHITESPACE : (peekc < 256 ? ctype[peekc] : CT_ALPHA);
	    	return true;
		}
		return false;
	}
}
